package fr.ifremer.tutti.ui.swing.launcher;

/*
 * #%L
 * Tutti :: Launcher
 * %%
 * Copyright (C) 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import javax.swing.JOptionPane;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Launcher application that do post-update logic, launch application and listen to exit code.
 *
 * @author Tony Chemit (chemit@codelutin.com)
 * @since 1.0
 */
public class Launcher {

    public static final String APPLICATION_LANCHER_TITLE = "Tutti UI Launcher";

    public static final String APPLICATION_DIR = "tutti";

    public static final String DATA_DIR = "data";

    public static final String UPDATER_JAR = "updater.jar";

    public static final String UPDATER_LOGFILE = "updater.log";

    public static final String FULL_LAUNCH_MODE = "full";

    public static final String DEBUG_ARG = "--debug";

    private final Path basedir;

    private final List<String> arguments;

    private final Path javaCommand;

    private final boolean debugMode;

    public static void main(String... args) {

        boolean windowsOs = System.getProperty("os.name").startsWith("Windows");

        String javaCommandFilename = windowsOs ? "java.exe" : "java";

        Launcher launcher = new Launcher(javaCommandFilename, args);
        launcher.execute();

    }

    public Launcher(String javaCommandFilename, String... args) {

        this.basedir = Paths.get(System.getProperty("user.dir"));

        this.javaCommand = basedir.resolve("jre").resolve("bin").resolve(javaCommandFilename);

        List<String> fullArguments = Arrays.asList(args);

        this.debugMode = fullArguments.remove(DEBUG_ARG);

        this.arguments = Collections.unmodifiableList(fullArguments);

    }

    public void execute() {

        try {

            ExitCode exitCode;

            do {

                // call update process
                exitCode = executeUpdate();

                if (exitCode == ExitCode.RUNTIME_UPDATE) {

                    // if updater has detected runtime update, stop now
                    System.exit(exitCode.getExitCode());

                }

                if (exitCode != ExitCode.NORMAL) {

                    // update fails
                    throw new RuntimeException("Tutti Updater has encountered a problem, program stopped.\nPlease check " + UPDATER_LOGFILE);

                }

                // launch application
                exitCode = executeApplication();

                if (exitCode == ExitCode.DELETE_DB) {
                    deleteDb();
                }

            } while (exitCode.isRelaunch());

            if (exitCode != ExitCode.NORMAL) {

                // show exit code 
                throw new RuntimeException("exit code = " + exitCode);

            }

        } catch (Exception ex) {
            JOptionPane.showMessageDialog(null, ex.toString(), APPLICATION_LANCHER_TITLE, JOptionPane.ERROR_MESSAGE);
        }

    }

    protected ExitCode executeUpdate() throws Exception {

        Path updatePath = basedir.resolve(APPLICATION_DIR).resolve(UPDATER_JAR);
        Path updateExecPath = basedir.resolve(UPDATER_JAR);
        Files.copy(updatePath, updateExecPath, StandardCopyOption.REPLACE_EXISTING);

        Path logfile = basedir.resolve(UPDATER_LOGFILE);

        try {

            JavaCommand program = JavaCommand
                    .create(javaCommand)
                    .setJarPath(updateExecPath)
                    .setLogFile(logfile);

            ExitCode exitCode = program.execute();
            return exitCode;

        } finally {

            Files.delete(updateExecPath);

        }

    }

    protected ExitCode executeApplication() throws Exception {

        LauncherProperties launcherProperties = new LauncherProperties(basedir);
        launcherProperties.load();

        JavaCommand program = JavaCommand
                .create(javaCommand)
                .setDebug(debugMode)
                .setInheritIO(true)
                .addArguments(launcherProperties.getJavaOptions())
                .addArgument("-Dtutti.log.file=" + launcherProperties.getLogFile().toString())
                .setJarPath(launcherProperties.getJarFile())
                .addArguments("--launch-mode", FULL_LAUNCH_MODE)
                .addArguments("--basedir", basedir.toString())
//                .addArgument("--config.path", basedir.toString())
                .addArgument(arguments);

        ExitCode exitCode = program.execute();
        return exitCode;

    }

    protected void deleteDb() throws Exception {

        Path dataPath = basedir.resolve(DATA_DIR);
        Files.walkFileTree(dataPath.resolve("db"), new RecursiveDeleteFileVisitor());
        Files.walkFileTree(dataPath.resolve("dbcache"), new RecursiveDeleteFileVisitor());
        Files.walkFileTree(dataPath.resolve("meas_files"), new RecursiveDeleteFileVisitor());

    }

    private class RecursiveDeleteFileVisitor extends SimpleFileVisitor<Path> {

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Files.deleteIfExists(file);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            Files.deleteIfExists(dir);
            return FileVisitResult.CONTINUE;
        }

    }

}
