package fr.ifremer.tutti.ui.swing.launcher;

/*
 * #%L
 * Tutti :: Launcher
 * %%
 * Copyright (C) 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Created on 1/3/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 1.0
 */
public class JavaCommand {

    private static final String DEBUG_JAVA_OPTS = "-agentlib:jdwp=transport=dt_socket,server=y,suspend=y,address=%s";

    static JavaCommand create(Path basedir, int debugPort) {
        return new JavaCommand(basedir, debugPort);
    }

    static JavaCommand create(Path basedir) {
        return new JavaCommand(basedir, 8000);
    }

    protected JavaCommand(Path javaCommand, int debugPort) {
        this.javaCommand = javaCommand;
        this.debugPort = debugPort;
        this.preJarArguments = new ArrayList<>();
        this.postJarArguments = new ArrayList<>();
    }

    private final int debugPort;

    private final Path javaCommand;

    private final List<String> preJarArguments;

    private final List<String> postJarArguments;

    private Path jarPath;

    private boolean inheritIO;

    private boolean debug;

    private Path logFile;

    public JavaCommand setInheritIO(boolean inheritIO) {
        this.inheritIO = inheritIO;
        return this;
    }

    public JavaCommand setDebug(boolean debug) {
        this.debug = debug;
        return this;
    }

    public JavaCommand setJarPath(Path jarPath) {
        this.jarPath = jarPath;
        return this;
    }

    public JavaCommand setLogFile(Path logFile) {
        this.logFile = logFile;
        return this;
    }

    public JavaCommand addArgument(Object argument) {

        List<String> whereToAdd;
        if (jarPath == null) {
            whereToAdd = preJarArguments;

        } else {
            whereToAdd = postJarArguments;
        }

        whereToAdd.add(argument.toString());

        return this;
    }

    public JavaCommand addArguments(Object argument, Object... arguments) {

        List<String> whereToAdd;
        if (jarPath == null) {
            whereToAdd = preJarArguments;

        } else {
            whereToAdd = postJarArguments;
        }

        whereToAdd.add(argument.toString());

        for (Object argument1 : arguments) {
            whereToAdd.add(argument1.toString());
        }

        return this;
    }

    public JavaCommand addArguments(Collection<?> arguments) {

        List<String> whereToAdd;
        if (jarPath == null) {
            whereToAdd = preJarArguments;

        } else {
            whereToAdd = postJarArguments;
        }

        for (Object argument : arguments) {
            whereToAdd.add(argument.toString());
        }

        return this;
    }

    public ExitCode execute() throws IOException, InterruptedException {

        if (jarPath == null) {

            throw new IllegalStateException("No jarPath given, can't execute.");

        }

        List<String> command = new ArrayList<>();

        command.add(javaCommand.toString());
        command.addAll(preJarArguments);
        if (debug) {
            command.add(String.format(DEBUG_JAVA_OPTS, debugPort));
        }
        command.add("-jar");
        command.add(jarPath.toString());
        command.addAll(postJarArguments);

        ProcessBuilder pb = new ProcessBuilder(command);

        if (inheritIO) {

            pb.inheritIO();

        }

        if (logFile != null) {

            File updaterLogFile = logFile.toFile();
            pb.redirectOutput(ProcessBuilder.Redirect.to(updaterLogFile));
            pb.redirectError(ProcessBuilder.Redirect.appendTo(updaterLogFile));

        }

        System.out.println("Starts java command: " + command);

        int exitCode = pb.start().waitFor();
        return ExitCode.valueOf(exitCode);

    }
}
