/*
 * #%L
 * $Id: PlanHandler.java 4263 2015-06-11 09:01:47Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/ui/simulator/PlanHandler.java $
 * %%
 * Copyright (C) 2013 - 2014 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package fr.ifremer.isisfish.ui.simulator;

import java.util.List;

import javax.swing.table.DefaultTableModel;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.SimulationPlanStorage;
import fr.ifremer.isisfish.datastore.StorageChangeEvent;
import fr.ifremer.isisfish.datastore.StorageChangeListener;
import fr.ifremer.isisfish.simulator.SimulationPlan;
import fr.ifremer.isisfish.ui.models.common.GenericComboModel;
import fr.ifremer.isisfish.ui.models.common.GenericListModel;
import fr.ifremer.isisfish.ui.models.common.ScriptParametersTableCellEditor;
import fr.ifremer.isisfish.ui.models.common.ScriptParametersTableCellRenderer;
import fr.ifremer.isisfish.ui.models.common.ScriptParametersTableModel;
import fr.ifremer.isisfish.ui.widget.editor.ScriptParameterDialog;

/**
 * Plan UI handler.
 * 
 * @author Eric Chatellier
 */
public class PlanHandler extends SimulatorTabHandler {

    protected PlanUI tabUI;

    protected StorageChangeListener simulationPlanStorageListener;

    protected PlanHandler(PlanUI tabUI) {
        super(tabUI);
        this.tabUI = tabUI;
    }

    protected void afterInit() {
        simulationPlanStorageListener = new StorageChangeListener() {
            @Override
            public void versionDataChanged(StorageChangeEvent e) {
                refresh();
            }
        };
        SimulationPlanStorage.addStorageChangeListener(simulationPlanStorageListener);
        
        // model init
        tabUI.fieldSimulParamsSimulationPlansSelect.setModel(getSimulationPlansComboModel());
        tabUI.listSimulParamsSimulationPlansList.setModel(getSimulationSimulationPlansListModel());
    }
    
    public void refresh() {
        // simulation plans names list
        tabUI.fieldSimulParamsSimulationPlansSelect.setModel(getSimulationPlansComboModel());
        // simulation simulation plans instance
        tabUI.listSimulParamsSimulationPlansList.setModel(getSimulationSimulationPlansListModel());
    }

    /**
     * Return simulation plan names list.
     * 
     * @return simulation plan names
     */
    public List<String> getSimulationPlanNames() {
        return SimulationPlanStorage.getSimulationPlanNames();
    }

    public List<SimulationPlan> getParamSimulationPlans() {
        return getParameters().getSimulationPlans();
    }

    public void addSimulationPlan(PlanUI planUI, String name) {
        try {
            SimulationPlan sp = SimulationPlanStorage.getSimulationPlan(name).getNewInstance();
            
            // add it after autoconfiguration (if enabled)
            sp = (SimulationPlan)ScriptParameterDialog.displayConfigurationFrame(planUI, sp);
            if (sp != null) {
                getParameters().addSimulationPlan(sp);
            }
        } catch (IsisFishException ex) {
            throw new IsisFishRuntimeException("Can't add simulation plan", ex);
        }
    }

    public void removeSimulationPlan(SimulationPlan sp) {
        getParameters().removeSimulationPlan(sp);
    }

    public void clearSimulationPlans() {
        getParameters().clearPlans();
    }

    /**
     * Refresh simulation plan names list.
     */
    protected GenericComboModel<String> getSimulationPlansComboModel() {
        List<String> simulationPlanNames = getSimulationPlanNames();
        GenericComboModel<String> model = new GenericComboModel<>(simulationPlanNames);
        return model;
    }
    
    /**
     * Refresh simulation simulation plan list.
     */
    protected GenericListModel<SimulationPlan> getSimulationSimulationPlansListModel() {
        List<SimulationPlan> simulationPlans = getParamSimulationPlans();
        GenericListModel<SimulationPlan> model = new GenericListModel<>(simulationPlans);
        return model;
    }

    protected void addSimulationPlan() {
        String selectedSimulationPlanName = (String)tabUI.fieldSimulParamsSimulationPlansSelect.getSelectedItem();
        addSimulationPlan(tabUI, selectedSimulationPlanName);
        tabUI.listSimulParamsSimulationPlansList.setModel(getSimulationSimulationPlansListModel());
        setSimulationPlanButtons();
    }
    
    protected void removeSimulationPlan() {
        SimulationPlan selectedSimulationPlan = (SimulationPlan)tabUI.listSimulParamsSimulationPlansList.getSelectedValue();
        removeSimulationPlan(selectedSimulationPlan);
        tabUI.listSimulParamsSimulationPlansList.setModel(getSimulationSimulationPlansListModel());
        setSimulationPlanButtons();
    }
    
    protected void clearSimulationPlan() {
        clearSimulationPlans();
        tabUI.listSimulParamsSimulationPlansList.setModel(getSimulationSimulationPlansListModel());
        setSimulationPlanButtons();
    }
    
    protected void setSimulationPlanButtons() {
        if (getParamSimulationPlans().isEmpty()) {
            tabUI.buttonSimulParamsSimulationPlansClear.setEnabled(false);
            tabUI.buttonSimulParamsSimulationPlansRemove.setEnabled(false);
        } else {
            if (tabUI.listSimulParamsSimulationPlansList.getSelectedIndex() != -1) {
                tabUI.buttonSimulParamsSimulationPlansRemove.setEnabled(true);
            } else {
                tabUI.buttonSimulParamsSimulationPlansRemove.setEnabled(false);
            }
            tabUI.buttonSimulParamsSimulationPlansClear.setEnabled(true);
        }
        setSimulParamsSimulationPlans();
    }
    
    protected void setSimulParamsSimulationPlans() {
        SimulationPlan selectedSimulationPlan = (SimulationPlan)tabUI.listSimulParamsSimulationPlansList.getSelectedValue();
        if (selectedSimulationPlan != null) {
            ScriptParametersTableModel model = new ScriptParametersTableModel(selectedSimulationPlan);
            tabUI.simulParamsSimulationPlans.setModel(model);
            ScriptParametersTableCellEditor cellEditor = new ScriptParametersTableCellEditor(selectedSimulationPlan);
            cellEditor.setRegionStorage(tabUI.getContextValue(RegionStorage.class));
            tabUI.simulParamsSimulationPlans.getColumnModel().getColumn(0).setCellRenderer(new ScriptParametersTableCellRenderer(selectedSimulationPlan));
            tabUI.simulParamsSimulationPlans.getColumnModel().getColumn(1).setCellEditor(cellEditor);
        } else {
            // clear table
            tabUI.simulParamsSimulationPlans.setModel(new DefaultTableModel());
        }
    }
}
