package fr.ifremer.isisfish.ui.script;

/*
 * #%L
 * ISIS-Fish
 * %%
 * Copyright (C) 1999 - 2015 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.isisfish.ui.script.model.ScriptTree;
import fr.ifremer.isisfish.ui.script.model.ScriptTreeModel;
import fr.ifremer.isisfish.ui.widget.text.SyntaxEditor;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.JButton;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.binding.SimpleJAXXObjectBinding;
import jaxx.runtime.context.DefaultJAXXContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n.t;

public class ScriptUI extends JPanel implements JAXXObject {

    /*-----------------------------------------------------------------------*/
    /*---------------- Constants for all javaBean properties ----------------*/
    /*-----------------------------------------------------------------------*/

    public static final String PROPERTY_FILE_SELECTED = "fileSelected";
    public static final String PROPERTY_JAVA_FILE_SELECTED = "javaFileSelected";
    public static final String PROPERTY_SINGLE_FILE_SELECTED = "singleFileSelected";
    public static final String PROPERTY_TEXT_SELECTED = "textSelected";

    /*-----------------------------------------------------------------------*/
    /*------------------ Constants for all public bindings ------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String BINDING_BUTTON_SCRIPT_CHECK_ENABLED = "buttonScriptCheck.enabled";
    public static final String BINDING_BUTTON_SCRIPT_COMMIT_ENABLED = "buttonScriptCommit.enabled";
    public static final String BINDING_BUTTON_SCRIPT_EVAL_ENABLED = "buttonScriptEval.enabled";
    public static final String BINDING_BUTTON_SCRIPT_PASTE_ENABLED = "buttonScriptPaste.enabled";
    public static final String BINDING_BUTTON_SCRIPT_SAVE_ENABLED = "buttonScriptSave.enabled";
    public static final String BINDING_MI_CHECK_SYNTAX_ENABLED = "miCheckSyntax.enabled";
    public static final String BINDING_MI_COMMIT_VCS_ENABLED = "miCommitVCS.enabled";
    public static final String BINDING_MI_DELETE_ENABLED = "miDelete.enabled";
    public static final String BINDING_MI_DIFF_VCS_ENABLED = "miDiffVCS.enabled";
    public static final String BINDING_MI_EVALUATE_ENABLED = "miEvaluate.enabled";
    public static final String BINDING_MI_SAVE_ENABLED = "miSave.enabled";

    /*-----------------------------------------------------------------------*/
    /*------------------------- Other static fields -------------------------*/
    /*-----------------------------------------------------------------------*/

    private static final String $jaxxObjectDescriptor = "H4sIAAAAAAAAAK2az2/cRBTHJ6H5uUmbpDQEtYW2RBUC5PQHQogWmjRNaKK0hDitSiMEE3s2O+3Y49qzyUYIKCAuCARInOAABXEAISEkbpwQB84cuCD+BYQ4cEW8sXf9Y+udHTT0sBv5vffx9715M36722/+QH1RiI7ewI2GFdZ9QT1iLc9du/bc5g3iiAskckIaCB6i5F9PL+rdQBU3vR4J9NDGigyfaYbPzHMv4D7xc9FnVtBwJHYZiWqECIEOFyOcKJqxU/OZRlAPW9RUVBn1s7/+7P3AvX2nF6FGAOrOQSpHukVlmexZQb3UFWgC7rSNZxj2t0BGSP0t0Dsqr80zHEWXsUduodfQwArqD3AIMIGO6accM+L4RiDQ4PTyKvYJOyHQ8Wpo0WpIPALvEY2qNKpZdWolBMuO364sBUEc3S/QQA37LiOhQI/qhF5MvLP4kSplxCYMFpZA2uMyQ0tmbZ3nnBHsZ64TEVSBkcVcgLSM52CCNEQH25gkK2JHE52XiF8/jyGde6V/w4p24KbWcvOy9B1qD1lvCMltii+ESMepdv/LZKdlOyxfDqYOFY+CMamUQAfuwi0J4knXB9KIvUkE9eoMwwLLq8eKHglzoRHwUJSY9+XMtiBBictQ7LJWZ6Rj/BqJ6kws+VVe4rI/0bgUa6TcX4XlLXEbS9yIH1FBt6nYLfGZbPfpmFdlOq3YCbX5ZIl5IqnKrXos+BJ3CSvx6veojbezqmQLOTK9bBPYl3JNYFdNFlYytRRjBj16AVpTlPGyTltwqUg7J2utPo/O17M6zOY1zvNgt8Qy4NFVHOVuN1sqPy7P00U7SLUJDp1aSSx0yxoJGHbKwKDmWS542S3TDK/O2yUJVmQenkdb5jh8NX/XC7RaLTfm0zmVprOaZyeVLw/Pij8PjVCibRS01Yhz0971BW6khOv51o4d5hjr6DPs0YVtzKDjSIk1n8LpNIXMPg6FJT4BO1mGXnO5UwIBnXaN73R2aD0ITsIOKLRsfLV4+A1t1oXgfvNQvCFfatmpkBjt1nkHh0CBdz42S+ebWZHyMa19xYs+E3mfbO+rvJKuKfEqaGxuHq6Q1NpGbT7jeZ90Q6mc4lbodjfZCyU+sg8CRoVckLsPlpaluBijEOOEnLFm0H3FoNQk3W9n5Wl2fUhIfP4JdFr9gPekV/Mxn0Zlz9fhjCeQ9d9QEvFWtrzYkafyCt8qJPxmUX8/gZMS+gPNdLjXDnW3iLDkyGAlu3IhjpDx75YXLz4L325zSNXMhQS3P7XXgS6vS+cPAxgFDxZGQRgyrWzIzKbAng3UF8IDFybZqY2751L5LE4m0qm2iVQCY+s/k/t//eH37xdbY6gD9z5Q6pqbomE8DEIewDRH5a33JTNoXVA2cwkHZzbQUBQPT/GIfahEmN00gzi4XzLMyXDrIo5qgOgb+O3HnyZf/uUe1LuIhhnH7iKW/ktoSNRCqAJnbiM4NxsrGtkZhNcxqQ2Wk+FdDvsD7T3rYoGPbFLfhQo/04AiHCopQqpkc+jnv/fb3822CtEDwu7v6J4Vo+866qc+oz6JJ3K5gAMdJvBKEJG6y7OhumzM7pHvw0FzQ3wSv35aluke2ZHQ5WeTux/BAsZ/OBoI5CqDvoiTkH99GTMP6TCpkxy3n5czHuzKkJe/ki9fmxC+VWg4aqxBj6DSMG2sQY+g0nDcWIMeQaXhYWMNegSVhkeMNegRVBoeM9agR1BpsIw16BFUGk4Ya9AjqDScMtagR1BpeNxYgx5BpeEJYw16BJWGJ401dCfAJ1Ti402WfFPTQchTxqnoEVSpaBF6phQazhlnoUdQZWFOmDMmPG9cSdu4knoEVRbdCQJVsOPAyJl+DdQBdcW4IFoEZUH0CKqCmGt4wViDOWHDeC1eNK6DHkGVhbmGl4w1aBGUlcTGWegRVFmYE4gxYUuLcEdRB2pMYMZr4RvXoTsBjlzBOVungfxKxgjVrTlvGRdEj6AqiBah52NjwpSCEBnXQY+gqoMWQVkHPYKqH+rGddAjqOqgRVDWwVzDjrEGLYJSw66xBj2Cqh9eMc5Ci6DMQo+gyuJV467WI6jqoEVQ1kGPoKrD68Z10COo6qBFUNahO0GgfS7dpi4JV7gT/yyuwL2hgZvgPlnndae20Aiw78pP+obECg8p8cX/IW4EPp2xNeJDuiRUrN07Gqw+r/X/BwwglZBzcZVGVF2m7ii43DumILynRThgRJC/mkU3l6qXefxDr6uAva8BG5Q/9XXpn490fiWpcl8o1kky/gWeLxw8nSYAAA==";
    private static final Log log = LogFactory.getLog(ScriptUI.class);
    private static final long serialVersionUID = 1L;

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Internal states ---------------------------*/
    /*-----------------------------------------------------------------------*/

    protected List<Object> $activeBindings = new ArrayList<Object>();
    protected Map<String, Object> $bindingSources = new HashMap<String, Object>();
    protected final Map<String, JAXXBinding> $bindings = new TreeMap<String, JAXXBinding>();
    protected Map<String, Object> $objectMap = new HashMap<String, Object>();
    protected Map<?, ?> $previousValues = new HashMap<Object, Object>();
    private boolean allComponentsCreated;
    protected final JAXXContext delegateContext = new DefaultJAXXContext();

    /*-----------------------------------------------------------------------*/
    /*------------------------ Protected components  ------------------------*/
    /*-----------------------------------------------------------------------*/

    protected JTextArea actionLogArea;
    protected JSplitPane actionLogSplitPane;
    protected JPanel buttonBar;
    protected JButton buttonScriptCheck;
    protected JButton buttonScriptCommit;
    protected JButton buttonScriptCopy;
    protected JButton buttonScriptCut;
    protected JButton buttonScriptEval;
    protected JButton buttonScriptExport;
    protected JButton buttonScriptNew;
    protected JButton buttonScriptPaste;
    protected JButton buttonScriptSave;
    protected SyntaxEditor editor;
    protected Boolean fileSelected;
    protected ScriptHandler handler;
    protected Boolean javaFileSelected;
    protected JMenuItem miCheckAllSyntax;
    protected JMenuItem miCheckSyntax;
    protected JMenuItem miCommitVCS;
    protected JMenuItem miCopy;
    protected JMenuItem miCut;
    protected JMenuItem miDelete;
    protected JMenuItem miDeleteVCS;
    protected JMenuItem miDiffVCS;
    protected JMenuItem miEvaluate;
    protected JMenuItem miGenerateJavadoc;
    protected JMenuItem miGoto;
    protected JMenuItem miNewEquationModel;
    protected JMenuItem miNewExport;
    protected JMenuItem miNewExportStep;
    protected JMenuItem miNewResultInfo;
    protected JMenuItem miNewRule;
    protected JMenuItem miNewSImulationPlan;
    protected JMenuItem miNewScript;
    protected JMenuItem miNewSensitivity;
    protected JMenuItem miNewSensitivityExport;
    protected JMenuItem miNewSimulator;
    protected JMenuItem miPaste;
    protected JMenuItem miReplace;
    protected JMenuItem miSave;
    protected JMenuItem miSearch;
    protected JMenuItem miShowJavadoc;
    protected JMenuBar scriptMenuBar;
    protected JMenu scriptNewMenu;
    protected ScriptTree scriptTree;
    protected ScriptTreeModel scriptTreeModel;
    protected JMenu scriptTxtCode;
    protected JMenu scriptTxtEdit;
    protected JMenu scriptTxtFile;
    protected JMenu scriptTxtVCS;
    protected Boolean singleFileSelected;
    protected Boolean textSelected;

    /*-----------------------------------------------------------------------*/
    /*------------------------- Private components  -------------------------*/
    /*-----------------------------------------------------------------------*/

    private JMenuItem $JMenuItem0;
    private JMenuItem $JMenuItem1;
    private ScriptUI $JPanel0 = this;
    private JPanel $JPanel1;
    private JScrollPane $JScrollPane0;
    private JScrollPane $JScrollPane1;
    private JSeparator $JSeparator0;
    private JSeparator $JSeparator1;
    private JSeparator $JSeparator2;
    private JSeparator $JSeparator3;
    private JSplitPane $JSplitPane0;

    /*-----------------------------------------------------------------------*/
    /*---------------------- Raw body code from script ----------------------*/
    /*-----------------------------------------------------------------------*/

    protected void $afterCompleteSetup() {
            handler.afterInit();
        }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Constructors  ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public ScriptUI(LayoutManager param0, boolean param1) {
        super(param0 ,param1);
        $initialize();
    }

    public ScriptUI(JAXXContext parentContext, LayoutManager param1, boolean param2) {
        super(param1 ,param2);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public ScriptUI(LayoutManager param0) {
        super(param0);
        $initialize();
    }

    public ScriptUI(JAXXContext parentContext, LayoutManager param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public ScriptUI() {
        $initialize();
    }

    public ScriptUI(JAXXContext parentContext) {
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public ScriptUI(boolean param0) {
        super(param0);
        $initialize();
    }

    public ScriptUI(JAXXContext parentContext, boolean param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Statics methods ---------------------------*/
    /*-----------------------------------------------------------------------*/

    public static JAXXObjectDescriptor $getJAXXObjectDescriptor() {
        return JAXXUtil.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------- JAXXObject implementation ----------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public void applyDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).applyDataBinding();
        }
        processDataBinding($binding);
    }

    @Override
    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        super.firePropertyChange(propertyName, oldValue, newValue);
    }

    @Override
    public Map<String, Object> get$objectMap() {
        return $objectMap;
    }

    @Override
    public JAXXBinding getDataBinding(String bindingId) {
        return $bindings.get(bindingId);
    }

    @Override
    public JAXXBinding[] getDataBindings() {
        return $bindings.values().toArray(new JAXXBinding[$bindings.size()]);
    }

    @Override
    public Object getObjectById(String id) {
        return $objectMap.get(id);
    }

    @Override
    public void processDataBinding(String $binding, boolean $force) {
        if (!$force && $activeBindings.contains($binding)) { 
            return;
        }
        $activeBindings.add($binding);
        try {
            if (allComponentsCreated && $bindings.containsKey($binding)) {
                getDataBinding($binding).processDataBinding();
            }
        } finally {
            $activeBindings.remove($binding);
        }
    }

    @Override
    public void processDataBinding(String $binding) {
        processDataBinding($binding, false);
    }

    @Override
    public void registerDataBinding(JAXXBinding binding) {
        $bindings.put(binding.getId(), binding);
    }

    @Override
    public void removeDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).removeDataBinding();
        }
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- JAXXContext implementation  ---------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return delegateContext.getContextValue(clazz, null);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        return delegateContext.getContextValue(clazz, name);
    }

    @Override
    public JAXXContext getDelegateContext() {
        return delegateContext;
    }

    @Override
    public <O extends Container> O getParentContainer(Class<O> clazz) {
        return SwingUtil.getParentContainer(this, clazz);
    }

    @Override
    public <O extends Container> O getParentContainer(Object source, Class<O> clazz) {
        return SwingUtil.getParentContainer(source, clazz);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz) {
        delegateContext.removeContextValue(clazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz, String name) {
        delegateContext.removeContextValue(clazz, name);
    }

    @Override
    public <T> void setContextValue(T o) {
        delegateContext.setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        delegateContext.setContextValue(o, name);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Event methods ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public void doActionPerformed__on__$JMenuItem0(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityObjective);
    }

    public void doActionPerformed__on__$JMenuItem1(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityOptimization);
    }

    public void doActionPerformed__on__buttonScriptCheck(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.checkScript();
    }

    public void doActionPerformed__on__buttonScriptCommit(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.commitScript();
    }

    public void doActionPerformed__on__buttonScriptEval(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.evaluateScript();
    }

    public void doActionPerformed__on__buttonScriptExport(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.exportScript();
    }

    public void doActionPerformed__on__buttonScriptSave(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.saveScript();
    }

    public void doActionPerformed__on__miCheckAllSyntax(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.checkAllScripts();
    }

    public void doActionPerformed__on__miCheckSyntax(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.checkScript();
    }

    public void doActionPerformed__on__miCommitVCS(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.commitScript();
    }

    public void doActionPerformed__on__miDelete(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.deleteScript(false);
    }

    public void doActionPerformed__on__miDeleteVCS(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.deleteScript(true);
    }

    public void doActionPerformed__on__miDiffVCS(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.diffScript();
    }

    public void doActionPerformed__on__miEvaluate(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.evaluateScript();
    }

    public void doActionPerformed__on__miGenerateJavadoc(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.generateScriptJavadoc();
    }

    public void doActionPerformed__on__miNewEquationModel(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityFormule);
    }

    public void doActionPerformed__on__miNewExport(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityExport);
    }

    public void doActionPerformed__on__miNewExportStep(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityExportStep);
    }

    public void doActionPerformed__on__miNewResultInfo(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityResultInfo);
    }

    public void doActionPerformed__on__miNewRule(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityRule);
    }

    public void doActionPerformed__on__miNewSImulationPlan(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunitySimulationPlan);
    }

    public void doActionPerformed__on__miNewScript(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunityScript);
    }

    public void doActionPerformed__on__miNewSensitivity(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunitySensitivity);
    }

    public void doActionPerformed__on__miNewSensitivityExport(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunitySensitivityExport);
    }

    public void doActionPerformed__on__miNewSimulator(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(ScriptHandler.ScriptMapping.CommunitySimulator);
    }

    public void doActionPerformed__on__miSave(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.saveScript();
    }

    public void doActionPerformed__on__miShowJavadoc(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.showScriptJavadoc();
    }

    public void doMousePressed__on__buttonScriptNew(MouseEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.newScript(event);
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public acessor methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public JTextArea getActionLogArea() {
        return actionLogArea;
    }

    public JSplitPane getActionLogSplitPane() {
        return actionLogSplitPane;
    }

    public JPanel getButtonBar() {
        return buttonBar;
    }

    public JButton getButtonScriptCheck() {
        return buttonScriptCheck;
    }

    public JButton getButtonScriptCommit() {
        return buttonScriptCommit;
    }

    public JButton getButtonScriptCopy() {
        return buttonScriptCopy;
    }

    public JButton getButtonScriptCut() {
        return buttonScriptCut;
    }

    public JButton getButtonScriptEval() {
        return buttonScriptEval;
    }

    public JButton getButtonScriptExport() {
        return buttonScriptExport;
    }

    public JButton getButtonScriptNew() {
        return buttonScriptNew;
    }

    public JButton getButtonScriptPaste() {
        return buttonScriptPaste;
    }

    public JButton getButtonScriptSave() {
        return buttonScriptSave;
    }

    public SyntaxEditor getEditor() {
        return editor;
    }

    public Boolean getFileSelected() {
        return fileSelected;
    }

    public ScriptHandler getHandler() {
        return handler;
    }

    public Boolean getJavaFileSelected() {
        return javaFileSelected;
    }

    public JMenuItem getMiCheckAllSyntax() {
        return miCheckAllSyntax;
    }

    public JMenuItem getMiCheckSyntax() {
        return miCheckSyntax;
    }

    public JMenuItem getMiCommitVCS() {
        return miCommitVCS;
    }

    public JMenuItem getMiCopy() {
        return miCopy;
    }

    public JMenuItem getMiCut() {
        return miCut;
    }

    public JMenuItem getMiDelete() {
        return miDelete;
    }

    public JMenuItem getMiDeleteVCS() {
        return miDeleteVCS;
    }

    public JMenuItem getMiDiffVCS() {
        return miDiffVCS;
    }

    public JMenuItem getMiEvaluate() {
        return miEvaluate;
    }

    public JMenuItem getMiGenerateJavadoc() {
        return miGenerateJavadoc;
    }

    public JMenuItem getMiGoto() {
        return miGoto;
    }

    public JMenuItem getMiNewEquationModel() {
        return miNewEquationModel;
    }

    public JMenuItem getMiNewExport() {
        return miNewExport;
    }

    public JMenuItem getMiNewExportStep() {
        return miNewExportStep;
    }

    public JMenuItem getMiNewResultInfo() {
        return miNewResultInfo;
    }

    public JMenuItem getMiNewRule() {
        return miNewRule;
    }

    public JMenuItem getMiNewSImulationPlan() {
        return miNewSImulationPlan;
    }

    public JMenuItem getMiNewScript() {
        return miNewScript;
    }

    public JMenuItem getMiNewSensitivity() {
        return miNewSensitivity;
    }

    public JMenuItem getMiNewSensitivityExport() {
        return miNewSensitivityExport;
    }

    public JMenuItem getMiNewSimulator() {
        return miNewSimulator;
    }

    public JMenuItem getMiPaste() {
        return miPaste;
    }

    public JMenuItem getMiReplace() {
        return miReplace;
    }

    public JMenuItem getMiSave() {
        return miSave;
    }

    public JMenuItem getMiSearch() {
        return miSearch;
    }

    public JMenuItem getMiShowJavadoc() {
        return miShowJavadoc;
    }

    public JMenuBar getScriptMenuBar() {
        return scriptMenuBar;
    }

    public JMenu getScriptNewMenu() {
        return scriptNewMenu;
    }

    public ScriptTree getScriptTree() {
        return scriptTree;
    }

    public ScriptTreeModel getScriptTreeModel() {
        return scriptTreeModel;
    }

    public JMenu getScriptTxtCode() {
        return scriptTxtCode;
    }

    public JMenu getScriptTxtEdit() {
        return scriptTxtEdit;
    }

    public JMenu getScriptTxtFile() {
        return scriptTxtFile;
    }

    public JMenu getScriptTxtVCS() {
        return scriptTxtVCS;
    }

    public Boolean getSingleFileSelected() {
        return singleFileSelected;
    }

    public Boolean getTextSelected() {
        return textSelected;
    }

    public Boolean isFileSelected() {
        return fileSelected !=null && fileSelected;
    }

    public Boolean isJavaFileSelected() {
        return javaFileSelected !=null && javaFileSelected;
    }

    public Boolean isSingleFileSelected() {
        return singleFileSelected !=null && singleFileSelected;
    }

    public Boolean isTextSelected() {
        return textSelected !=null && textSelected;
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public mutator methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public void setFileSelected(Boolean fileSelected) {
        Boolean oldValue = this.fileSelected;
        this.fileSelected = fileSelected;
        firePropertyChange(PROPERTY_FILE_SELECTED, oldValue, fileSelected);
    }

    public void setJavaFileSelected(Boolean javaFileSelected) {
        Boolean oldValue = this.javaFileSelected;
        this.javaFileSelected = javaFileSelected;
        firePropertyChange(PROPERTY_JAVA_FILE_SELECTED, oldValue, javaFileSelected);
    }

    public void setSingleFileSelected(Boolean singleFileSelected) {
        Boolean oldValue = this.singleFileSelected;
        this.singleFileSelected = singleFileSelected;
        firePropertyChange(PROPERTY_SINGLE_FILE_SELECTED, oldValue, singleFileSelected);
    }

    public void setTextSelected(Boolean textSelected) {
        Boolean oldValue = this.textSelected;
        this.textSelected = textSelected;
        firePropertyChange(PROPERTY_TEXT_SELECTED, oldValue, textSelected);
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Protected acessors methods  ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected JMenuItem get$JMenuItem0() {
        return $JMenuItem0;
    }

    protected JMenuItem get$JMenuItem1() {
        return $JMenuItem1;
    }

    protected JPanel get$JPanel1() {
        return $JPanel1;
    }

    protected JScrollPane get$JScrollPane0() {
        return $JScrollPane0;
    }

    protected JScrollPane get$JScrollPane1() {
        return $JScrollPane1;
    }

    protected JSeparator get$JSeparator0() {
        return $JSeparator0;
    }

    protected JSeparator get$JSeparator1() {
        return $JSeparator1;
    }

    protected JSeparator get$JSeparator2() {
        return $JSeparator2;
    }

    protected JSeparator get$JSeparator3() {
        return $JSeparator3;
    }

    protected JSplitPane get$JSplitPane0() {
        return $JSplitPane0;
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Components creation methods ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected void addChildrenToActionLogSplitPane() {
        if (!allComponentsCreated) {
            return;
        }
        actionLogSplitPane.add(editor, JSplitPane.LEFT);
        actionLogSplitPane.add($JScrollPane1, JSplitPane.RIGHT);
    }

    protected void addChildrenToButtonBar() {
        if (!allComponentsCreated) {
            return;
        }
        buttonBar.add(buttonScriptNew);
        buttonBar.add(buttonScriptSave);
        buttonBar.add(buttonScriptExport);
        buttonBar.add(buttonScriptCommit);
        buttonBar.add(buttonScriptCut);
        buttonBar.add(buttonScriptCopy);
        buttonBar.add(buttonScriptPaste);
        buttonBar.add(buttonScriptCheck);
        buttonBar.add(buttonScriptEval);
    }

    protected void addChildrenToScriptMenuBar() {
        if (!allComponentsCreated) {
            return;
        }
        scriptMenuBar.add(scriptTxtFile);
        scriptMenuBar.add(scriptTxtEdit);
        scriptMenuBar.add(scriptTxtVCS);
        scriptMenuBar.add(scriptTxtCode);
    }

    protected void addChildrenToScriptNewMenu() {
        if (!allComponentsCreated) {
            return;
        }
        scriptNewMenu.add(miNewScript);
        scriptNewMenu.add(miNewSimulator);
        scriptNewMenu.add(miNewExport);
        scriptNewMenu.add(miNewExportStep);
        scriptNewMenu.add(miNewRule);
        scriptNewMenu.add(miNewResultInfo);
        scriptNewMenu.add(miNewSImulationPlan);
        scriptNewMenu.add(miNewSensitivity);
        scriptNewMenu.add(miNewSensitivityExport);
        scriptNewMenu.add($JMenuItem0);
        scriptNewMenu.add($JMenuItem1);
        scriptNewMenu.add(miNewEquationModel);
    }

    protected void addChildrenToScriptTxtCode() {
        if (!allComponentsCreated) {
            return;
        }
        scriptTxtCode.add(miCheckSyntax);
        scriptTxtCode.add(miCheckAllSyntax);
        scriptTxtCode.add(miEvaluate);
        scriptTxtCode.add($JSeparator3);
        scriptTxtCode.add(miGenerateJavadoc);
        scriptTxtCode.add(miShowJavadoc);
    }

    protected void addChildrenToScriptTxtEdit() {
        if (!allComponentsCreated) {
            return;
        }
        scriptTxtEdit.add(miCut);
        scriptTxtEdit.add(miCopy);
        scriptTxtEdit.add(miPaste);
        scriptTxtEdit.add($JSeparator1);
        scriptTxtEdit.add(miSearch);
        scriptTxtEdit.add(miReplace);
        scriptTxtEdit.add(miGoto);
    }

    protected void addChildrenToScriptTxtFile() {
        if (!allComponentsCreated) {
            return;
        }
        scriptTxtFile.add(scriptNewMenu);
        scriptTxtFile.add(miSave);
        scriptTxtFile.add($JSeparator0);
        scriptTxtFile.add(miDelete);
    }

    protected void addChildrenToScriptTxtVCS() {
        if (!allComponentsCreated) {
            return;
        }
        scriptTxtVCS.add(miCommitVCS);
        scriptTxtVCS.add(miDiffVCS);
        scriptTxtVCS.add($JSeparator2);
        scriptTxtVCS.add(miDeleteVCS);
    }

    protected void createActionLogArea() {
        $objectMap.put("actionLogArea", actionLogArea = new JTextArea());
        
        actionLogArea.setName("actionLogArea");
        actionLogArea.setColumns(15);
        actionLogArea.setLineWrap(true);
        actionLogArea.setWrapStyleWord(true);
        actionLogArea.setEditable(false);
    }

    protected void createActionLogSplitPane() {
        $objectMap.put("actionLogSplitPane", actionLogSplitPane = new JSplitPane());
        
        actionLogSplitPane.setName("actionLogSplitPane");
        actionLogSplitPane.setOneTouchExpandable(true);
        actionLogSplitPane.setOrientation(0);
    }

    protected void createButtonBar() {
        $objectMap.put("buttonBar", buttonBar = new JPanel());
        
        buttonBar.setName("buttonBar");
        buttonBar.setLayout(new GridLayout());
    }

    protected void createButtonScriptCheck() {
        $objectMap.put("buttonScriptCheck", buttonScriptCheck = new JButton());
        
        buttonScriptCheck.setName("buttonScriptCheck");
        buttonScriptCheck.setText(t("isisfish.script.check"));
        buttonScriptCheck.setToolTipText(t("isisfish.script.check.tooltip"));
        buttonScriptCheck.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__buttonScriptCheck"));
    }

    protected void createButtonScriptCommit() {
        $objectMap.put("buttonScriptCommit", buttonScriptCommit = new JButton());
        
        buttonScriptCommit.setName("buttonScriptCommit");
        buttonScriptCommit.setText(t("isisfish.script.commit"));
        buttonScriptCommit.setToolTipText(t("isisfish.script.commit.tooltip"));
        buttonScriptCommit.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__buttonScriptCommit"));
    }

    protected void createButtonScriptCopy() {
        $objectMap.put("buttonScriptCopy", buttonScriptCopy = new JButton());
        
        buttonScriptCopy.setName("buttonScriptCopy");
        buttonScriptCopy.setText(t("isisfish.script.copy"));
        buttonScriptCopy.setToolTipText(t("isisfish.script.copy.tooltip"));
    }

    protected void createButtonScriptCut() {
        $objectMap.put("buttonScriptCut", buttonScriptCut = new JButton());
        
        buttonScriptCut.setName("buttonScriptCut");
        buttonScriptCut.setText(t("isisfish.script.cut"));
        buttonScriptCut.setToolTipText(t("isisfish.script.cut.tooltip"));
    }

    protected void createButtonScriptEval() {
        $objectMap.put("buttonScriptEval", buttonScriptEval = new JButton());
        
        buttonScriptEval.setName("buttonScriptEval");
        buttonScriptEval.setText(t("isisfish.script.evaluate"));
        buttonScriptEval.setToolTipText(t("isisfish.script.evaluate.tooltip"));
        buttonScriptEval.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__buttonScriptEval"));
    }

    protected void createButtonScriptExport() {
        $objectMap.put("buttonScriptExport", buttonScriptExport = new JButton());
        
        buttonScriptExport.setName("buttonScriptExport");
        buttonScriptExport.setEnabled(false);
        buttonScriptExport.setText(t("isisfish.script.export"));
        buttonScriptExport.setToolTipText(t("isisfish.script.export.tooltip"));
        buttonScriptExport.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__buttonScriptExport"));
    }

    protected void createButtonScriptNew() {
        $objectMap.put("buttonScriptNew", buttonScriptNew = new JButton());
        
        buttonScriptNew.setName("buttonScriptNew");
        buttonScriptNew.setText(t("isisfish.script.newplus"));
        buttonScriptNew.setToolTipText(t("isisfish.script.new.tooltip"));
        buttonScriptNew.addMouseListener(JAXXUtil.getEventListener(MouseListener.class, "mousePressed", this, "doMousePressed__on__buttonScriptNew"));
    }

    protected void createButtonScriptPaste() {
        $objectMap.put("buttonScriptPaste", buttonScriptPaste = new JButton());
        
        buttonScriptPaste.setName("buttonScriptPaste");
        buttonScriptPaste.setText(t("isisfish.script.paste"));
        buttonScriptPaste.setToolTipText(t("isisfish.script.paste.tooltip"));
    }

    protected void createButtonScriptSave() {
        $objectMap.put("buttonScriptSave", buttonScriptSave = new JButton());
        
        buttonScriptSave.setName("buttonScriptSave");
        buttonScriptSave.setText(t("isisfish.script.save"));
        buttonScriptSave.setToolTipText(t("isisfish.script.save.tooltip"));
        buttonScriptSave.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__buttonScriptSave"));
    }

    protected void createEditor() {
        $objectMap.put("editor", editor = new SyntaxEditor());
        
        editor.setName("editor");
        editor.setAskIfNotSaved(true);
    }

    protected void createFileSelected() {
        $objectMap.put("fileSelected", fileSelected = false);
    }

    protected void createHandler() {
        $objectMap.put("handler", handler = new ScriptHandler(this));
    }

    protected void createJavaFileSelected() {
        $objectMap.put("javaFileSelected", javaFileSelected = false);
    }

    protected void createMiCheckAllSyntax() {
        $objectMap.put("miCheckAllSyntax", miCheckAllSyntax = new JMenuItem());
        
        miCheckAllSyntax.setName("miCheckAllSyntax");
        miCheckAllSyntax.setText(t("isisfish.script.menu.checkAllSyntax"));
        miCheckAllSyntax.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miCheckAllSyntax"));
    }

    protected void createMiCheckSyntax() {
        $objectMap.put("miCheckSyntax", miCheckSyntax = new JMenuItem());
        
        miCheckSyntax.setName("miCheckSyntax");
        miCheckSyntax.setText(t("isisfish.script.menu.checkSyntax"));
        miCheckSyntax.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miCheckSyntax"));
    }

    protected void createMiCommitVCS() {
        $objectMap.put("miCommitVCS", miCommitVCS = new JMenuItem());
        
        miCommitVCS.setName("miCommitVCS");
        miCommitVCS.setText(t("isisfish.script.menu.commit"));
        miCommitVCS.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miCommitVCS"));
    }

    protected void createMiCopy() {
        $objectMap.put("miCopy", miCopy = new JMenuItem());
        
        miCopy.setName("miCopy");
    }

    protected void createMiCut() {
        $objectMap.put("miCut", miCut = new JMenuItem());
        
        miCut.setName("miCut");
    }

    protected void createMiDelete() {
        $objectMap.put("miDelete", miDelete = new JMenuItem());
        
        miDelete.setName("miDelete");
        miDelete.setText(t("isisfish.script.menu.deleteLocaly"));
        miDelete.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miDelete"));
    }

    protected void createMiDeleteVCS() {
        $objectMap.put("miDeleteVCS", miDeleteVCS = new JMenuItem());
        
        miDeleteVCS.setName("miDeleteVCS");
        miDeleteVCS.setText(t("isisfish.script.menu.deleteLocalyServer"));
        miDeleteVCS.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miDeleteVCS"));
    }

    protected void createMiDiffVCS() {
        $objectMap.put("miDiffVCS", miDiffVCS = new JMenuItem());
        
        miDiffVCS.setName("miDiffVCS");
        miDiffVCS.setAccelerator(null);
        miDiffVCS.setText(t("isisfish.script.menu.diff"));
        miDiffVCS.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miDiffVCS"));
    }

    protected void createMiEvaluate() {
        $objectMap.put("miEvaluate", miEvaluate = new JMenuItem());
        
        miEvaluate.setName("miEvaluate");
        miEvaluate.setText(t("isisfish.script.menu.evaluate"));
        miEvaluate.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miEvaluate"));
    }

    protected void createMiGenerateJavadoc() {
        $objectMap.put("miGenerateJavadoc", miGenerateJavadoc = new JMenuItem());
        
        miGenerateJavadoc.setName("miGenerateJavadoc");
        miGenerateJavadoc.setText(t("isisfish.script.menu.javadocgenerate"));
        miGenerateJavadoc.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miGenerateJavadoc"));
    }

    protected void createMiGoto() {
        $objectMap.put("miGoto", miGoto = new JMenuItem());
        
        miGoto.setName("miGoto");
    }

    protected void createMiNewEquationModel() {
        $objectMap.put("miNewEquationModel", miNewEquationModel = new JMenuItem());
        
        miNewEquationModel.setName("miNewEquationModel");
        miNewEquationModel.setText(t("isisfish.script.menu.txtNewEquationModel"));
        miNewEquationModel.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewEquationModel"));
    }

    protected void createMiNewExport() {
        $objectMap.put("miNewExport", miNewExport = new JMenuItem());
        
        miNewExport.setName("miNewExport");
        miNewExport.setText(t("isisfish.script.menu.txtNewExport"));
        miNewExport.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewExport"));
    }

    protected void createMiNewExportStep() {
        $objectMap.put("miNewExportStep", miNewExportStep = new JMenuItem());
        
        miNewExportStep.setName("miNewExportStep");
        miNewExportStep.setText(t("isisfish.script.menu.txtNewExportStep"));
        miNewExportStep.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewExportStep"));
    }

    protected void createMiNewResultInfo() {
        $objectMap.put("miNewResultInfo", miNewResultInfo = new JMenuItem());
        
        miNewResultInfo.setName("miNewResultInfo");
        miNewResultInfo.setText(t("isisfish.script.menu.txtNewResultInfo"));
        miNewResultInfo.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewResultInfo"));
    }

    protected void createMiNewRule() {
        $objectMap.put("miNewRule", miNewRule = new JMenuItem());
        
        miNewRule.setName("miNewRule");
        miNewRule.setText(t("isisfish.script.menu.txtNewRule"));
        miNewRule.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewRule"));
    }

    protected void createMiNewSImulationPlan() {
        $objectMap.put("miNewSImulationPlan", miNewSImulationPlan = new JMenuItem());
        
        miNewSImulationPlan.setName("miNewSImulationPlan");
        miNewSImulationPlan.setText(t("isisfish.script.menu.txtNewSimulationPlan"));
        miNewSImulationPlan.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewSImulationPlan"));
    }

    protected void createMiNewScript() {
        $objectMap.put("miNewScript", miNewScript = new JMenuItem());
        
        miNewScript.setName("miNewScript");
        miNewScript.setText(t("isisfish.script.menu.txtNewScript"));
        miNewScript.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewScript"));
    }

    protected void createMiNewSensitivity() {
        $objectMap.put("miNewSensitivity", miNewSensitivity = new JMenuItem());
        
        miNewSensitivity.setName("miNewSensitivity");
        miNewSensitivity.setText(t("isisfish.script.menu.txtNewSensitivity"));
        miNewSensitivity.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewSensitivity"));
    }

    protected void createMiNewSensitivityExport() {
        $objectMap.put("miNewSensitivityExport", miNewSensitivityExport = new JMenuItem());
        
        miNewSensitivityExport.setName("miNewSensitivityExport");
        miNewSensitivityExport.setText(t("isisfish.script.menu.txtNewSensitivityExport"));
        miNewSensitivityExport.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewSensitivityExport"));
    }

    protected void createMiNewSimulator() {
        $objectMap.put("miNewSimulator", miNewSimulator = new JMenuItem());
        
        miNewSimulator.setName("miNewSimulator");
        miNewSimulator.setText(t("isisfish.script.menu.txtNewSimulator"));
        miNewSimulator.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miNewSimulator"));
    }

    protected void createMiPaste() {
        $objectMap.put("miPaste", miPaste = new JMenuItem());
        
        miPaste.setName("miPaste");
    }

    protected void createMiReplace() {
        $objectMap.put("miReplace", miReplace = new JMenuItem());
        
        miReplace.setName("miReplace");
    }

    protected void createMiSave() {
        $objectMap.put("miSave", miSave = new JMenuItem());
        
        miSave.setName("miSave");
        miSave.setText(t("isisfish.script.menu.save"));
        miSave.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miSave"));
    }

    protected void createMiSearch() {
        $objectMap.put("miSearch", miSearch = new JMenuItem());
        
        miSearch.setName("miSearch");
    }

    protected void createMiShowJavadoc() {
        $objectMap.put("miShowJavadoc", miShowJavadoc = new JMenuItem());
        
        miShowJavadoc.setName("miShowJavadoc");
        miShowJavadoc.setText(t("isisfish.script.menu.javadocshow"));
        miShowJavadoc.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__miShowJavadoc"));
    }

    protected void createScriptMenuBar() {
        $objectMap.put("scriptMenuBar", scriptMenuBar = new JMenuBar());
        
        scriptMenuBar.setName("scriptMenuBar");
    }

    protected void createScriptNewMenu() {
        $objectMap.put("scriptNewMenu", scriptNewMenu = new JMenu());
        
        scriptNewMenu.setName("scriptNewMenu");
        scriptNewMenu.setText(t("isisfish.script.menu.new"));
    }

    protected void createScriptTree() {
        $objectMap.put("scriptTree", scriptTree = new ScriptTree(this));
        
        scriptTree.setName("scriptTree");
        scriptTree.setRootVisible(false);
    }

    protected void createScriptTreeModel() {
        $objectMap.put("scriptTreeModel", scriptTreeModel = new ScriptTreeModel());
    }

    protected void createScriptTxtCode() {
        $objectMap.put("scriptTxtCode", scriptTxtCode = new JMenu());
        
        scriptTxtCode.setName("scriptTxtCode");
        scriptTxtCode.setText(t("isisfish.script.menu.code"));
    }

    protected void createScriptTxtEdit() {
        $objectMap.put("scriptTxtEdit", scriptTxtEdit = new JMenu());
        
        scriptTxtEdit.setName("scriptTxtEdit");
        scriptTxtEdit.setText(t("isisfish.script.menu.edit"));
    }

    protected void createScriptTxtFile() {
        $objectMap.put("scriptTxtFile", scriptTxtFile = new JMenu());
        
        scriptTxtFile.setName("scriptTxtFile");
        scriptTxtFile.setText(t("isisfish.script.menu.file"));
    }

    protected void createScriptTxtVCS() {
        $objectMap.put("scriptTxtVCS", scriptTxtVCS = new JMenu());
        
        scriptTxtVCS.setName("scriptTxtVCS");
        scriptTxtVCS.setText(t("isisfish.script.menu.txtVCS"));
    }

    protected void createSingleFileSelected() {
        $objectMap.put("singleFileSelected", singleFileSelected = false);
    }

    protected void createTextSelected() {
        $objectMap.put("textSelected", textSelected = false);
    }

    /*-----------------------------------------------------------------------*/
    /*------------------------ Internal jaxx methods ------------------------*/
    /*-----------------------------------------------------------------------*/

    private void $completeSetup() {
        allComponentsCreated = true;
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        // inline complete setup of $JPanel0
        add(scriptMenuBar, BorderLayout.NORTH);
        add($JPanel1, BorderLayout.CENTER);
        addChildrenToScriptMenuBar();
        addChildrenToScriptTxtFile();
        addChildrenToScriptNewMenu();
        addChildrenToScriptTxtEdit();
        addChildrenToScriptTxtVCS();
        addChildrenToScriptTxtCode();
        // inline complete setup of $JPanel1
        $JPanel1.add(buttonBar, BorderLayout.NORTH);
        $JPanel1.add($JSplitPane0, BorderLayout.CENTER);
        addChildrenToButtonBar();
        // inline complete setup of $JSplitPane0
        $JSplitPane0.add($JScrollPane0, JSplitPane.LEFT);
        $JSplitPane0.add(actionLogSplitPane, JSplitPane.RIGHT);
        // inline complete setup of $JScrollPane0
        $JScrollPane0.getViewport().add(scriptTree);
        addChildrenToActionLogSplitPane();
        // inline complete setup of $JScrollPane1
        $JScrollPane1.getViewport().add(actionLogArea);
        
        // apply 11 data bindings
        JAXXUtil.applyDataBinding(this, $bindings.keySet());
        
        // apply 33 property setters
        scriptNewMenu.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewScript.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewSimulator.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewExport.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewExportStep.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewRule.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewResultInfo.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewSImulationPlan.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewSensitivity.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewSensitivityExport.setIcon(SwingUtil.createImageIcon("script/script.png"));
        $JMenuItem0.setIcon(SwingUtil.createImageIcon("script/script.png"));
        $JMenuItem1.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miNewEquationModel.setIcon(SwingUtil.createImageIcon("script/script.png"));
        miSave.setIcon(SwingUtil.createImageIcon("script/disk-black.png"));
        miDelete.setIcon(SwingUtil.createImageIcon("script/script--minus.png"));
        miCommitVCS.setIcon(SwingUtil.createImageIcon("script/database-import.png"));
        miDiffVCS.setIcon(SwingUtil.createImageIcon("script/edit-diff.png"));
        miDeleteVCS.setIcon(SwingUtil.createImageIcon("script/database--minus.png"));
        miCheckSyntax.setIcon(SwingUtil.createImageIcon("script/compile.png"));
        miCheckAllSyntax.setIcon(SwingUtil.createImageIcon("script/script-code.png"));
        miEvaluate.setIcon(SwingUtil.createImageIcon("script/terminal.png"));
        buttonScriptNew.setIcon(SwingUtil.createImageIcon("script/script.png"));
        buttonScriptSave.setIcon(SwingUtil.createImageIcon("script/disk-black.png"));
        buttonScriptExport.setIcon(SwingUtil.createImageIcon("script/folder-zipper.png"));
        buttonScriptCommit.setIcon(SwingUtil.createImageIcon("script/database-import.png"));
        buttonScriptCheck.setIcon(SwingUtil.createImageIcon("script/compile.png"));
        buttonScriptEval.setIcon(SwingUtil.createImageIcon("script/terminal.png"));
        scriptTree.setCellRenderer(new fr.ifremer.isisfish.ui.script.model.ScriptTreeCellRenderer());
        scriptTree.setModel(scriptTreeModel);
        actionLogArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
        $afterCompleteSetup();
    }

    private void $initialize() {
        if (allComponentsCreated) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        $objectMap.put("$JPanel0", $JPanel0);
        createHandler();
        createFileSelected();
        createSingleFileSelected();
        createTextSelected();
        createJavaFileSelected();
        createScriptMenuBar();
        createScriptTxtFile();
        createScriptNewMenu();
        createMiNewScript();
        createMiNewSimulator();
        createMiNewExport();
        createMiNewExportStep();
        createMiNewRule();
        createMiNewResultInfo();
        createMiNewSImulationPlan();
        createMiNewSensitivity();
        createMiNewSensitivityExport();
        // inline creation of $JMenuItem0
        $objectMap.put("$JMenuItem0", $JMenuItem0 = new JMenuItem());
        
        $JMenuItem0.setName("$JMenuItem0");
        $JMenuItem0.setText(t("isisfish.script.menu.txtNewObjective"));
        $JMenuItem0.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__$JMenuItem0"));
        // inline creation of $JMenuItem1
        $objectMap.put("$JMenuItem1", $JMenuItem1 = new JMenuItem());
        
        $JMenuItem1.setName("$JMenuItem1");
        $JMenuItem1.setText(t("isisfish.script.menu.txtNewOptimization"));
        $JMenuItem1.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__$JMenuItem1"));
        createMiNewEquationModel();
        createMiSave();
        // inline creation of $JSeparator0
        $objectMap.put("$JSeparator0", $JSeparator0 = new JSeparator());
        
        $JSeparator0.setName("$JSeparator0");
        createMiDelete();
        createScriptTxtEdit();
        createMiCut();
        createMiCopy();
        createMiPaste();
        // inline creation of $JSeparator1
        $objectMap.put("$JSeparator1", $JSeparator1 = new JSeparator());
        
        $JSeparator1.setName("$JSeparator1");
        createMiSearch();
        createMiReplace();
        createMiGoto();
        createScriptTxtVCS();
        createMiCommitVCS();
        createMiDiffVCS();
        // inline creation of $JSeparator2
        $objectMap.put("$JSeparator2", $JSeparator2 = new JSeparator());
        
        $JSeparator2.setName("$JSeparator2");
        createMiDeleteVCS();
        createScriptTxtCode();
        createMiCheckSyntax();
        createMiCheckAllSyntax();
        createMiEvaluate();
        // inline creation of $JSeparator3
        $objectMap.put("$JSeparator3", $JSeparator3 = new JSeparator());
        
        $JSeparator3.setName("$JSeparator3");
        createMiGenerateJavadoc();
        createMiShowJavadoc();
        // inline creation of $JPanel1
        $objectMap.put("$JPanel1", $JPanel1 = new JPanel());
        
        $JPanel1.setName("$JPanel1");
        $JPanel1.setLayout(new BorderLayout());
        createButtonBar();
        createButtonScriptNew();
        createButtonScriptSave();
        createButtonScriptExport();
        createButtonScriptCommit();
        createButtonScriptCut();
        createButtonScriptCopy();
        createButtonScriptPaste();
        createButtonScriptCheck();
        createButtonScriptEval();
        // inline creation of $JSplitPane0
        $objectMap.put("$JSplitPane0", $JSplitPane0 = new JSplitPane());
        
        $JSplitPane0.setName("$JSplitPane0");
        $JSplitPane0.setDividerLocation(250);
        $JSplitPane0.setOneTouchExpandable(true);
        $JSplitPane0.setOrientation(1);
        // inline creation of $JScrollPane0
        $objectMap.put("$JScrollPane0", $JScrollPane0 = new JScrollPane());
        
        $JScrollPane0.setName("$JScrollPane0");
        createScriptTreeModel();
        createScriptTree();
        createActionLogSplitPane();
        createEditor();
        // inline creation of $JScrollPane1
        $objectMap.put("$JScrollPane1", $JScrollPane1 = new JScrollPane());
        
        $JScrollPane1.setName("$JScrollPane1");
        createActionLogArea();
        // inline creation of $JPanel0
        setName("$JPanel0");
        setLayout(new BorderLayout());
        
        // registers 11 data bindings
        $registerDefaultBindings();
        $completeSetup();
    }

    private void $registerDefaultBindings() {
        // register 11 data bindings
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_SAVE_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                miSave.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_DELETE_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                miDelete.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_COMMIT_VCS_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                miCommitVCS.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_DIFF_VCS_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                miDiffVCS.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_CHECK_SYNTAX_ENABLED, true ,"javaFileSelected") {
        
            @Override
            public void processDataBinding() {
                miCheckSyntax.setEnabled(isJavaFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MI_EVALUATE_ENABLED, true ,"javaFileSelected") {
        
            @Override
            public void processDataBinding() {
                miEvaluate.setEnabled(isJavaFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_SCRIPT_SAVE_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                buttonScriptSave.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_SCRIPT_COMMIT_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                buttonScriptCommit.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_SCRIPT_PASTE_ENABLED, true ,"singleFileSelected") {
        
            @Override
            public void processDataBinding() {
                buttonScriptPaste.setEnabled(isSingleFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_SCRIPT_CHECK_ENABLED, true ,"javaFileSelected") {
        
            @Override
            public void processDataBinding() {
                buttonScriptCheck.setEnabled(isJavaFileSelected());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_SCRIPT_EVAL_ENABLED, true ,"javaFileSelected") {
        
            @Override
            public void processDataBinding() {
                buttonScriptEval.setEnabled(isJavaFileSelected());
            }
        });
    }

}