/*
 * #%L
 * IsisFish
 * 
 * $Id: LineReader.java 4156 2014-12-09 11:27:18Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/logging/io/LineReader.java $
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, CodeLutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging.io;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * A lineReader reads lines from a file using a {@link java.io.RandomAccessFile}.
 * 
 * To perform efficient io operations, we use a {@link OffsetReader} to obtain
 * the offset of the first char of a line.
 * 
 * You can create a new LineReader from a previous one, the matchings lines of
 * the new LineReader will all match the parent one (subset LineReader).
 *
 * @author chemit
 */
public class LineReader {

    protected static final Log log = LogFactory.getLog(LineReader.class);

    /*  file to read */
    protected File file;

    /** file reader */
    protected RandomAccessFile reader;

    /** size of the file after last synchronization */
    protected long length;
    /** last modified time of the file after last synchronization */
    protected long lastModified;

    /** parent LineReader */
    protected LineReader parent;

    /** offset reader used */
    protected OffsetReader offstReader;

    /** an identifier for the reader */
    protected String id;

    public LineReader(LineReader parent, OffsetReader offstReader) {
        this(parent.getFile(), offstReader);
        this.parent = parent;
    }

    public LineReader(File file, OffsetReader offstReader) {
        this.file = file;
        length = this.file.length();
        lastModified = this.file.lastModified();
        this.offstReader = offstReader;
    }

    /**
     * method we must inovke before any access to lines
     *
     * @throws IOException if any probem while opening reader
     */
    public void open() throws IOException {
        if (parent != null) {
            // the parent LineReader must be already opened
            parent.ensureOpen();
        }
        try {
            reader = new RandomAccessFile(this.file, "r");
            offstReader.open(this);
        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException(e);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Close the reader.
     *
     * @throws IOException if any problem while closing reader
     */
    public void close() throws IOException {
        log.info(this);
        IOException error = null;
        try {
            if (reader != null) {
                reader.close();
            }
        } catch (IOException e) {
            error = e;
        } finally {
            reader = null;
        }
        if (offstReader != null) {
            offstReader.close();
        }
        if (error != null) {
            throw error;
        }
    }

    /**
     * Read a line from a given position.
     *
     * @param position position of the line in the file
     * @return the line found, or null
     * @throws IOException if any problem while reading
     */
    public String readLine2(long position) throws IOException {
        ensureOpen();
        if (position >= offstReader.getNbLines()) {
            return null;
        }
        // find offset
        long offset = offstReader.getOffset(position);
        String result = null;
        if (offset > -1) {
            // we found a offset
            reader.seek(offset);
            result = reader.readLine();
        }
        return result;
    }

    /**
     * Read some line(s) from a given position.
     *
     * @param position position of the line in the file
     * @param length   the number of lines we want
     * @return the array of lines required, or null if eof
     * @throws IOException if any problem while reading
     */
    public String[] readLine2(long position, int length) throws IOException {
        ensureOpen();
        long offset = offstReader.getOffset(position);
        if (offset == -1) {
            // we are out ouf range
            return new String[0];
        }

        // the offset of the last line to be access
        long endOffset = offstReader.getOffset((position + length));

        // define the size of return
        int size = (int) (endOffset == -1 ? offstReader.getNbLines() - position : length);

        String[] result = new String[size];

        for (int i = 0; i < size; i++) {
            offset = offstReader.getOffset(position + i);
            if (reader.getFilePointer() != offset) {
                reader.seek(offset);
            }
            result[i] = reader.readLine();
        }
        return result;
    }

    /**
     * Read a line from a given position.
     *
     * @param position position of the line in the file
     * @return the line found, or null
     * @throws IOException if any problem while reading
     */
    public String readLine(long position) throws IOException {
        ensureOpen();
        if (position >= offstReader.getNbLines()) {
            return null;
        }
        // find offset
        long offset = offstReader.getOffset(position);
        String result = null;

        if (offset > -1) {
            // we found a offset, find next line offset
            reader.seek(offset);
            long offset2 = offstReader.getOffset(position + 1);
            if (offset2 == -1) {
                offset2 = reader.length();
            }
            int delta = (int) (offset2 - offset - 1);
            StringBuilder sb = new StringBuilder(delta);
            for (int i = 0; i < delta; i++) {
                try {
                    sb.append((char) reader.read());
                } catch (IOException e) {
                    // ignore
                }
            }
            result = sb.toString();
        }
        return result;
    }

    /**
     * Read some line(s) from a given position.
     *
     * @param position position of the line in the file
     * @param length   the number of lines we want
     * @return the array of lines required, or null if eof
     * @throws java.io.IOException if any problem while reading
     */
    public String[] readLine(long position, int length) throws IOException {
        ensureOpen();
        long offset = offstReader.getOffset(position);
        if (offset == -1) {
            // we are out ouf range
            return new String[0];
        }

        // the offset of the last line to be access
        long endOffset = offstReader.getOffset((position + length));

        // define the size of return
        int size = (int) (endOffset == -1 ? offstReader.getNbLines() - position : length);

        String[] result = new String[size];

        for (int i = 0; i < size; i++) {
            offset = offstReader.getOffset(position + i);
            if (reader.getFilePointer() != offset) {
                reader.seek(offset);
            }
            long offset2 = offstReader.getOffset(position + i + 1);
            if (offset2 == -1) {
                //offset2 = reader.length() - 1;
                result[i] = readLine2(position+i);
                //System.out.println("last line "+result[i]);
                continue;
            }
            result[i] = readLine2(position+i);

//            int delta = (int) (offset2 - offset - 1);
//            StringBuilder sb = new StringBuilder(delta);
//            for (int i1 = 0; i1 < delta; i1++) {
//                try {
//                    sb.append((char) reader.read());
//                } catch (IOException e) {
//                    // ignore
//                }
//            }
//            result[i] = sb.toString();
//            System.out.println("line found "+result[i]);
        }
        return result;
    }


    /**
     * check if reader is up to date, and if not perform an update.
     *
     * @throws IOException if any problem while updating
     */
    public void update() throws IOException {
        ensureOpen();
        if (getParent() != null) {
            // always update parent before, since this reader depends on it
            getParent().update();
        }
        offstReader.update(this);
        lastModified = file.lastModified();
        length = file.length();
    }

    /**
     * Matcher of line for this reader.
     *
     * @param line the line to be accepted by this line reader
     * @return <code>true</code> if the line must be used,<code>false</code>
     *         otherwise
     */
    public boolean match(String line) {
        // be default we accept everything
        return true;
    }

    /** @return the number of lines found in the file */
    public long getNbLines() {
        ensureOpen();
        return offstReader.getNbLines();
    }

    /** @return the file used to read lines */
    public File getFile() {
        return file;
    }

    /** @return the offset reader used to obtain offsets. */
    public OffsetReader getOffsetReader() {
        return offstReader;
    }

    /** @return the parent LineReader, or null if none */
    public LineReader getParent() {
        return parent;
    }

    /**
     * @return <code>true</code> if file is up to date,<code>false</code>
     *         otherwise
     */
    public boolean isUpToDate() {
        return file.length() == length && file.lastModified() == lastModified;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("(file:").append(file.getName()).append(" ").append(offstReader).append(')');
        return sb.toString();
    }

    @Override
    protected void finalize() throws Throwable {
        try {
            close();
        } finally {
            super.finalize();
        }
    }

    protected void ensureOpen() {
        if (!isOpen()) {
            throw new IllegalStateException("LineReader " + getClass() + " is not open, use method open before all.");
        }
    }

    public boolean isOpen() {
        return reader != null;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }
}
