/*
 * #%L
 * IsisFish
 * 
 * $Id: SSHSimulatorLauncher.java 4017 2014-06-16 16:04:46Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2008 - 2014 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n.t;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.MD5InputStream;
import org.nuiton.util.StringUtil;
import org.nuiton.util.ZipUtil;

import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.util.ssh.InvalidPassphraseException;
import fr.ifremer.isisfish.util.ssh.ProgressMonitor;
import fr.ifremer.isisfish.util.ssh.ProxyCommand;
import fr.ifremer.isisfish.util.ssh.SSHAgent;
import fr.ifremer.isisfish.util.ssh.SSHException;
import fr.ifremer.isisfish.util.ssh.SSHUserInfo;
import fr.ifremer.isisfish.util.ssh.SSHUtils;
import freemarker.cache.ClassTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

/**
 * Use a remote simulation server.
 * 
 * Upload zip simulation file on server and launch simulation on that file.
 * 
 * Isis-Fish must be installed on remote server.
 * 
 * Caparmor file layout ($i = plan/as increment) :
 * <ul>
 *  <li>$ISIS-TMP/simulation-$id-preparation.zip</li>
 *  <li>$ISIS-TMP/simulation-$id-prescript.bsh</li>
 *  <li>$ISIS-TMP/simulation-$id-script.seq</li>
 *  <li>$ISIS-TMP/simulation-$id-result.zip</li>
 *  <li>$ISIS-TMP/simulation-$id-result.zip.md5</li>
 *  <li>$ISIS-TMP/simulation-$id-output.txt</li>
 *  <li>$ISIS-TMP/simulation-$id-pbs.id</li>
 * </ul>
 * All $ISIS-TMP/simulation-$id-* files are deteled after result download.
 * 
 * Special case :
 * <ul>
 *  <li>standalone zip : $ISIS-TMP/simulation-$shortid-result.zip
 *  (uploaded at first simulation)</li>
 *  <li>standalone simulations : $ISIS-TMP/simulation-$shortid-script.seq
 *  (uploaded at last simulation)</li>
 * </ul>
 * where {@code $shortid} is id of parent job (without increment), they are not
 * deleted after result download.
 * 
 * @see JSch
 * 
 * @author chatellier
 * @version $Revision: 4017 $
 * 
 * Last update : $Date: 2014-06-16 18:04:46 +0200 (Mon, 16 Jun 2014) $
 * By : $Author: echatellier $
 */
public class SSHSimulatorLauncher implements SimulatorLauncher {

    /** Class logger */
    protected static Log log = LogFactory.getLog(SSHSimulatorLauncher.class);

    /** Freemarker configuration */
    protected Configuration freemarkerConfiguration;

    /** Freemarker qsub template. */
    protected static final String QSUB_SCRIPT_TEMPLATE = "templates/ssh/qsub-script.ftl";

    /**
     * Opened session to ssh service. Stored in static context to not reask passphrase at each
     * connection.
     */
    protected static Session sshSession;

    /**
     * Opened session to sftp service. Stored in static context to not reask passphrase at each
     * connection.
     */
    protected static Session sshSftpSession;

    /**
     * Constructor.
     * 
     * Init freemarker.
     */
    public SSHSimulatorLauncher() {
        initFreemarker();
    }

    /**
     * Init freemarker configuration.
     */
    protected void initFreemarker() {

        freemarkerConfiguration = new Configuration();

        // needed to overwrite "Defaults to default system encoding."
        // fix encoding issue on some systems
        freemarkerConfiguration.setDefaultEncoding("utf-8");

        // specific template loader to get template from jars (classpath)
        ClassTemplateLoader templateLoader = new ClassTemplateLoader(SSHSimulatorLauncher.class, "/");
        freemarkerConfiguration.setTemplateLoader(templateLoader);

    }

    /**
     * Display message both in commons-logging and control text progress.
     * 
     * @param control control
     * @param message message to display
     */
    protected void message(SimulationControl control, String message) {
        // log
        if (log.isInfoEnabled()) {
            log.info(message);
        }
        // control
        if (control != null) {
            control.setText(message);
        }
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#maxSimulationThread()
     */
    @Override
    public int maxSimulationThread() {

        // mais le serveur les lance quand il veut
        int maxSimulationThread = IsisFish.config.getSimulatorSshMaxThreads();
        
        if (maxSimulationThread <= 0) {
            // always set a minimun of 1
            maxSimulationThread = 1;
        }

        return maxSimulationThread;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#getCheckProgressionInterval()
     */
    @Override
    public int getCheckProgressionInterval() {

        // par defaut, pour ssh, on utilise 20 secondes

        int interval = IsisFish.config.getSimulatorSshControlCheckInterval();
        return interval;
    }

    /*
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return t("isisfish.simulator.launcher.remote");
    }

    /**
     * {@inheritDoc}
     * 
     * Try to send a qdel command.
     */
    @Override
    public void simulationStopRequest(SimulationJob job) throws RemoteException {

        // make sure user is connected
        try {
            getSSHSession();
        }
        catch(JSchException e) {
            throw new RemoteException("Can't connect", e);
        }
        
        // get simulation item info
        SimulationItem simulationItem = job.getItem();
        if (simulationItem.isStandaloneSimulation()) {
            try {
                sendStopSimulationRequest(sshSession, simulationItem.getControl().getId());
            } catch (SSHException e) {
                throw new RemoteException("Can't connect", e);
            }
        }
        else {
            // for multi job process, try to optimize process
            // by send stop request only fo last job
            
            // multi jobs has been started with last one
            // must be killed with last one too
            if (simulationItem.isLastSimulation()) {
                String simulationid = simulationItem.getControl().getId();
                String shortSimulationId = simulationid.substring(0, simulationid.lastIndexOf('_'));
                try {
                    sendStopSimulationRequest(sshSession, shortSimulationId);
                } catch (SSHException e) {
                    throw new RemoteException("Can't connect", e);
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * Dans le cas de ssh: 
     * <ul>
     *  <li>upload la simulation</li>
     *  <li>construit le script pour qsub</li>
     *  <li>upload le script qsub</li>
     *  <li>ajoute le script a qsub</li>
     * </ul>
     * 
     * Et : 
     * <ul>
     *  <li>lance le thread de control de la simulation</li>
     * </ul>
     */
    @Override
    public void simulate(SimulationService simulationService,
            SimulationItem simulationItem) throws RemoteException {

        // get simulation information for this launcher
        SimulationControl control = simulationItem.getControl();
        File simulationZip = simulationItem.getSimulationZip();
        String simulationPrescript = simulationItem.getGeneratedPrescriptContent();

        // check username
        if (StringUtils.isBlank(IsisFish.config.getSimulatorSshUsername())) {
            throw new RemoteException("Username is empty");
        }

        // start ssh session
        try {

            String simulationid = control.getId();

            // connection
            message(control, t("isisfish.simulation.remote.message.connection"));
            Session sshSession = getSSHSession();
            //Session sshSftpSession = getSSHSftpSession();

            // upload simulation on server
            message(control, t("isisfish.simulation.remote.message.upload"));
            String simulationRemotePath = uploadSimulationIfNecessary(sshSession, simulationItem, simulationid, simulationZip);

            String remoteResultZip = getRemoteResultArchivePath(simulationid);

            // build du contenu du script
            message(control,t("isisfish.simulation.remote.message.waitingstart"));
            String simulationPreScriptPath = uploadPreScriptIfNecessary(
                    sshSession, control.getId(), simulationPrescript);
            
            // start simulation if necessary (multi jobs) ...
            startSimulation(sshSession, simulationItem, simulationid, simulationRemotePath, remoteResultZip, simulationPreScriptPath);

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error(t("isisfish.error.simulation.remote.global"));
            }
            throw new RemoteException(
                    t("isisfish.error.simulation.remote.global"), e);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * Se connecte au serveur distant et télécharge les résultats de la
     * simulation.
     * 
     * Simulation must have been downloaded with
     * {@link #updateControl(SimulationService, SimulationControl)} before calling
     * this method.
     */
    @Override
    public SimulationStorage getSimulationStorage(
            SimulationService simulationService, SimulationControl control)
            throws RemoteException {

        // make sure that simulation has been downloaded by #updateControl()
        // before calling this method
        String simulationId = control.getId();

        SimulationStorage simulationStorage = SimulationStorage
                    .getSimulation(simulationId);

        return simulationStorage;
    }

    /**
     * {@inheritDoc}
     * 
     * Se connecte au serveur distant et télécharge le fichier de control.
     * Injecte ensuite ce fichier dans le {@link SimulationControl}.
     * 
     * Essaye aussi de telecharger le fichier md5 de la simulation, et, s'il
     * est present, l'archive de résultat.
     * Supprime tous les fichiers de la simulations apres avoir télécharger les
     * résultats.
     */
    @Override
    public void updateControl(SimulationService simulationService,
            SimulationControl control) throws RemoteException {

        // make sure user is connected
        try {
            getSSHSession();
        } catch(JSchException e) {
            throw new RemoteException("Can't connect", e);
        }

        try {

            // CONTROL file
            try {
                // download control file
                File controlFile = downloadSimulationFile(sshSession, control.getId(),
                        SimulationStorage.CONTROL_FILENAME);
                if (log.isDebugEnabled()) {
                    log.debug("Control have been downloaded : "
                            + controlFile.getAbsolutePath());
                }

                synchronized (control) {
                    // le thread principal a pu le modifier pendant le sleep
                    //if (control.isRunning()) {
                    // on ne lit pas le stop, car le stop ne peut-etre appeler
                    // que par l'utilisateur qui est de ce cote de la machine
                    SimulationStorage.readControl(controlFile, control, "stop");
                    //}
                }

                // deleteTempFile
                controlFile.delete();
            } catch (SSHException e) {

                // file doesn't exist
                if (log.isDebugEnabled()) {
                    // not add ,e plz :)
                    log.debug(t("Remote control file doesn't exists %s", e.getMessage()));
                }
            }

            // can be because, simulation has not begun
            // or simulation is ended
            // try do download md5 control file

            // MD5 + SIMULATION zip file
            // FIXME echatellier 20140402 check not yet existing md5ControlFile and simulation ended
            try {
                File md5ControlFile = downloadResultsMD5File(sshSession, control.getId());

                if (md5ControlFile != null) {
                    control.setText(t("isisfish.simulation.remote.message.downloadresults"));

                    String md5sum = FileUtils.readFileToString(md5ControlFile);

                    if (log.isDebugEnabled()) {
                        log.debug("MD5 Control file have been downloaded : " + md5ControlFile.getAbsolutePath());
                    }

                    File resultArchiveFile = downloadResultsArchive(sshSession, control, md5sum);

                    if (resultArchiveFile != null) {

                        ZipUtil.uncompressFiltred(resultArchiveFile, SimulationStorage.getSimulationDirectory());

                        if (log.isDebugEnabled()) {
                            log.debug("Simulation imported : " + resultArchiveFile.getAbsolutePath());
                        }

                        resultArchiveFile.delete();

                        // read control from downloaded simulation
                        synchronized (control) {
                            SimulationStorage.readControl(control.getId(), control, "stop");
                        }

                        // clear all simulation input/output files on remote
                        // server temp directory
                        // control need to be read before, otherwize, setText
                        // in clearSimulationFiles will erase file :(
                        clearSimulationFiles(sshSession, control);
                    } else {
                        if (log.isWarnEnabled()) {
                            log.warn("Simulation zip download failed");
                        }
                    }

                    // remove temp file
                    md5ControlFile.delete();
                }
            } catch(SSHException e) {
                if (log.isDebugEnabled()) {
                    log.debug(t("Can't download archive : %s", e.getMessage())); 
                }
            }

            // INFORMATION file
            try {
                File infoFile = downloadSimulationFile(sshSession, control.getId(),
                        SimulationStorage.INFORMATION_FILENAME);
                if (log.isDebugEnabled()) {
                    log.debug("Information have been downloaded : "
                            + infoFile.getAbsolutePath());
                }

                // s'il y a une exception,
                // on dit juste que la simulation a eu une demande
                // d'arret pour qu'elle s'arrete dans l'UI
                Properties infoProperties = new Properties();
                InputStream isInfoFile = new FileInputStream(infoFile);
                try {
                    infoProperties.load(isInfoFile);
                }
                finally {
                    isInfoFile.close();
                }
                if (!StringUtils.isEmpty(infoProperties
                        .getProperty("exception"))) {
                    synchronized (control) {
                        control.setStopSimulationRequest(true);
                    }
                }

                // deleteTempFile
                infoFile.delete();
            } catch (SSHException e) {
                // file doesn't exist
                if (log.isDebugEnabled()) {
                    // not add ,e plz :)
                    log.debug(t("Remote information file doesn't exists %s", e
                            .getMessage()));
                }
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                // not add ,e plz :)
                log.error(t("Can't download file"), e);
            }
        }
    }

    /**
     * Get opened ssh session or try to open a new one.
     * 
     * This method must synchronized.
     * 
     * @return opened ssh session.
     * @throws JSchException
     */
    protected synchronized Session getSSHSession() throws JSchException {

        if (sshSession == null || !sshSession.isConnected()) {
            sshSession = openSSHSession();
        }

        return sshSession;
    }

    /**
     * Connect to remote server throw SSH, and return session.
     * 
     * @return valid opened session
     * 
     * @throws JSchException
     */
    protected Session openSSHSession() throws JSchException {

        JSch jsch = new JSch();

        // extract connection infos
        String host = IsisFish.config.getSimulatorSshServer();
        String username = IsisFish.config.getSimulatorSshUsername();

        int port = 22; // by default, 22

        if (host.indexOf(':') > 0) {
            String sPort = host.substring(host.indexOf(':') + 1);
            try {
                port = Integer.parseInt(sPort);
            } catch (NumberFormatException e) {
                if (log.isWarnEnabled()) {
                    log.warn(t("isisfish.error.simulation.remote.wrongportvalue",
                            sPort));
                }
            }
            host = host.substring(0, host.indexOf(':'));
        }

        if (log.isInfoEnabled()) {
            log.info(t("Try to log on %s@%s:%d", username, host, port));
        }

        // add ssh key
        boolean sshKeyUsed = false;
        File sshKey = IsisFish.config.getSSHPrivateKeyFilePath();
        if (sshKey.canRead()) {
            if (log.isInfoEnabled()) {
                log.info(t("Ssh key found '%s' will be used to connect to",
                        sshKey.getAbsoluteFile(), host));
            }
            jsch.addIdentity(sshKey.getAbsolutePath());
            sshKeyUsed = true;
        }
        else {
            if (log.isInfoEnabled()) {
                log.info(t("Can't read ssh key : %s", sshKey));
            }
        }

        Session session = jsch.getSession(username, host, port);

        // add proxy command if specified
        String proxyCommand = IsisFish.config.getSimulatorSshProxyCommand();
        if (StringUtils.isNotBlank(proxyCommand)) {
            session.setProxy(new ProxyCommand(proxyCommand));
        }

        // username and password will be given via UserInfo interface.
        SSHUserInfo ui = new SSHUserInfo();
        if (sshKeyUsed) {
            String passphrase = null;
            try {
                passphrase = SSHAgent.getAgent().getPassphrase(sshKey);
                ui.setPassphrase(passphrase);
            } catch (InvalidPassphraseException e) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't key passphrase for key", e);
                }
            }
        }
        session.setUserInfo(ui);
        session.connect(20000); // timeout

        // test here, if password has been asked to user
        if (session.isConnected() && sshKeyUsed && ui.getPassword() != null) {
            putSshKeyOnRemoteServer(session, sshKey);
        }
        return session;
    }

    /**
     * Close ssh session.
     * 
     * @param session session to close
     */
    protected void closeSSHSession(Session session) {
        if (session != null) {
            session.disconnect();
        }
    }

    /**
     * Add ssh key into $HOME/.ssh/authorized_keys file.
     * 
     * Just connect and do an "echo xx &gt;&gt; .ssh/authorized_keys"
     * 
     * @param session opened session
     * @param sshKey
     * @throws JSchException
     */
    protected void putSshKeyOnRemoteServer(Session session, File sshKey)
            throws JSchException {

        // get public key for argument private key file
        File publicKey = new File(sshKey.getAbsoluteFile() + ".pub");

        // command to :
        // - make ssh directory
        // - add key to authorized_keys

        // tested on bash/csh
        String command = "test -d .ssh||mkdir .ssh;echo \"%s\" >> .ssh/authorized_keys";

        try {
            // use usefull readLines from commons-io
            List<String> contents = FileUtils.readLines(publicKey);

            // only one line
            if (contents != null && contents.size() == 1) {
                command = String.format(command, contents.get(0));

                if (log.isInfoEnabled()) {
                    log.info("Add key on remote authorized keys");
                }
                if (log.isDebugEnabled()) {
                    log.debug("command is : " + command);
                }

                SSHUtils.exec(session, command);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error(t("Error while uploading public key to remote serveur authorized_keys"),
                                e);
            }
        } catch (SSHException e) {
            if (log.isErrorEnabled()) {
                log.error(t("Error while uploading public key to remote serveur authorized_keys"),
                                e);
            }
        }
    }

    /**
     * Upload simulation if necessary and always return the remote
     * simulation zip path to use.
     * 
     * @param session already open valid ssh session
     * @param simulationItem simulation item
     * @param simulationid simulation id
     * @param simulationFile simulation file to upload
     * 
     * @return remote file path or <tt>null</tt> if errors
     * @throws SSHException if upload fail
     */
    protected String uploadSimulationIfNecessary(Session session, SimulationItem simulationItem, String simulationid, File simulationFile)
            throws SSHException {

        // first check that remote directory exists
        String remoteTemp = getRemoteTempDirectory();
        String remotePath = null;

        if (!simulationItem.isStandaloneSimulationZip()) {
            // get simulation file path for each simulation...
            String shortSimulationId = simulationid.substring(0, simulationid.lastIndexOf('_'));
            String shortSimulationZip = "simulation-" + shortSimulationId + "-preparation.zip";
            remotePath = remoteTemp + shortSimulationZip;

            // ...but perform real upload only for frist one !
            if (simulationItem.getSimulationNumber() == 0) {
                uploadSimulation(session, remoteTemp, remotePath, simulationFile);
            }
        }
        else {
            // not standalone, name always different
            String simulationZip = "simulation-" + simulationid + "-preparation.zip";
            remotePath = remoteTemp + simulationZip;
            // perform upload ech time
            uploadSimulation(session, remoteTemp, remotePath, simulationFile);
        }

        return remotePath;
    }

    /**
     * Perform simulation upload.
     * 
     * Create remote temp directory if not exists.
     * 
     * @param session already open valid ssh session
     * @param remoteDirectory
     * @param remoteSimulationZipPath
     * 
     * @throws SSHException 
     */
    protected void uploadSimulation(Session session, String remoteDirectory, String remoteSimulationZipPath, File simulationFile) throws SSHException {

        // following command work on bash and csh
        String command = "test -d \"" + remoteDirectory + "\"||mkdir -p \"" + remoteDirectory + "\"";

        if (log.isInfoEnabled()) {
            log.info("Creating remote temp directory (if not exists) " + remoteDirectory);
            if (log.isDebugEnabled()) {
                log.debug("Executing command : " + command);
            }
        }
        int exit = SSHUtils.exec(session, command);

        if (exit != 0) {
            throw new SSHException(t("Command '%s' fail to execute", command));
        }

        SSHUtils.scpTo(session, simulationFile, remoteSimulationZipPath);
    }

    /**
     * Download simulation zip results.
     * 
     * MD5 control check sum if done, return null, if checkSum fail.
     * 
     * File if configured to auto delete at JVM shutdown.
     * 
     * @throws SSHException if download fail (can happen if remote file doesn't exist
     * @throws IOException if download fail (can happen if remote file doesn't exist
     */
    protected File downloadResultsArchive(Session session, SimulationControl simulationControl, String md5sum)
            throws SSHException, IOException {

        File localFile = File.createTempFile("simulation-results", ".zip");
        localFile.deleteOnExit();

        if (log.isDebugEnabled()) {
            log.debug("Downloading results in " + localFile.getAbsolutePath());
        }

        // build remote file path
        // FIXME this path should be given by remote IsisFish app
        String simulationId = simulationControl.getId();
        String remoteFile = getRemoteResultArchivePath(simulationId);

        try {
            ProgressMonitor progress = new ControlProgressMonitor(simulationControl);
            SSHUtils.scpFrom(session, remoteFile, localFile, progress);
        }
        catch(SSHException e) {
            localFile.delete();
            throw e;
        }

        if (!StringUtils.isEmpty(md5sum)) {
            String localMd5 = StringUtil.asHex(MD5InputStream.hash(new BufferedInputStream(new FileInputStream(localFile))));
            if (!localMd5.equals(md5sum)) {
                if (log.isWarnEnabled()) {
                    log.warn("Warning md5 checksum failed (got " + localMd5 + ", expected : " + md5sum + ")");
                }
                localFile.delete();
                localFile = null;
            }
        }

        return localFile;
    }

    /**
     * Redefine a custom progress monitor that update control.
     */
    protected static class ControlProgressMonitor extends ProgressMonitor {

        /** Control to update. */
        protected SimulationControl control;

        /**
         * Constructor with control.
         * 
         * @param control control
         */
        public ControlProgressMonitor(SimulationControl control) {
            this.control = control;
        }

        /*
         * @see fr.ifremer.isisfish.util.ssh.ProgressMonitor#init(long)
         */
        @Override
        public void init(long max) {
            super.init(max);
            control.setProgressMax(initFileSize);
        }

        /*
         * @see fr.ifremer.isisfish.util.ssh.ProgressMonitor#count(long)
         */
        @Override
        public void count(long len) {
            super.count(len);
            control.setProgress(totalLength);
        }

        /*
         * @see fr.ifremer.isisfish.util.ssh.ProgressMonitor#end()
         */
        @Override
        public void end() {
            super.end();
            control.setProgress(initFileSize);
        }
    }

    /**
     * Download remote simulation control file and store its content into temp
     * file.
     * 
     * @param sshSession valid opened ssh session
     * @param simulationId id de la simulation
     * @param fileName nom du fichier a telecharger
     * @return downloaded temp file (file have to be manually deleted)
     * @throws IOException
     * @throws SSHException if remote file doesn't exists
     */
    protected File downloadSimulationFile(Session sshSession, String simulationId, String fileName)
            throws IOException, SSHException {

        File localFile = null;

        // build remote file path
        // FIXME this path should be given by remote IsisFish app
        String remoteFile = IsisFish.config.getSimulatorSshDataPath();
        remoteFile += "/" + SimulationStorage.SIMULATION_PATH;
        remoteFile += "/" + simulationId;
        remoteFile += "/" + fileName;

        // local tmp file
        localFile = File.createTempFile(simulationId, fileName);

        try {
            SSHUtils.scpFrom(sshSession, remoteFile, localFile);
        } catch (SSHException e) {
            localFile.delete();
            throw e;
        }

        return localFile;
    }

    /**
     * Download remote simulation md5 control file and store its content into temp
     * file.
     * 
     * @param sshSession valid opened ssh session
     * @param simulationId id de la simulation
     * @return downloaded temp file (file have to be manually deleted)
     * @throws IOException
     * @throws SSHException if remote file doesn't exists
     */
    protected File downloadResultsMD5File(Session sshSession, String simulationId)
            throws IOException, SSHException {

        // build remote file path
        // this path is not configurable
        // same as fr.ifremer.isisfish.actions.SimulationAction.simulateRemotellyWithPreScript(String, File, File, File)
        // defined by org.nuiton.util.ZipUtil.compressFiles(File, File, Collection<File>, boolean)
        String remoteFile = getRemoteResultArchivePath(simulationId) + ".md5";

        // local tmp file
        File localFile = File.createTempFile(simulationId, ".md5");

        try {
            SSHUtils.scpFrom(sshSession, remoteFile, localFile);
        } catch (SSHException e) {
            localFile.delete();
            throw e;
        }

        return localFile;
    }

    /**
     * Remove all {@code $ISIS-TMP/simulation-$id-*} files on caparmor.
     * 
     * @param session valid opened ssh session
     * @param control simulation control
     */
    protected void clearSimulationFiles(Session session,
            SimulationControl control) throws IOException, SSHException {

        control.setText(t("isisfish.simulation.remote.message.deletingfiles"));

        // execute rm -f "isis-tmp/simulation-$id-"*
        // on remote. Note * outside quotes !!!
        String simulationId = control.getId();
        String command = "rm -f \"" + getRemoteTempDirectory() + "simulation-" + simulationId + "-\"*";

        if (log.isDebugEnabled()) {
            log.debug("Deleting simulation files with command : " + command);
        }

        SSHUtils.exec(session, command);

        // can return other things than 0
        // but not a big deal
    }

    /**
     * Upload script on remote server.
     * 
     * @param session valid opened ssh session
     * @param simulationScript file to upload
     * 
     * @throws SSHException if upload fail
     */
    protected String uploadSimulationScript(Session session,
            String simulationid, File simulationScript) throws SSHException {

        // remote temp directory should have been created
        // by #uploadSimulation(Session, String)
        String remotePath = getRemoteTempDirectory();

        // always rename uploaded script to "simulation-$id-script.seq"
        remotePath += "simulation-" + simulationid + "-script.seq";

        SSHUtils.scpTo(session, simulationScript, remotePath);

        return remotePath;
    }

    /**
     * Get remote simulation zip path.
     * 
     * @param simulationId simulation id
     */
    protected String getRemoteResultArchivePath(String simulationId) {
        String remotePath = getRemoteTempDirectory();
        remotePath += "simulation-" + simulationId + "-result.zip";
        return remotePath;
    }

    /**
     * Upload pre script on remote server.
     * 
     * Return path if uploaded or null if no upload needed.
     * 
     * @param session valid opened ssh session
     * @param simulationId simulation id
     * @param simulationPreScript script content
     * 
     * @throws SSHException if upload fail
     * @throws IOException if upload fail
     */
    protected String uploadPreScriptIfNecessary(Session session,
            String simulationId, String simulationPreScript)
            throws SSHException, IOException {

        // if there is no pre script, do nothings
        if (StringUtils.isEmpty(simulationPreScript)) {
            return null;
        }

        File tempPreScriptFile = File.createTempFile("simulation-" + simulationId
                + "-prescript", ".bsh");
        tempPreScriptFile.deleteOnExit();
        FileUtils.writeStringToFile(tempPreScriptFile, simulationPreScript, "utf-8");

        // remote temp directory should have been created
        // by #uploadSimulation(Session, String)
        String remotePath = getRemoteTempDirectory();

        // always rename simulation prescript to "simulation-$id-prescript.bsh"
        remotePath += "simulation-" + simulationId + "-prescript.bsh";

        SSHUtils.scpTo(session, tempPreScriptFile, remotePath);

        return remotePath;
    }

    /**
     * Start simulation if necessary.
     * 
     * Current simulation can be started later with a PBS multi job.
     * 
     * @param session ssh session
     * @param simulationItem simulation item (needed for additionnal info, simulation, number, indenpendant, etc...)
     * @param simulationid simulation id
     * @param simulationRemoteZipPath simulation preparation (input) zip path
     * @param remoteResultZip simulation result (output) zip path
     * @param simulationPreScriptPath simulation prescript
     * 
     * @throws Exception 
     */
    protected void startSimulation(Session session, SimulationItem simulationItem, String simulationid, String simulationRemoteZipPath, String remoteResultZip, String simulationPreScriptPath) throws Exception {

        // standalone simulation
        // no question, generate script, launch it each time
        if (simulationItem.isStandaloneSimulation()) {
            // single simulation
            File simulationPSBScript = getLaunchSimulationScriptFile(simulationid,
                    simulationRemoteZipPath, true, remoteResultZip, simulationPreScriptPath, false);
            String scriptRemotePath = uploadSimulationScript(session, simulationid, simulationPSBScript);

            // prescript uploaded, delete
            simulationPSBScript.delete();

            sendStartSimulationRequest(session, simulationid, scriptRemotePath, -1);
        }
        else {
            // standalone, on do it for last simulation
            if (simulationItem.isLastSimulation()) {
                String shortSimulationId = simulationid.substring(0, simulationid.lastIndexOf('_'));

                if (log.isDebugEnabled()) {
                    log.debug("Last simulation start requested, send multijob start request for " + shortSimulationId);
                }

                // multiples jobs simulation
                File simulationPSBScript = getLaunchSimulationScriptFile(shortSimulationId,
                        simulationRemoteZipPath, simulationItem.isStandaloneSimulationZip(), remoteResultZip, simulationPreScriptPath, true);
                String scriptRemotePath = uploadSimulationScript(session, shortSimulationId, simulationPSBScript);

                // prescript uploaded, delete
                simulationPSBScript.delete();

                // file will be named with shortSimulationId (instead of simulationId)
                sendStartSimulationRequest(session, shortSimulationId, scriptRemotePath, simulationItem.getSimulationNumber());
            }
            else {
                if (log.isDebugEnabled()) {
                    log.debug("Current simulation is not last simulation in pool, skip start");
                }
            }
        }
    }

    /**
     * Retourne un fichier temporaire contenant le script de lancement de
     * simulation.
     * 
     * Le fichier temporaire est configuré pour se supprimer tout seul.
     * 
     * @param simuationId id de la simulation
     * @param simulationZip zip de la simulation
     * @param standaloneZip standalone simulation zip
     * @param preScriptPath simulation pre script path (can be null)
     * @param multipleSimulationScript if {@code true} build a multijob simulation script
     * 
     * @return un Fichier temporaire ou {@code null} en cas d'exception
     * 
     * @throws IOException if can't build script
     */
    protected File getLaunchSimulationScriptFile(String simuationId,
            String simulationZip, boolean standaloneZip, String simulationResultZip, String preScriptPath, boolean multipleSimulationScript) throws IOException {

        File tempScript = File.createTempFile("simulation-" + simuationId + "-script", ".seq");
        tempScript.deleteOnExit(); // auto delete

        String fileContent = getSimulationScriptLaunchContent(
                QSUB_SCRIPT_TEMPLATE, simuationId, simulationZip, standaloneZip, simulationResultZip, preScriptPath, multipleSimulationScript);
        FileUtils.writeStringToFile(tempScript, fileContent, "utf-8");

        return tempScript;
    }

    /**
     * Utilise freemarker pour recuperer le contenu du script.
     * 
     * Remplace aussi la variable $simulation du template.
     * 
     * @param templateName url du template
     * @param simuationId id de la simulation
     * @param simulationZip zip de la simulation
     * @param standaloneZip standalone simulation zip
     * @param simulationZipResult zip resultat de la simulation
     * @param preScriptPath simulation pre script path (can be null)
     * @param multipleSimulationScript if {@code true} build a multijob simulation script
     * 
     * @throws IOException if can't get script content
     */
    protected String getSimulationScriptLaunchContent(String templateName,
            String simuationId, String simulationZip, boolean standaloneZip,
            String simulationZipResult, String preScriptPath, boolean multipleSimulationScript)
            throws IOException {

        String scriptContent = null;

        // test null values for prescript
        String remotePreScript = preScriptPath;
        if (remotePreScript == null) {
            remotePreScript = "";
        }

        try {
            // get template
            Template template = freemarkerConfiguration
                    .getTemplate(templateName);

            // context values
            Map<String, Object> root = new HashMap<String, Object>();
            root.put("isishome", IsisFish.config.getSimulatorSshIsisHome());
            root.put("isistemp", getRemoteTempDirectory());
            root.put("javapath", IsisFish.config.getSimulatorSshJavaPath());
            root.put("javamemory", IsisFish.config.getSimulatorSshMaxMemory());
            root.put("simulationid", simuationId);
            root.put("simulationzip", simulationZip);
            root.put("simulationstandalonezip", standaloneZip);
            root.put("simulationresultzip", simulationZipResult);
            root.put("simulationprescript", remotePreScript);
            root.put("qsubmutiplejob", multipleSimulationScript);

            // process template
            Writer out = new StringWriter();
            template.process(root, out);
            out.flush();
            scriptContent = out.toString();

        } catch (TemplateException e) {
            if (log.isErrorEnabled()) {
                log.error(t("Process template error"), e);
            }

            throw new IOException(t("Process template error"), e);
        }

        return scriptContent;
    }

    /**
     * Add script in remote qsub queue.
     * 
     * @param session valid opened session
     * @param simulationId simulation id (short version for a multiple job)
     * @param scriptRemotePath remote script path
     * @param lastSimulationNumber if {@code >=0} start a multiple pbs job form 0 to {@code lastSimulationNumber}
     * 
     * @throws SSHException if call fail
     */
    protected void sendStartSimulationRequest(Session session, String simulationId, String scriptRemotePath, int lastSimulationNumber)
            throws SSHException {

        // command to :
        // - add script in qsub queue
        String remoteFilenameId = getRemoteTempDirectory() + "simulation-" + simulationId + "-pbs.id";
        String command = IsisFish.config.getSimulatorSshPbsBinPath() + "/qsub";

        // add qsub options 
        String qsubOptions = IsisFish.config.getSimulatorSshPbsQsubOptions();
        if (StringUtils.isNotEmpty(qsubOptions)) {
            command += " " + qsubOptions;
        }

        // multi job specific
        if (lastSimulationNumber >=0 ) {
            command+= " -J 0-" + String.valueOf(lastSimulationNumber);
        }

        // end with squb script path and redirect id into file (to stop it later)
        command += " \"" + scriptRemotePath + "\"|tee \"" + remoteFilenameId + "\"";

        if (log.isDebugEnabled()) {
            log.debug("Send qsub job starting command : " + command);
        }

        Writer output = new StringWriter();
        int exit = SSHUtils.exec(session, command, output);

        if (exit != 0) {
            throw new SSHException(t("Command '%s' fail to execute", command));
        }

        String out = output.toString();
        // multiple jobs id are like : 78600[].service4 ou 6199230.service0
        if (out.trim().matches("\\d+(\\[\\])?\\.\\w+") && log.isInfoEnabled()) {
            log.info("Job submitted with job id : " + out);
        }
    }

    /**
     * Send qdel request on job.
     * 
     * @param session valid opened session
     * @param simulationId simulation id
     * 
     * @throws SSHException if call fail
     */
    protected void sendStopSimulationRequest(Session session, String simulationId)
            throws SSHException {

        // command to :
        String remoteFilenameId = getRemoteTempDirectory() + "simulation-" + simulationId + "-pbs.id";
        String command = IsisFish.config.getSimulatorSshPbsBinPath() + "/qdel `cat \"" + remoteFilenameId + "\"`";

        // and delete simulation
        //command += " && rm -rf \"" + IsisFish.config.getSimulatorSshDataPath() + "/simulations/" + simulationId + "\"";

        if (log.isDebugEnabled()) {
            log.debug("Send stop request : " + command);
        }

        SSHUtils.exec(session, command);

        // can fail, already stopped
        /*if (exit != 0) {
            throw new SSHException(t("Command '%s' fail to execute", command));
        }*/
    }

    /**
     * Get remote directory absolute path.
     * 
     * Don't use {@link java.io.File#separator} here, caparmor is always unix.
     * 
     * @return remote temp directory path
     */
    protected String getRemoteTempDirectory() {
        String remotePath = IsisFish.config.getSimulatorSshTmpPath();

        if (!remotePath.startsWith("/")) {
            remotePath = IsisFish.config.getSimulatorSshUserHome() + "/" + remotePath;
        }

        // upload directory in that dir
        if (!remotePath.endsWith("/")) {
            remotePath += "/";
        }

        return remotePath;
    }
}
