/*
 * #%L
 * IsisFish
 * 
 * $Id: OptimizationContext.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: https://svn.codelutin.com/isis-fish/tags/isis-fish-4.3.1.1/src/main/java/fr/ifremer/isisfish/simulator/OptimizationContext.java $
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package fr.ifremer.isisfish.simulator;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.simulator.launcher.SimulationJob;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorHelper;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.commons.collections4.CollectionUtils;

/**
 * Context utilisé pour gérer les différentes simulation lancées par un script d'optimisation.
 * 
 * @author Eric Chatellier
 * @since 4.3.0.0
 */
public class OptimizationContext extends SimulationPlanContext {

    protected SimulationJob job;

    protected List<List<SimulationStorage>> generations =
            new ArrayList<List<SimulationStorage>>();

    /** Simulation to do during next generation simulation */
    protected List<SimulationStorage> nextSimulations;

    public OptimizationContext(String id, SimulationParameter param, SimulationJob job) {
        super(id, param);
        this.job = job;
    }

    public List<SimulationStorage> getNextSimulations() {
        if (nextSimulations == null) {
            nextSimulations = new ArrayList<SimulationStorage>();
        }
        return nextSimulations;
    }

    /**
     * Get simulation to do, put it generations and increment currentGeneration
     * This method must be call be Simulator when it want to get next simulations
     * to do.
     * @return null or empty collection if no more simulation to do
     */
    public List<SimulationStorage> clearNextSimulation() {
        List<SimulationStorage> result = nextSimulations;

        if (CollectionUtils.isNotEmpty(result)) {
            generations.add(result);
            nextSimulations = null;
        }

        return result;
    }

    public List<SimulationStorage> getLastSimulations() {
        List<SimulationStorage> result = getGeneration(generations.size() - 1);
        return result;
    }

    /**
     * Return new simulation. This new simulation is automaticaly added to
     * next simulation.
     * 
     * @return new Simulation
     */
    public SimulationStorage newSimulation() {
        try {
            String simId = id + "_" + number;
            SimulationParameter childParam = param.copy();
            childParam.setSimulationPlanNumber(number);
            SimulationStorage result = SimulationStorage.importAndRenameZip(job.getItem().getSimulationZip(), simId);
            
            incNumber();
            addSimulation(result);

            return result;
        } catch (Exception eee) {
            throw new IsisFishRuntimeException("can't create new simulation", eee);
        }
    }

    protected void addSimulation(SimulationStorage s) {
        getNextSimulations().add(s);
    }

    /**
     * Create simulation. This new simulation is automatically added to
     * next simulations. Database will be modified with factors in parameters
     */
    public void newSimulation(Factor ... factors) {
        SimulationStorage s = newSimulation();

        Collection<Factor> colFactors = new ArrayList<Factor>();
        Collections.addAll(colFactors, factors);
        String script = FactorHelper.generatePreScript(colFactors);

        SimulationParameter childParam = s.getParameter();
        childParam.setGeneratedPreScript(script);
    }

    /**
     * Return the current generation.
     * 0 for no generation
     * 1 for one generation simulation
     * ...
     * @return
     */
    public int getCurrentGeneration() {
        return generations.size();
    }

    public int getSimulationNumber() {
        return number;
    }
    
    public List<SimulationStorage> getGeneration(int n) {
        List<SimulationStorage> result = null;
        if (n >= 0 && CollectionUtils.isNotEmpty(generations)) {
            result = generations.get(n);
        }

        return result;
    }

}
