/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer, CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.sensitivity.domain;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.simulator.sensitivity.Domain;
import fr.ifremer.isisfish.simulator.sensitivity.visitor.DomainVisitor;

/**
 *
 * All the continuous domains can be of 2 different types (min/max or
 * percentage).
 *
 * A min/max domain can take 2 values : {@link #minBound} and {@link #maxBound}.
 * A percentage domain got 2 values {@link #coefficient} and
 * {@link #referenceValue} that are the basis to calculate the {@link #minBound}
 * and {@link #maxBound} values.
 *
 * A continuous domain also got a cardinality, either filled in by the user or
 * the sensitivity method if it manages it.
 *
 * To do sensitivity calculations, you MUST use {@link #getCalculatorMinBound()},
 * {@link #getCalculatorMaxBound()}.
 * 
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 24 févr. 2009 $
 * By : $Author: chatellier $
 */
public class ContinuousDomain implements Domain {

    /** serialVersionUID. */
    private static final long serialVersionUID = -2037768174807839046L;

    /** Min value. */
    protected Object minBound;

    /** Max value. */
    protected Object maxBound;

    /** Value used to create the factor. */
    protected Object referenceValue;

    /** Coefficient in percent */
    protected Double coefficient;

    /** Flag to know the type of the domain **/
    protected boolean percentageType;

    /**
     * Min/max type constructor.
     */
    public ContinuousDomain() {
        this(false);
    }

    /**
     * Constructor with bounds.
     * 
     * @param minBound min bound
     * @param maxBound max bound
     */
    public ContinuousDomain(Object minBound, Object maxBound) {
        this();
        this.minBound = minBound;
        this.maxBound = maxBound;
    }

    /**
     * Constructor with type.
     * 
     * @param percentageType if {@code true} percentage type
     */
    public ContinuousDomain(boolean percentageType) {
        this.percentageType = percentageType;
    }

    /**
     * Return factor type (min/max or percentage).
     * 
     * @return {@code true} if percentage type
     */
    public boolean isPercentageType() {
        return percentageType;
    }

    /**
     * Set domain type (min/max or percentage).
     * 
     * @param percentageType new type value
     */
    public void setPercentageType(boolean percentageType) {
        this.percentageType = percentageType;
    }
    
    /**
     * Return the minimum bound of the domain. Do not use this value for
     * sensitivity calculation, prefer using {@link #getCalculatorMinBound}.
     * 
     * @return the minimum bound
     */
    public Object getMinBound() {
        Object result;
        if (isPercentageType()) {
            result = ((Double)referenceValue + (((Double)referenceValue * coefficient) * ((2 * getCalculatorMinBound() - 1))));
        } else {
            result = minBound;
        }
        return result;
    }
    
    /**
     * Return the minimum bound for sensitivity calculators. It always returns 0.
     * 
     * @return the minimum bound for sensitivity analysis.
     */
    public double getCalculatorMinBound() {
        return 0;
    }

    /**
     * Set the minimum bound of the domain.
     * 
     * @param minBound the minimum bound.
     */
    public void setMinBound(Object minBound) {
        this.minBound = minBound;
    }

    /**
     * Return the maximum bound of the domain. Do not use this value for
     * sensitivity calculation, prefer using {@link #getCalculatorMaxBound}.
     * 
     * @return the maximum bound
     */
    public Object getMaxBound() {
        Object result;
        if (isPercentageType()) {
            result = ((Double) referenceValue +
                    (((Double) referenceValue * coefficient)
                            * ((2 * getCalculatorMaxBound() - 1))));
        } else {
            result = maxBound;
        }
        return result;
    }
    
    /**
     * Return the minimum bound for sensitivity calculators. It always returns 1.
     * 
     * @return the maximum bound for sensitivity analysis.
     */
    public double getCalculatorMaxBound() {
        return 1;
    }

    /**
     * Set max value.
     * 
     * @param maxBound the maxBound to set
     */
    public void setMaxBound(Object maxBound) {
        this.maxBound = maxBound;
    }

    
    /**
     * Get coefficient.
     * 
     * @return the coefficient
     */
    public Double getCoefficient() {
        return coefficient;
    }

    /**
     * Set coefficient.
     * 
     * @param coefficient the coefficient to set
     */
    public void setCoefficient(Double coefficient) {
        this.coefficient = coefficient;
    }

    /**
     * Get reference value.
     * 
     * @return the reference value
     */
    public Object getReferenceValue() {
        return referenceValue;
    }

    /**
     * Set reference value.
     * 
     * @param referenceValue the reference value.to set
     */
    public void setReferenceValue(Object referenceValue) {
        this.referenceValue = referenceValue;
    }

    /**
     * {@inheritDoc}.
     */
    public Object getValueForIdentifier(Object identifier) {

        Double result;

        if (isPercentageType()) {
            // since 3.4.0.0, operator is always *
            result = ((Double)referenceValue + (((Double)referenceValue * coefficient)
                * ((2 * (Double)identifier - 1))));
        } else {
            result = ((Double)identifier * ((Double)maxBound - (Double)minBound))
                    + (Double)minBound;
        }
        return result;
    }

    /**
     * Accept a new visitor.
     * 
     * @param visitor the visitor
     */
    public void accept(DomainVisitor visitor) {
        visitor.start(this);
        visitor.end(this);
    }
    
    @Override
    public ContinuousDomain clone() {
        ContinuousDomain cloned;
        try {
            cloned = (ContinuousDomain)super.clone();
        } catch (CloneNotSupportedException e) {
            throw new IsisFishRuntimeException("Can't clone domain", e);
        }
        return cloned;
    }
}
