/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.image.awt;

import java.awt.Graphics2D;
import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.IndexColorModel;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Locale;
import java.util.NoSuchElementException;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.plugins.jpeg.JPEGImageWriteParam;
import javax.imageio.stream.ImageOutputStream;

import de.intarsys.pdf.cos.COSArray;
import de.intarsys.pdf.cos.COSInteger;
import de.intarsys.pdf.cos.COSString;
import de.intarsys.pdf.filter.Filter;
import de.intarsys.pdf.pd.PDColorSpace;
import de.intarsys.pdf.pd.PDImage;

public class ImageConverterAwt2Pdf {

	public static final String IMAGEWRITER_JPEG = "JPEG"; //$NON-NLS-1$

	private final BufferedImage bufferedImage;

	private PDImage pdImage;

	private boolean preferJpeg = false;

	private float compressionQuality = 0.75f;

	public ImageConverterAwt2Pdf(BufferedImage bufferedImage) {
		this.bufferedImage = bufferedImage;
	}

	/**
	 * Use the given image as a template for this resource. The image
	 * information is copied to the appropriate dictionary keys and its raster
	 * bytes will be used for the stream's contents. If the preferJpeg parameter
	 * is true and JPEG encoding is possible with the image's color model and
	 * depth the stream will be compressed with a DCT filter otherwise a Flate
	 * filter will be used.
	 * 
	 * @param img
	 *            the image object to encapsulate in this resource
	 * @param preferJpeg
	 *            true if JPEG encoding of the image should be attempted
	 * 
	 */
	protected PDImage createPDImage() {
		pdImage = (PDImage) PDImage.META.createNew();
		renderImage();
		if (getPDImage().cosGetStream().getFirstFilter() == null) {
			try {
				getPDImage().cosGetStream().addFilter(
						Filter.CN_Filter_FlateDecode);
			} catch (Exception e) {
				// ignore
			}
		}
		return pdImage;
	}

	public float getCompressionQuality() {
		return compressionQuality;
	}

	public PDImage getPDImage() {
		if (pdImage == null) {
			pdImage = createPDImage();
		}
		return pdImage;
	}

	public boolean isPreferJpeg() {
		return preferJpeg;
	}

	/**
	 * render the img as a gray level raster.
	 * 
	 * @param image
	 *            the image content
	 * 
	 * @throws UnsupportedOperationException
	 *             If the image could not be rended, e.g. not support image type
	 */
	protected void renderGray() {
		int pixelSize;
		Raster raster;
		byte[] dataElements;
		byte[] bytes;

		getPDImage().cosSetColorSpace(PDColorSpace.CN_CS_DeviceGray);

		pixelSize = bufferedImage.getColorModel().getPixelSize();
		getPDImage().setBitsPerComponent(
				pixelSize / bufferedImage.getColorModel().getNumComponents());

		if ((pixelSize / bufferedImage.getColorModel().getNumComponents()) != 8) {
			// TODO 2 @ehk implement other component sizes
			throw new UnsupportedOperationException();
		}

		raster = bufferedImage.getData();
		getPDImage().setWidth(raster.getWidth());
		getPDImage().setHeight(raster.getHeight());

		if (!bufferedImage.getColorModel().hasAlpha() && preferJpeg
				&& renderJpegEncoded()) {
			return;
		}

		try {
			dataElements = (byte[]) raster.getDataElements(raster.getMinX(),
					raster.getMinY(), raster.getWidth(), raster.getHeight(),
					null);
		} catch (ClassCastException ex) {
			// TODO 2 @ehk implement int[] data elements
			throw new UnsupportedOperationException();
		}

		if (bufferedImage.getColorModel().hasAlpha()) {
			PDImage mask;
			byte[] maskBytes;

			bytes = new byte[dataElements.length / 2];
			maskBytes = new byte[dataElements.length / 2];

			mask = (PDImage) PDImage.META.createNew();
			mask.cosSetColorSpace(PDColorSpace.CN_CS_DeviceGray);
			mask.setBitsPerComponent(8);
			mask.setWidth(raster.getWidth());
			mask.setHeight(raster.getHeight());
			for (int i = 0; i < bytes.length; i++) {
				bytes[i] = dataElements[i * 2];
				maskBytes[i] = dataElements[(i * 2) + 1];
			}
			mask.setBytes(maskBytes);
			getPDImage().setSMask(mask);
		} else {
			bytes = new byte[dataElements.length];
			System.arraycopy(dataElements, 0, bytes, 0, dataElements.length);
		}

		getPDImage().setBytes(bytes);
	}

	/**
	 * render the image. the image's color model is not yet known
	 * 
	 * @param image
	 *            the image content
	 * 
	 * @throws UnsupportedOperationException
	 *             If the image could not be rended, e.g. not support image type
	 */
	protected void renderImage() {
		ColorModel colorModel;
		BufferedImage newImage;
		Graphics2D graphics;
		WritableRaster raster;

		colorModel = bufferedImage.getColorModel();
		if (colorModel instanceof IndexColorModel) {
			renderIndexed();
			return;
		}
		if (colorModel instanceof ComponentColorModel
				&& colorModel.getColorSpace().equals(
						ColorSpace.getInstance(ColorSpace.CS_GRAY))) {
			renderGray();
			return;
		}
		if (colorModel instanceof ComponentColorModel
				&& colorModel.getColorSpace().isCS_sRGB()) {
			renderRGB();
			return;
		}

		colorModel = new ComponentColorModel(ColorSpace
				.getInstance(ColorSpace.CS_sRGB), true, false,
				Transparency.TRANSLUCENT, DataBuffer.TYPE_BYTE);
		raster = colorModel.createCompatibleWritableRaster(bufferedImage
				.getWidth(), bufferedImage.getHeight());
		newImage = new BufferedImage(colorModel, raster, false, null);
		graphics = newImage.createGraphics();
		graphics.clearRect(0, 0, bufferedImage.getWidth(), bufferedImage
				.getHeight());
		graphics.drawImage(bufferedImage, 0, 0, null);
		graphics.dispose();
		renderRGB();
	}

	/**
	 * render the image. image has an indexed color model
	 * 
	 * @param image
	 *            the image content
	 * 
	 * @throws UnsupportedOperationException
	 *             If the image could not be rendered, e.g. not support image
	 *             type
	 */
	@SuppressWarnings("null")
	protected void renderIndexed() {
		IndexColorModel colorModel;
		COSArray colorSpace;
		int[] rgbs;
		byte[] colors;
		int pixelSize;
		Raster raster;
		int transparentPixel;
		int rowSize;
		byte[] bytes;
		byte[] maskBytes;

		colorModel = (IndexColorModel) bufferedImage.getColorModel();
		rgbs = new int[colorModel.getMapSize()];
		colorModel.getRGBs(rgbs);
		colors = new byte[rgbs.length * 3];
		for (int index = 0; index < rgbs.length; index++) {
			colors[index * 3] = (byte) (rgbs[index] >> 16);
			colors[(index * 3) + 1] = (byte) (rgbs[index] >> 8);
			colors[(index * 3) + 2] = (byte) rgbs[index];
		}
		colorSpace = COSArray.create(4);
		colorSpace.add(PDColorSpace.CN_CS_Indexed);
		colorSpace.add(PDColorSpace.CN_CS_DeviceRGB);
		colorSpace.add(COSInteger.create(colorModel.getMapSize() - 1));
		// TODO 2 @ehk spec says stream should be used
		// COSStream colorStream = COSStream.create(null);
		// colorStream.setUnfilteredBytes(colors);
		// colorSpace.add(colorStream);
		colorSpace.add(COSString.create(colors));
		getPDImage().cosSetColorSpace(colorSpace);

		pixelSize = bufferedImage.getColorModel().getPixelSize();
		getPDImage().setBitsPerComponent(pixelSize);

		raster = bufferedImage.getData();
		getPDImage().setWidth(raster.getWidth());
		getPDImage().setHeight(raster.getHeight());

		rowSize = ((raster.getWidth() * pixelSize) + 7) / 8;
		bytes = new byte[rowSize * raster.getHeight()];

		transparentPixel = ((IndexColorModel) bufferedImage.getColorModel())
				.getTransparentPixel();
		if (transparentPixel != -1) {
			/*
			 * this is for b/w masks; doesn't work maskBytes = new
			 * byte[(raster.getWidth() + 7) / 8];
			 */
			maskBytes = new byte[raster.getWidth() * raster.getHeight()];
			for (int index = 0; index < maskBytes.length; index++) {
				maskBytes[index] = -1;
			}
		} else {
			maskBytes = null;
		}

		for (int row = raster.getMinY(); row < (raster.getHeight() - raster
				.getMinY()); row++) {
			byte[] dataElements;

			try {
				dataElements = (byte[]) raster.getDataElements(
						raster.getMinX(), row, raster.getWidth(), 1, null);
			} catch (ClassCastException ex) {
				// TODO 2 @ehk implement int[] data elements
				throw new UnsupportedOperationException();
			}

			// following code assumes pixel size is one of 1, 2, 4, or 8
			// don't know if others are possible
			for (int index = 0; index < dataElements.length; index++) {
				int byteIndex;

				byteIndex = ((index * pixelSize) / 8) + (row * rowSize);
				bytes[byteIndex] = (byte) (bytes[byteIndex] | (dataElements[index] << (8 - pixelSize - ((index * pixelSize) % 8))));
				if ((dataElements[index] & 0xFF) == transparentPixel) {
					/*
					 * this ist for b/w masks; doesn't work byteIndex = (index /
					 * 8) + (row * ((raster.getWidth() + 7) / 8));
					 * maskBytes[byteIndex] = (byte) ( maskBytes[byteIndex] ^ (1
					 * << (7 - (index % 8))) );
					 */
					maskBytes[(row * raster.getWidth()) + index] = 0;
				}
			}
		}
		if (maskBytes != null) {
			PDImage maskImage;

			maskImage = (PDImage) PDImage.META.createNew();
			/*
			 * this ist for b/w masks; doesn't work
			 * maskImage.setBitsPerComponent(1); maskImage.setImageMask(true);
			 */
			maskImage.cosSetColorSpace(PDColorSpace.CN_CS_DeviceGray);
			maskImage.setBitsPerComponent(8);
			maskImage.setWidth(raster.getWidth());
			maskImage.setHeight(raster.getHeight());
			maskImage.setBytes(maskBytes);
			maskImage.cosGetStream().addFilter(Filter.CN_Filter_FlateDecode);
			// TODO 2 @ehk use b/w mask

			/* setMask(maskImage); */
			getPDImage().setSMask(maskImage);
		}

		getPDImage().setBytes(bytes);
	}

	/**
	 * render the img as a gray level raster.
	 * 
	 * @param image
	 *            the image content
	 * 
	 * @throws UnsupportedOperationException
	 *             If the image could not be rended, e.g. not support image type
	 */
	protected boolean renderJpegEncoded() {
		ImageWriter writer;
		ByteArrayOutputStream stream;
		ImageOutputStream imageStream = null;

		try {
			writer = ImageIO.getImageWritersByFormatName(IMAGEWRITER_JPEG)
					.next();
		} catch (NoSuchElementException ex) {
			return false;
		}
		try {
			stream = new ByteArrayOutputStream();
			imageStream = ImageIO.createImageOutputStream(stream);
			writer.setOutput(imageStream);
			JPEGImageWriteParam iwparam = new JPEGImageWriteParam(Locale
					.getDefault());
			iwparam.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
			iwparam.setCompressionQuality(getCompressionQuality());
			iwparam.setProgressiveMode(ImageWriteParam.MODE_COPY_FROM_METADATA);
			writer
					.write(null, new IIOImage(bufferedImage, null, null),
							iwparam);
		} catch (IOException ex) {
			return false;
		} finally {
			if (writer != null) {
				writer.reset();
				writer.dispose();
			}
			if (imageStream != null) {
				try {
					imageStream.flush();
				} catch (IOException e) {
					// 
				}
				try {
					imageStream.close();
				} catch (IOException e) {
					// 
				}
			}
		}

		getPDImage().cosGetStream().addFilter(Filter.CN_Filter_DCTDecode);
		getPDImage().cosGetStream().setEncodedBytes(stream.toByteArray());
		return true;
	}

	/**
	 * render the img as a RGB raster.
	 * 
	 * @param image
	 *            the image content
	 * 
	 * @throws UnsupportedOperationException
	 *             If the image could not be rended, e.g. not support image type
	 */
	protected void renderRGB() {
		int pixelSize;
		Raster raster;
		byte[] dataElements;
		byte[] bytes;

		getPDImage().cosSetColorSpace(PDColorSpace.CN_CS_DeviceRGB);

		pixelSize = bufferedImage.getColorModel().getPixelSize();

		if ((pixelSize / bufferedImage.getColorModel().getNumComponents()) != 8) {
			// TODO 2 @ehk implement other component sizes
			throw new UnsupportedOperationException();
		}

		getPDImage().setBitsPerComponent(8);

		raster = bufferedImage.getData();
		getPDImage().setWidth(raster.getWidth());
		getPDImage().setHeight(raster.getHeight());

		if (!bufferedImage.getColorModel().hasAlpha() && preferJpeg
				&& renderJpegEncoded()) {
			return;
		}
		try {
			dataElements = (byte[]) raster.getDataElements(raster.getMinX(),
					raster.getMinY(), raster.getWidth(), raster.getHeight(),
					null);
		} catch (ClassCastException ex) {
			throw new UnsupportedOperationException();
		}

		if (bufferedImage.getColorModel().hasAlpha()) {
			PDImage mask;
			byte[] maskBytes;

			bytes = new byte[dataElements.length / 4 * 3];
			maskBytes = new byte[dataElements.length / 4];

			mask = (PDImage) PDImage.META.createNew();
			mask.cosSetColorSpace(PDColorSpace.CN_CS_DeviceGray);
			mask.setBitsPerComponent(8);
			mask.setWidth(raster.getWidth());
			mask.setHeight(raster.getHeight());
			for (int i = 0; i < (dataElements.length / 4); i++) {
				bytes[i * 3] = dataElements[i * 4];
				bytes[(i * 3) + 1] = dataElements[(i * 4) + 1];
				bytes[(i * 3) + 2] = dataElements[(i * 4) + 2];
				maskBytes[i] = dataElements[(i * 4) + 3];
			}
			mask.setBytes(maskBytes);
			mask.cosGetStream().addFilter(Filter.CN_Filter_FlateDecode);
			getPDImage().setSMask(mask);
		} else {
			bytes = new byte[dataElements.length];
			System.arraycopy(dataElements, 0, bytes, 0, dataElements.length);
		}

		getPDImage().setBytes(bytes);
	}

	public void setCompressionQuality(float compression) {
		this.compressionQuality = compression;
	}

	public void setPreferJpeg(boolean preferJpeg) {
		this.preferJpeg = preferJpeg;
	}

}
