/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.adapter.awt;

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Paint;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.lang.ref.SoftReference;

import de.intarsys.cwt.awt.environment.CwtAwtGraphicsContext;
import de.intarsys.cwt.awt.environment.CwtAwtGraphicsEnvironment;
import de.intarsys.cwt.common.BlendMode;
import de.intarsys.cwt.environment.IGraphicsContext;
import de.intarsys.cwt.environment.IGraphicsEnvironment;
import de.intarsys.pdf.pd.PDAxialShading;
import de.intarsys.pdf.pd.PDFunction;
import de.intarsys.pdf.pd.PDImage;
import de.intarsys.pdf.platform.cwt.adapter.CommonGraphicsEnvironmentAdapter;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpace;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpaceFactory;
import de.intarsys.pdf.platform.cwt.image.awt.ImageConverterPdf2Awt;
import de.intarsys.pdf.platform.cwt.paint.ColorPaint;
import de.intarsys.pdf.platform.cwt.paint.PatternPaint;
import de.intarsys.pdf.platform.cwt.paint.ShadingPaint;
import de.intarsys.tools.attribute.Attribute;

public class AwtGraphicsEnvironmentAdapter extends
		CommonGraphicsEnvironmentAdapter {

	private static AwtGraphicsEnvironmentAdapter ACTIVE;

	private static final Attribute ATTR_BUFFEREDIMAGE = new Attribute(
			"bufferedImage");

	static public AwtGraphicsEnvironmentAdapter get() {
		return ACTIVE;
	}

	public AwtGraphicsEnvironmentAdapter() {
		ACTIVE = this;
	}

	protected float adjustAlpha(float[] rgb, float alpha) {
		float alphaFactor = .5f;
		if (rgb != null) {
			float luminance = (rgb[0] + rgb[1] + rgb[2]) / 3;
			alphaFactor = 1 - luminance;
		}
		return alpha * alphaFactor;
	}

	synchronized public BufferedImage createImage(PDImage pdImage) {
		BufferedImage result = (BufferedImage) pdImage
				.getAttribute(ATTR_BUFFEREDIMAGE);
		if (result == null) {
			ImageConverterPdf2Awt converter = new ImageConverterPdf2Awt(pdImage);
			result = converter.getBufferedImage();
			pdImage.setAttribute(ATTR_BUFFEREDIMAGE, result);
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.intarsys.pdf.environment.IImageFactory#createImage(de.intarsys.pdf
	 * .pd.PDImage, float[])
	 */
	public BufferedImage createImage(PDImage pdImage, float[] rgb) {
		int red;
		int green;
		int blue;
		String key;
		byte[] cmap;
		BufferedImage image;

		red = (int) (rgb[0] * 255);
		green = (int) (rgb[1] * 255);
		blue = (int) (rgb[2] * 255);

		key = "awt-iimage-" + ((red << 16) | (green << 8) | blue); //$NON-NLS-1$
		SoftReference ref = (SoftReference) pdImage.getAttribute(key);
		if (ref != null) {
			image = (BufferedImage) ref.get();
			if (image != null) {
				return image;
			}
		}

		image = createImage(pdImage);
		// make sure second color is different
		cmap = new byte[] { (byte) red, (byte) green, (byte) blue,
				(byte) (red == 0 ? 1 : 0), 0, 0 };
		ColorModel colorModel = new IndexColorModel(1, 2, cmap, 0, false, 1);
		// TODO see if we can avoid always creating a buffered image
		image = new BufferedImage(colorModel, image.getRaster(), false, null);
		pdImage.setAttribute(key, new SoftReference(image));
		return image;
	}

	protected Paint createPaintColorPaint(ColorPaint paint, boolean adjust) {
		AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
				.createColorSpace(paint.getPdColorSpace());
		float[] colorValues = paint.getColorValues();
		float[] rgb;
		if (colorValues == null) {
			rgb = new float[] { 0, 0, 0 };
		} else {
			rgb = awtColorSpace.getRGB(colorValues);
		}
		float alpha = paint.getAlphaValue();
		if (adjust) {
			alpha = adjustAlpha(rgb, alpha);
		}
		return new Color(rgb[0], rgb[1], rgb[2], alpha);
	}

	protected Paint createPaintPatternPaint(PatternPaint paint) {
		// TODO 2 @ehk
		// use a half transparent grey as a dummy
		return new Color(0.5f, 0.5f, 0.5f, 0.5f);
	}

	protected Paint createPaintShadingPaint(ShadingPaint paint) {
		// TODO 2 @ehk
		PDAxialShading axialShading;
		float[] coords;
		float[] domain;
		PDFunction function;
		float[] rgb;
		Color rgbColor1;
		Color rgbColor2;

		if (!(paint.getPdShading() instanceof PDAxialShading)) {
			// use a half transparent gray as a dummy
			return new Color(0.5f, 0.5f, 0.5f, 0.5f);
		}

		axialShading = (PDAxialShading) paint.getPdShading();
		coords = axialShading.getCoords();
		domain = axialShading.getDomain();
		function = axialShading.getFunction();
		AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
				.createColorSpace(paint.getPdShading().getColorSpace());
		rgb = awtColorSpace
				.getRGB(function.evaluate(new float[] { domain[0] }));
		rgbColor1 = new Color(rgb[0], rgb[1], rgb[2]);
		rgb = awtColorSpace
				.getRGB(function.evaluate(new float[] { domain[1] }));
		rgbColor2 = new Color(rgb[0], rgb[1], rgb[2]);
		return new GradientPaint(coords[0], coords[1], rgbColor1, coords[2],
				coords[3], rgbColor2);
	}

	public void drawImage(IGraphicsContext graphicsContext, PDImage pdImage,
			float x, float y) {
		BufferedImage awtImage = createImage(pdImage);
		// do some post transformations..
		if (pdImage.isImageMask()) {
			// todo support masked patterns....
			float[] rgb = graphicsContext.getBackgroundColor()
					.getRGBColorComponents(null);
			awtImage = createImage(pdImage, rgb);
		}
		//
		float definedWidth = pdImage.getWidth();
		float definedHeight = pdImage.getHeight();
		float realWidth = awtImage.getWidth();
		float realHeight = awtImage.getHeight();
		AffineTransform tempTransform = null;
		if (definedHeight < realHeight || definedWidth < realWidth) {
			tempTransform = graphicsContext.getTransform();
			AffineTransform transform = new AffineTransform();
			transform.scale(definedWidth / realWidth, definedHeight
					/ realHeight);
			graphicsContext.transform(transform);
		}
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		try {
			awtGC.getGraphics().drawImage(awtImage, (int) x, (int) y, null);
		} finally {
			if (tempTransform != null) {
				graphicsContext.setTransform(tempTransform);
			}
		}
	}

	public IGraphicsEnvironment getGraphicsEnvironment() {
		return CwtAwtGraphicsEnvironment.get();
	}

	public void setBackgroundColorPaint(IGraphicsContext graphicsContext,
			ColorPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintColorPaint(paint,
				awtGC.getBlendMode() == BlendMode.MULTIPLY);
		awtGC.setBackgroundPaint(awtPaint);
	}

	public void setBackgroundPatternPaint(IGraphicsContext graphicsContext,
			PatternPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintPatternPaint(paint);
		awtGC.setBackgroundPaint(awtPaint);
	}

	public void setBackgroundShadingPaint(IGraphicsContext graphicsContext,
			ShadingPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintShadingPaint(paint);
		awtGC.setBackgroundPaint(awtPaint);
	}

	public void setForegroundColorPaint(IGraphicsContext graphicsContext,
			ColorPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintColorPaint(paint,
				awtGC.getBlendMode() == BlendMode.MULTIPLY);
		awtGC.setForegroundPaint(awtPaint);
	}

	public void setForegroundPatternPaint(IGraphicsContext graphicsContext,
			PatternPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintPatternPaint(paint);
		awtGC.setForegroundPaint(awtPaint);
	}

	public void setForegroundShadingPaint(IGraphicsContext graphicsContext,
			ShadingPaint paint) {
		CwtAwtGraphicsContext awtGC = (CwtAwtGraphicsContext) graphicsContext;
		Paint awtPaint = createPaintShadingPaint(paint);
		awtGC.setForegroundPaint(awtPaint);
	}
}
