package com.franciaflex.faxtomail.ui.swing.util;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: AbstractFaxToMailDemandListHandler.java 139 2014-06-05 16:43:31Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/util/AbstractFaxToMailDemandListHandler.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.HasLabel;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.persistence.entities.Priority;
import com.franciaflex.faxtomail.persistence.entities.RangeRow;
import com.franciaflex.faxtomail.persistence.entities.Reply;
import com.franciaflex.faxtomail.ui.swing.FaxToMailUIContext;
import com.franciaflex.faxtomail.ui.swing.actions.ShowDemandeAction;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.content.attachment.AttachmentCellEditor;
import com.franciaflex.faxtomail.ui.swing.content.attachment.AttachmentCellRenderer;
import com.franciaflex.faxtomail.ui.swing.content.demande.replies.RepliesCellEditor;
import com.franciaflex.faxtomail.ui.swing.content.demande.replies.RepliesCellRenderer;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.jdesktop.swingx.table.TableColumnModelExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.validator.NuitonValidatorFactory;
import org.nuiton.validator.NuitonValidatorResult;

import javax.swing.*;
import javax.swing.event.ChangeListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

/**
 * @author Kevin Morin (Code Lutin)
 * @since x.x
 */
public abstract class AbstractFaxToMailDemandListHandler<M, UI extends FaxToMailUI<M, ?>> extends AbstractFaxToMailUIHandler<M, UI> {

    private static final Log log = LogFactory.getLog(AbstractFaxToMailDemandListHandler.class);

    public void initDemandeTable(final JXTable table, boolean sortable) {
//        table.setAutoResizeMode(JTable.AUTO_RESIZE_NEXT_COLUMN);

        populateColumnModel(table, sortable);

        table.getTableHeader().setReorderingAllowed(false);

        Highlighter highlighter = new Highlighter() {
            @Override
            public Component highlight(Component renderer, ComponentAdapter adapter) {
                int viewRow = adapter.row;
                int modelRow = adapter.convertRowIndexToModel(viewRow);
                DemandeUIModel row = ((AbstractApplicationTableModel<DemandeUIModel>) table.getModel()).getEntry(modelRow);
                boolean highlight = row.getTakenBy() == null;
                if (highlight) {
                    Font font = renderer.getFont();
                    renderer.setFont(font.deriveFont(Font.BOLD));
                }
                highlight = row.getArchiveDate() != null;
                if (highlight) {
                    Font font = renderer.getFont();
                    renderer.setFont(font.deriveFont(Font.ITALIC));
                    renderer.setForeground(adapter.isSelected() ? Color.WHITE : new Color(96, 96, 96));
                }
                return renderer;
            }

            @Override
            public void addChangeListener(ChangeListener l) {
            }

            @Override
            public void removeChangeListener(ChangeListener l) {
            }

            @Override
            public ChangeListener[] getChangeListeners() {
                return new ChangeListener[0];
            }
        };
        table.addHighlighter(highlighter);

        HighlightPredicate rowIsInvalidPredicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                int viewRow = adapter.row;
                int modelRow = adapter.convertRowIndexToModel(viewRow);
                DemandeUIModel row = ((AbstractApplicationTableModel<DemandeUIModel>) table.getModel()).getEntry(modelRow);
                return !row.isValid() && row.getTakenBy()!= null;
            }
        };
        Color color = new Color(255, 51, 51);
        table.addHighlighter(new ColorHighlighter(rowIsInvalidPredicate, color, Color.WHITE, color.darker(), Color.WHITE));

        HighlightPredicate testPredicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                int viewRow = adapter.row;
                int modelRow = adapter.convertRowIndexToModel(viewRow);
                DemandeUIModel row = ((AbstractApplicationTableModel<DemandeUIModel>) table.getModel()).getEntry(modelRow);
                return row.getLastAttachmentOpeningInThisFolderUser() != null
                        && !row.getLastAttachmentOpeningInThisFolderUser().equals(row.getTakenBy());
            }
        };
        color = Color.ORANGE;
        table.addHighlighter(new ColorHighlighter(testPredicate, color, Color.WHITE, color.darker(), Color.WHITE));

        table.addMouseListener(getDataTableMouseListener());
    }

    protected abstract Collection<String> getColumns();

    public String populateColumnModel(JXTable table, boolean sortable) {
        Collection<String> columns = getColumns();

        TableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        if (CollectionUtils.isEmpty(columns)) {
            columns = Lists.newArrayList(Email.PROPERTY_PRIORITY,
                                         Email.PROPERTY_DEMAND_TYPE,
                                         Email.PROPERTY_OBJECT,
                                         DemandeUIModel.PROPERTY_CLIENT_CODE,
                                         DemandeUIModel.PROPERTY_CLIENT_BRAND,
                                         Email.PROPERTY_SENDER,
                                         Email.PROPERTY_PROJECT_REFERENCE,
                                         Email.PROPERTY_COMPANY_REFERENCE,
                                         DemandeUIModel.PROPERTY_REFERENCE,
                                         Email.PROPERTY_RECEPTION_DATE,
                                         Email.PROPERTY_RECIPIENT,
                                         Email.PROPERTY_DEMAND_STATUS,
                                         Email.PROPERTY_ETAT_ATTENTE,
                                         DemandeUIModel.PROPERTY_PF_NB,
                                         DemandeUIModel.PROPERTY_SAV_NB,
                                         Email.PROPERTY_TAKEN_BY,
                                         DemandeUIModel.PROPERTY_LAST_ATTACHMENT_OPENING_IN_THIS_FOLDER_USER,
                                         Email.PROPERTY_REPLIES,
                                         Email.PROPERTY_ATTACHMENT);
        }

        for (String columnName : columns) {

            if (Email.PROPERTY_PRIORITY.equals(columnName)) {
                addComboDataColumnToModel(columnModel,
                                          DemandeTableModel.COLUMN_IDENTIFIERS.get(columnName),
                                          getDecorator(Priority.class, null),
                                          getContext().getReferentielService().getAllPriority()).setSortable(sortable);

            } else if (Email.PROPERTY_ATTACHMENT.equals(columnName)) {
                addColumnToModel(columnModel,
                                 AttachmentCellEditor.newEditor(ui),
                                 AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                                 DemandeTableModel.COLUMN_IDENTIFIERS.get(columnName)).setSortable(sortable);

            } else if (Email.PROPERTY_REPLIES.equals(columnName)) {
                addColumnToModel(columnModel,
                                 RepliesCellEditor.newEditor(ui),
                                 RepliesCellRenderer.newRender(getDecorator(Reply.class, null)),
                                 DemandeTableModel.COLUMN_IDENTIFIERS.get(columnName)).setSortable(sortable);

            } else {
                TableCellRenderer tableCellRenderer;
                switch (columnName) {
                    case Email.PROPERTY_RECEPTION_DATE:
                        tableCellRenderer = newTableCellRender(Date.class);
                        break;

                    case DemandeUIModel.PROPERTY_PF_NB:
                        tableCellRenderer = new QuantityTableCellRenderer(new Function<RangeRow, Integer>() {
                            @Override
                            public Integer apply(RangeRow input) {
                                return input.getProductQuantity();
                            }
                        });
                        break;
                    case DemandeUIModel.PROPERTY_SAV_NB:
                        tableCellRenderer = new QuantityTableCellRenderer(new Function<RangeRow, Integer>() {
                            @Override
                            public Integer apply(RangeRow input) {
                                return input.getSavQuantity();
                            }
                        });
                        break;

                    case Email.PROPERTY_DEMAND_STATUS:
                        tableCellRenderer = newTableCellRender(DemandStatus.class);
                        break;

                    case Email.PROPERTY_DEMAND_TYPE:
                    case Email.PROPERTY_ETAT_ATTENTE:
                        tableCellRenderer = newTableCellRender(HasLabel.class);
                        break;

                    case Email.PROPERTY_TAKEN_BY:
                    case DemandeUIModel.PROPERTY_LAST_ATTACHMENT_OPENING_IN_THIS_FOLDER_USER:
                        tableCellRenderer = newTableCellRender(FaxToMailUser.class);
                        break;

                    default:
                        tableCellRenderer = newTableCellRender(new Decorator<Object>(Object.class) {
                            @Override
                            public String toString(Object bean) {
                                return bean == null ? "" : bean.toString();
                            }
                        });
                }
                addColumnToModel(columnModel,
                                 null,
                                 tableCellRenderer,
                                 DemandeTableModel.COLUMN_IDENTIFIERS.get(columnName)).setSortable(sortable);
            }
        }

        final AbstractApplicationTableModel<DemandeUIModel> tableModel = new DemandeTableModel(columnModel,
                                                                                               getEditableTableProperties());
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        return StringUtils.join(columns, "-");
    }

    protected MouseListener getDataTableMouseListener() {
        return new MouseAdapter() {
            public void mouseClicked(MouseEvent evt) {
                if (evt.getClickCount() == 2) {
                    JXTable table = (JXTable) evt.getSource();
                    int rowIndex = table.getSelectedRow();
                    if (rowIndex < 0 || rowIndex >= table.getRowCount()) {
                        return;
                    }
                    rowIndex = table.convertRowIndexToModel(rowIndex);

                    AbstractApplicationTableModel<DemandeUIModel> tableModel =
                            (AbstractApplicationTableModel<DemandeUIModel>) table.getModel();
                    DemandeUIModel selectedEmail = tableModel.getEntry(rowIndex);
                    onDoubleClickOnDemande(selectedEmail);
                    FaxToMailUIContext context = getContext();
                    Email email = context.getEmailService().addToHistory(selectedEmail.getTopiaId(),
                                                                         HistoryType.OPENING,
                                                                         context.getCurrentUser(),
                                                                         new Date());
                    selectedEmail.fromEntity(email);
                    openDemand(selectedEmail);
                }
            }
        };
    }

    @Override
    public void autoSelectRowInTable(MouseEvent e, JPopupMenu popup) {
        boolean rightClick = SwingUtilities.isRightMouseButton(e);
        if (rightClick) {
            // get the coordinates of the mouse click
            Point p = e.getPoint();

            JXTable source = (JXTable) e.getSource();

            int[] selectedRows = source.getSelectedRows();

            // get the row index at this point
            int rowIndex = source.rowAtPoint(p);
            if (rowIndex == -1 || !ArrayUtils.contains(selectedRows, rowIndex)) {
                source.clearSelection();
            }
        }

        super.autoSelectRowInTable(e, popup);
    }

    protected abstract String[] getEditableTableProperties();

    protected void onDoubleClickOnDemande(DemandeUIModel selectedEmail) {
    }

    protected void openDemand(DemandeUIModel selectedEmail) {
        FaxToMailUIContext context = getContext();
        context.setCurrentEmail(selectedEmail);
        context.getActionEngine().runAction(new ShowDemandeAction(context.getMainUI().getHandler()));
    }

    public boolean isDemandeValid(DemandeUIModel d) {
        NuitonValidatorResult result = NuitonValidatorFactory.newValidator(DemandeUIModel.class).validate(d);
        return result.isValid();
    }

    public class QuantityTableCellRenderer extends DefaultTableCellRenderer {

        protected Function<RangeRow, Integer> getQuantityFunction;

        public QuantityTableCellRenderer(Function<RangeRow, Integer> getQuantityFunction) {
            this.getQuantityFunction = getQuantityFunction;
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasfocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasfocus, row, column);

            AbstractApplicationTableModel<DemandeUIModel> tableModel = (AbstractApplicationTableModel<DemandeUIModel>) table.getModel();
            int rowInModel = table.convertRowIndexToModel(row);
            DemandeUIModel demande = tableModel.getEntry(rowInModel);

            String tooltipText = null;

            List<String> ranges = new ArrayList<String>();
            Collection<RangeRow> rangeRows = demande.getRangeRow();
            for (RangeRow rangeRow : rangeRows) {
                Integer qty = getQuantityFunction.apply(rangeRow);
                if (qty != null && qty > 0) {
                    ranges.add(rangeRow.getRange().getLabel() + " : " + qty);
                }
            }

            if (!ranges.isEmpty()) {
                tooltipText = "<html><body>" + StringUtils.join(ranges, "<br/>") + "</body></html>";
            }
            setToolTipText(tooltipText);

            return this;
        }
    }
}
