package com.franciaflex.faxtomail.ui.swing.content.demande;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Franciaflex
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.DemandType;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.EtatAttente;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.persistence.entities.MailField;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.Range;
import com.franciaflex.faxtomail.persistence.entities.RangeRow;
import com.franciaflex.faxtomail.services.service.ReferentielService;
import com.franciaflex.faxtomail.ui.swing.actions.SaveDemandeAction;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.CloseableUI;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.franciaflex.faxtomail.ui.swing.content.attachment.AttachmentEditorUIModel;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.jdesktop.swingx.table.TableColumnModelExt;
import org.nuiton.jaxx.application.swing.tab.TabHandler;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.jaxx.application.swing.table.MoveToNextEditableCellAction;
import org.nuiton.jaxx.application.swing.table.MoveToNextEditableRowAction;
import org.nuiton.jaxx.application.swing.table.MoveToPreviousEditableCellAction;
import org.nuiton.jaxx.application.swing.table.MoveToPreviousEditableRowAction;
import org.nuiton.util.beans.BeanMonitor;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableCellEditor;
import java.awt.*;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler of UI {@link DemandeUIHandler}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class DemandeUIHandler extends AbstractFaxToMailUIHandler<DemandeUIModel, DemandeUI> implements CloseableUI, TabHandler {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DemandeUIHandler.class);

    protected BeanMonitor monitor;

    /**
     * Persistence service.
     *
     * @since 0.1
     */
//    private final PersistenceService persistenceService;

    @Override
    public void afterInit(DemandeUI ui) {

        initUI(ui);

        final DemandeUIModel model = getModel();
        Set<String> propertiesToIgnore = getPropertiesToIgnore();
        monitor = new BeanMonitor(true, propertiesToIgnore.toArray(new String[propertiesToIgnore.size()]));
        monitor.setBean(model);

        model.addPropertyChangeListener(Email.PROPERTY_DEMAND_TYPE, new PropertyChangeListener() {

            private int dividerLocation = -1;

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                DemandType newType = (DemandType) evt.getNewValue();
                DemandType oldType = (DemandType) evt.getOldValue();

                if (newType != null && (oldType == null
                        || !Objects.equals(oldType.containsFields(MailField.RANGE_ROW),
                                           newType.containsFields(MailField.RANGE_ROW)))) {
                    JSplitPane leftVerticalSplitPanel = getUI().getLeftVerticalSplitPanel();

                    if (Boolean.TRUE.equals(newType.containsFields(MailField.RANGE_ROW))) {
                        if (dividerLocation < 0) {
                            dividerLocation = leftVerticalSplitPanel.getLeftComponent().getPreferredSize().height + 10;
                        }
                        leftVerticalSplitPanel.setDividerLocation(dividerLocation);

                    } else {
                        dividerLocation = leftVerticalSplitPanel.getDividerLocation();
                    }
                }
            }
        });

        MailFolder folder = model.getMailFolder();
        ReferentielService referentielService = getContext().getReferentielService();

        initBeanFilterableComboBox(ui.getDocTypeComboBox(), referentielService.getAllDemandType(), model.getDemandType());
        initBeanFilterableComboBox(ui.getPriorityComboBox(), referentielService.getAllPriority(), model.getPriority());
        initBeanFilterableComboBox(ui.getStatusComboBox(), Arrays.asList(DemandStatus.values()), model.getDemandStatus());

        List<EtatAttente> etatAttentes = new ArrayList<EtatAttente>();
        Collection<EtatAttente> folderEtatAttentes = getEtatAttenteForFolder(folder);
        if (CollectionUtils.isEmpty(folderEtatAttentes)) {
            etatAttentes.addAll(referentielService.getAllEtatAttente());
        } else {
            etatAttentes.addAll(folderEtatAttentes);
        }
        initBeanFilterableComboBox(ui.getEtatAttenteComboBox(), etatAttentes, model.getEtatAttente());

        JTextPane editor = ui.getMailBodyField();
        FaxToMailUIUtil.setEmailContentInTextPane(editor, model);

        // init table
        final JXTable table = ui.getRangeTable();

        TableColumnModelExt columnModel = new DefaultTableColumnModelExt();
        addComboDataColumnToModel(columnModel,
                                  RangeTableModel.RANGE_COLUMN,
                                  getDecorator(Range.class, null),
                                  referentielService.getAllRange());
        addColumnToModel(columnModel,
                         RangeTableModel.COMMAND_NUMBER_COLUMN);
        addIntegerColumnToModel(columnModel,
                                RangeTableModel.QUOTATION_QUANTITY_COLUMN,
                                null,
                                table);
        addIntegerColumnToModel(columnModel,
                                RangeTableModel.PRODUCT_QUANTITY_COLUMN,
                                null,
                                table);
        addIntegerColumnToModel(columnModel,
                                RangeTableModel.SAV_QUANTITY_COLUMN,
                                null,
                                table);

        // init range model
        final RangeTableModel rangeTableModel = new RangeTableModel(columnModel);
        List<RangeRowModel> rangeRowModels = new ArrayList<RangeRowModel>();

        Collection<RangeRow> rangeRows = model.getRangeRow();
        if (rangeRows != null) {
            for (RangeRow rangeRow : rangeRows) {
                RangeRowModel rangeRowModel = new RangeRowModel();
                rangeRowModel.fromEntity(rangeRow);
                rangeRowModels.add(rangeRowModel);
                model.getValidRangeRowModels().add(rangeRowModel);
            }
        }
        rangeTableModel.setRows(rangeRowModels);

        Map<String, Integer> quantities = FaxToMailUIUtil.computeQuantities(rangeRowModels);
        model.setQuotationNb(quantities.get(DemandeUIModel.PROPERTY_QUOTATION_NB));
        model.setPfNb(quantities.get(DemandeUIModel.PROPERTY_PF_NB));
        model.setSavNb(quantities.get(DemandeUIModel.PROPERTY_SAV_NB));

        // add listener to update row validity and quantity totals
        table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {

            protected PropertyChangeListener listener = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    DemandeUIModel model = getModel();
                    String propertyName = evt.getPropertyName();

                    if (RangeRowModel.PROPERTY_VALID.equals(propertyName)) {
                        RangeRowModel row = (RangeRowModel) evt.getSource();
                        Boolean valid = (Boolean) evt.getNewValue();
                        if (Boolean.TRUE.equals(valid)) {
                            model.addValidRangeRow(row);
                        } else {
                            model.removeValidRangeRow(row);
                        }
                    }

                    if (RangeRowModel.PROPERTY_VALID.equals(propertyName)
                            || RangeRow.PROPERTY_QUOTATION_QUANTITY.equals(propertyName)
                            || RangeRow.PROPERTY_PRODUCT_QUANTITY.equals(propertyName)
                            || RangeRow.PROPERTY_SAV_QUANTITY.equals(propertyName)) {

                        List<RangeRowModel> rangeRows = rangeTableModel.getRows();
                        Map<String, Integer> quantities = FaxToMailUIUtil.computeQuantities(rangeRows);
                        model.setQuotationNb(quantities.get(DemandeUIModel.PROPERTY_QUOTATION_NB));
                        model.setPfNb(quantities.get(DemandeUIModel.PROPERTY_PF_NB));
                        model.setSavNb(quantities.get(DemandeUIModel.PROPERTY_SAV_NB));
                    }

                }
            };

            protected RangeRowModel currentRow;

            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel source = (ListSelectionModel) e.getSource();
                if (currentRow != null) {
                    currentRow.removePropertyChangeListener(listener);
                }

                if (source.isSelectionEmpty()) {
                    currentRow = null;
                } else {
                    int rowIndex = source.getLeadSelectionIndex();
                    currentRow = rangeTableModel.getEntry(rowIndex);
                    currentRow.addPropertyChangeListener(listener);
                }
            }
        });

        table.setModel(rangeTableModel);
        table.setColumnModel(columnModel);
        table.getTableHeader().setReorderingAllowed(false);

        final MoveToNextEditableCellAction nextCellAction =
                MoveToNextEditableCellAction.newAction(rangeTableModel, table);
        final MoveToPreviousEditableCellAction previousCellAction =
                MoveToPreviousEditableCellAction.newAction(rangeTableModel, table);

        final MoveToNextEditableRowAction nextRowAction =
                MoveToNextEditableRowAction.newAction(rangeTableModel, table);
        final MoveToPreviousEditableRowAction previousRowAction =
                MoveToPreviousEditableRowAction.newAction(rangeTableModel, table);

        KeyAdapter keyAdapter = new KeyAdapter() {

            @Override
            public void keyPressed(KeyEvent e) {
                TableCellEditor editor = table.getCellEditor();

                int keyCode = e.getKeyCode();
                if (keyCode == KeyEvent.VK_LEFT ||
                        (keyCode == KeyEvent.VK_TAB && e.isShiftDown())) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    previousCellAction.actionPerformed(null);

                } else if (keyCode == KeyEvent.VK_RIGHT ||
                        keyCode == KeyEvent.VK_TAB) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    nextCellAction.actionPerformed(null);

                } else if (keyCode == KeyEvent.VK_UP ||
                        (keyCode == KeyEvent.VK_ENTER && e.isShiftDown())) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    previousRowAction.actionPerformed(null);

                } else if (e.getKeyCode() == KeyEvent.VK_ENTER ||
                        keyCode == KeyEvent.VK_DOWN) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    nextRowAction.actionPerformed(null);
                }
            }
        };

        table.addKeyListener(keyAdapter);
        HighlightPredicate rowIsInvalidPredicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                boolean result = false;
                AbstractApplicationTableModel model = (AbstractApplicationTableModel) table.getModel();
                int viewRow = adapter.row;
                int modelRow = adapter.convertRowIndexToModel(viewRow);
                RangeRowModel row = (RangeRowModel) model.getEntry(modelRow);
                result = !row.isValid();
                return result;
            }
        };
        Color color = new Color(255, 51, 51);
        table.addHighlighter(new ColorHighlighter(rowIsInvalidPredicate, color, Color.WHITE, color.darker(), Color.WHITE));

        SwingValidator validator = getValidator();
        listenValidatorValid(validator, model);

        // if new fishingOperation can already cancel his creation
        model.setModify(false);

        registerValidators(validator);

        ui.getAttachmentsButton().getBean().addAttachmentListener(
                new AttachmentEditorUIModel.AttachmentListener() {

                    @Override
                    public void onAttachmentOpened(Attachment attachment, boolean original) {
                        String topiaId = getModel().getTopiaId();
                        if (topiaId != null) {
                            String filename;
                            if (original) {
                                filename = attachment.getOriginalFile().getFilename();
                            } else {
                                filename = FaxToMailUIUtil.getEditedFileName(attachment.getOriginalFile());
                            }
                            Email email = getContext().getEmailService().addToHistory(topiaId,
                                                                                      HistoryType.ATTACHMENT_OPENING,
                                                                                      getContext().getCurrentUser(),
                                                                                      new Date(),
                                                                                      filename);
                            getModel().setHistory(email.getHistory());
                        }
                    }

                    @Override
                    public void onAttachmentEdited(Attachment attachment) {
                        String topiaId = getModel().getTopiaId();
                        if (topiaId != null) {
                            Email email = getContext().getEmailService().addToHistory(topiaId,
                                                                                      HistoryType.ATTACHMENT_MODIFICATION,
                                                                                      getContext().getCurrentUser(),
                                                                                      new Date(),
                                                                                      FaxToMailUIUtil.getEditedFileName(attachment.getOriginalFile()));
                            getModel().setHistory(email.getHistory());
                        }
                    }


                }
        );

        final JSplitPane leftVerticalSplitPanel = getUI().getLeftVerticalSplitPanel();
        model.addPropertyChangeListener(DemandeUIModel.PROPERTY_RANGE_PANEL_VISIBLE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                leftVerticalSplitPanel.setName("leftVerticalSplitPanel" + evt.getNewValue());
                getContext().getSwingSession().add(leftVerticalSplitPanel, true);
                if (Boolean.FALSE.equals(evt.getNewValue())) {
                    leftVerticalSplitPanel.setDividerLocation(leftVerticalSplitPanel.getHeight());
                }
            }
        });

        leftVerticalSplitPanel.setName("leftVerticalSplitPanel" + model.isRangePanelVisible());
        getContext().getSwingSession().add(leftVerticalSplitPanel, true);

        listModelIsModify(getModel());
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getObjectField();
    }

    @Override
    protected Set<String> getPropertiesToIgnore() {
        Set<String> result = super.getPropertiesToIgnore();
        result.add(DemandeUIModel.PROPERTY_EDITABLE);
        result.add(Email.PROPERTY_HISTORY);
        result.add(DemandeUIModel.PROPERTY_GROUPED_DEMANDES);
        result.add(Email.PROPERTY_REPLIES);
        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
    }

    public void closeButtonPopups() {
        ui.getAttachmentsButton().setSelected(false);
        ui.getHistoryButton().setSelected(false);
        ui.getDemandRepliesButton().setSelected(false);
    }

    @Override
    public boolean quitUI() {
        BeanMonitor monitor = getMonitor();
        boolean result = quitScreen(
                true,
                monitor.wasModified(),
                t("faxtomail.demande.askCancelEditBeforeLeaving.cancelSave"),
                t("faxtomail.demande.askSaveBeforeLeaving.save"),
                getContext().getActionFactory().createUIAction(this, SaveDemandeAction.class)
        );
        return result;
    }

    @Override
    public SwingValidator<DemandeUIModel> getValidator() {
        return ui.getValidator();
    }

    public Collection<EtatAttente> getEtatAttenteForFolder(MailFolder folder) {
        Collection<EtatAttente> result = null;
        while (CollectionUtils.isEmpty(result) && folder != null) {
            result = folder.getEtatAttentes();
            folder = folder.getParent();
        }
        return result;
    }

    public BeanMonitor getMonitor() {
        return monitor;
    }

    @Override
    public boolean onHideTab(int currentIndex, int newIndex) {
        closeButtonPopups();
        return true;
    }

    @Override
    public void onShowTab(int currentIndex, int newIndex) {
    }

    @Override
    public boolean onRemoveTab() {
        return quitUI();
    }
}
