package com.franciaflex.faxtomail.ui.swing.content.demande;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Franciaflex
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.ezware.oxbow.swingbits.table.filter.DistinctColumnItem;
import com.ezware.oxbow.swingbits.table.filter.ITableFilter;
import com.ezware.oxbow.swingbits.table.filter.JTableFilter;
import com.ezware.oxbow.swingbits.table.filter.TableRowFilterSupport;
import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.MailField;
import com.franciaflex.faxtomail.persistence.entities.History;
import com.franciaflex.faxtomail.persistence.entities.HistoryImpl;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.ui.swing.actions.ArchiveFromListAction;
import com.franciaflex.faxtomail.ui.swing.actions.ComputeQuantitiesByRangeAction;
import com.franciaflex.faxtomail.ui.swing.actions.LoadFolderEmailsAction;
import com.franciaflex.faxtomail.ui.swing.actions.SaveDemandeFromListAction;
import com.franciaflex.faxtomail.ui.swing.content.reply.ReplyFormUI;
import com.franciaflex.faxtomail.ui.swing.content.reply.ReplyFormUIModel;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailDemandListHandler;
import com.franciaflex.faxtomail.ui.swing.util.CloseableUI;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.franciaflex.faxtomail.ui.swing.util.FolderTreeNode;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.table.TableModel;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler of UI {@link com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUIHandler}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class DemandeListUIHandler extends AbstractFaxToMailDemandListHandler<DemandeListUIModel, DemandeListUI> implements CloseableUI {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DemandeListUIHandler.class);

    public final PropertyChangeListener selectedDemandeChangeListener = new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            String propertyName = evt.getPropertyName();
            if (Email.PROPERTY_PRIORITY.equals(propertyName)) {
                if (evt.getNewValue() != null || evt.getOldValue() != null) {
                    SaveDemandeFromListAction saveAction =
                            getContext().getActionFactory().createLogicAction(DemandeListUIHandler.this,
                                                                              SaveDemandeFromListAction.class);
                    saveAction.setModifiedProperties(propertyName);
                    getContext().getActionEngine().runAction(saveAction);
                }
            }

        }
    };

    @Override
    public void beforeInit(DemandeListUI ui) {

        super.beforeInit(ui);

        DemandeListUIModel model = new DemandeListUIModel();
        Collection<MailFolder> folders = getContext().getMailFolderService().getRootMailFolders();
        model.setFolders(new ArrayList<MailFolder>(folders));

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_SELECTED_FOLDER, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LoadFolderEmailsAction loadFolderEmailsAction =
                        getContext().getActionFactory().createLogicAction(DemandeListUIHandler.this,
                                                                          LoadFolderEmailsAction.class);
                getContext().getActionEngine().runAction(loadFolderEmailsAction);
            }
        });

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(DemandeListUI ui) {

        initUI(ui);

        DemandeListUIModel model = getModel();

        // init table
        final JXTable dataTable = getUI().getDataTable();

        initDemandeTable(dataTable, false);

        final ITableFilter<JTable> tableFilter = new JTableFilter(dataTable) {

            @Override
            protected boolean execute(int col, Collection<DistinctColumnItem> items) {
                boolean b = super.execute(col, items);
                List<DemandeUIModel> filteredEmails = new ArrayList<DemandeUIModel>();
                AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                for (int i = 0; i < dataTable.getRowCount(); i++) {
                    int modelIndex = dataTable.convertRowIndexToModel(i);
                    filteredEmails.add(dataTableModel.getEntry(modelIndex));
                }
                getModel().setFilteredEmails(filteredEmails);
                return b;
            }

            @Override
            public void modelChanged(TableModel model) {
                // do nothing
            }

        };
        TableRowFilterSupport.forFilter(tableFilter).searchable(true).useTableRenderers(true).apply();

        dataTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {

            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel source = (ListSelectionModel) e.getSource();

                DemandeListUIModel model = getModel();
                if (source.isSelectionEmpty()) {
                    model.setCurrentEmails(null);
                } else {
                    List<DemandeUIModel> selectedRows = new ArrayList<DemandeUIModel>();
                    AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                    for (int i = source.getMinSelectionIndex() ; i <= source.getMaxSelectionIndex() ; i++) {
                        if (source.isSelectedIndex(i)) {
                            selectedRows.add(dataTableModel.getEntry(i));
                        }
                    }
                    model.setCurrentEmails(selectedRows);
                }
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_EMAILS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<DemandeUIModel> emails = (List<DemandeUIModel>) evt.getNewValue();
                AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                dataTableModel.setRows(emails);
                tableFilter.clear();
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_FILTERED_EMAILS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<DemandeUIModel> emails = (List<DemandeUIModel>) evt.getNewValue();

                int quotationNb = 0;
                int pfNb = 0;
                int savNb = 0;
                for (DemandeUIModel email : emails) {
                    quotationNb += email.getQuotationNb();
                    pfNb += email.getPfNb();
                    savNb += email.getSavNb();
                }

                DemandeListUIModel model = (DemandeListUIModel) evt.getSource();
                model.setQuotationNb(quotationNb);
                model.setPfNb(pfNb);
                model.setSavNb(savNb);
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_CURRENT_EMAILS, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<DemandeUIModel> oldDemands = (List<DemandeUIModel>) evt.getOldValue();
                if (oldDemands != null) {
                    for (DemandeUIModel demand : oldDemands) {
                        demand.removePropertyChangeListener(selectedDemandeChangeListener);
                    }
                }

                List<DemandeUIModel> newDemands = (List<DemandeUIModel>) evt.getNewValue();
                if (newDemands != null) {
                    for (DemandeUIModel demand : newDemands) {
                        demand.addPropertyChangeListener(selectedDemandeChangeListener);
                    }
                }
            }
        });

        // init tree
        Map<MailFolder, DefaultMutableTreeNode> nodesByFolder =
                FaxToMailUIUtil.initFolderTree(getContext(), ui.getNavigationTree(), model.getFolders());

        ui.getNavigationTree().addTreeSelectionListener(new TreeSelectionListener() {
            @Override
            public void valueChanged(TreeSelectionEvent e) {
                FolderTreeNode folderNode = (FolderTreeNode) e.getPath().getLastPathComponent();
                MailFolder folder = folderNode.getMailFolder();

                getModel().setSelectedFolder(folder);
                getContext().setCurrentMailFolder(folder);

                DemandeUIModel currentEmail = getContext().getCurrentEmail();
                if (currentEmail != null) {
                    AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                    int row = dataTableModel.getRowIndex(currentEmail);
                    if (row > 0) {
                        dataTable.setRowSelectionInterval(row, row);
                    }
                }
            }
        });

        MailFolder currentMailFolder = getContext().getCurrentMailFolder();
        DemandeUIModel currentEmail = getContext().getCurrentEmail();

        if (currentMailFolder == null && currentEmail != null) {
            currentMailFolder = currentEmail.getMailFolder();
        }
        if (currentMailFolder != null) {
            DefaultMutableTreeNode node = nodesByFolder.get(currentMailFolder);
            ui.getNavigationTree().setSelectionPath(new TreePath(node.getPath()));
        }

    }

    @Override
    protected Collection<String> getColumns() {
        Collection<String> columns = null;
        MailFolder selectedFolder = getModel().getSelectedFolder();
        if (selectedFolder != null) {
            List<MailField> tableColumns = selectedFolder.getFolderTableColumns();
            if (tableColumns != null) {
                Collections2.transform(tableColumns, new Function<MailField, String>() {
                    @Override
                    public String apply(MailField field) {
                        return field.toString();
                    }
                });
            }
        }
        return columns;
    }

    @Override
    protected String[] getEditableTableProperties() {
        return new String[] { Email.PROPERTY_PRIORITY, Email.PROPERTY_ATTACHMENT, Email.PROPERTY_REPLIES };
    }

    @Override
    protected void onDoubleClickOnDemande(DemandeUIModel selectedEmail) {
        super.onDoubleClickOnDemande(selectedEmail);
        selectedEmail.removePropertyChangeListener(selectedDemandeChangeListener);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNavigationTree();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        JTree tree = getUI().getNavigationTree();
        TreeModel treeModel = tree.getModel();
        Enumeration<TreePath> paths = tree.getExpandedDescendants(new TreePath(treeModel.getRoot()));

        List<MailFolder> folders = new ArrayList<MailFolder>();
        while (paths.hasMoreElements()) {
            TreePath path = paths.nextElement();
            Object lastPathComponent = path.getLastPathComponent();
            if (FolderTreeNode.class.isAssignableFrom(lastPathComponent.getClass())) {
                folders.add(((FolderTreeNode) lastPathComponent).getMailFolder());
            }
        }
        getContext().setExpandedFolders(folders);

        clearValidators();
    }

    @Override
    public boolean quitUI() {
//        boolean result = quitScreen(
//                getModel().isValid(),
//                getModel().isModify(),
//                _("tutti.editCruise.askCancelEditBeforeLeaving.cancelSaveCruise"),
//                _("tutti.editCruise.askSaveBeforeLeaving.saveCruise"),
//                ui.getSaveButton().getAction()
//        );
//        return result;
        return true;
    }

    @Override
    public SwingValidator<DemandeListUIModel> getValidator() {
        return null;
    }

    @Override
    protected void beforeOpenPopup(int rowIndex, int columnIndex) {
        super.beforeOpenPopup(rowIndex, columnIndex);

        int selectedRowCount = getUI().getDataTable().getSelectedRowCount();

        DemandeListUIModel model = getModel();
        //TODO kmorin 20140605 prendre en compte la conf en fonction des états d'attente
        model.setReplyEnabled(selectedRowCount == 1 && model.getCurrentEmails().get(0).isEditable());
        model.setArchiveEnabled(selectedRowCount > 1 ||
                                        selectedRowCount > 0 && model.getCurrentEmails().get(0).isEditable());
    }

    public void autoSelectNodeInTree(MouseEvent e, JPopupMenu popup) {

        boolean rightClick = SwingUtilities.isRightMouseButton(e);

        if (rightClick) {

            JTree source = (JTree) e.getSource();

            // get the row index at this point
            int rowIndex = source.getClosestRowForLocation(e.getX(), e.getY());

            if (log.isDebugEnabled()) {
                log.debug("At point [" + e.getPoint() + "] found Row " + rowIndex);
            }

            // select row (could empty selection)
            if (rowIndex == -1) {
                source.clearSelection();
            } else {
                // set selection
                source.setSelectionRow(rowIndex);
            }

            // on right click show popup
            popup.show(source, e.getX(), e.getY());
        }
    }

    public void computeQuantitiesByRange() {
        QuantitiesByRangeUI dialogContent = new QuantitiesByRangeUI(ui);
        getContext().getActionEngine().runAction(new ComputeQuantitiesByRangeAction(dialogContent.getHandler()));
    }

    public void newDemand() {
        FaxToMailUser currentUser = getContext().getCurrentUser();
        Date now = new Date();

        List<History> histories = new ArrayList<History>();
        DemandeUIModel email = new DemandeUIModel();
        History history = new HistoryImpl();
        history.setFaxToMailUser(currentUser);
        history.setType(HistoryType.CREATION);
        history.setModificationDate(now);
        histories.add(history);

        history = new HistoryImpl();
        history.setFaxToMailUser(currentUser);
        history.setType(HistoryType.OPENING);
        history.setModificationDate(now);
        histories.add(history);

        email.setMailFolder(getModel().getSelectedFolder());
        email.setReceptionDate(now);
        email.setTakenBy(currentUser);
        email.setHistory(histories);
        email.setDemandStatus(DemandStatus.UNTREATED);
        openDemand(email);
    }

    public void print() {
        //TODO kmorin 20140606 print a result page and maybe a page to separate the demands
        List<DemandeUIModel> currentEmails = getModel().getCurrentEmails();
        for (DemandeUIModel demandeUIModel : currentEmails) {
            for (Attachment attachment : demandeUIModel.getAttachment()) {
                AttachmentFile attachmentFile = attachment.getEditedFile();
                if (attachmentFile == null) {
                    attachmentFile = attachment.getOriginalFile();
                }
                FaxToMailUIUtil.print(attachmentFile, true);
            }
        }
    }

    public void reply() {
        DemandeUIModel demand = getModel().getCurrentEmails().get(0);
        getContext().setCurrentEmail(demand);

        ReplyFormUI dialogContent = new ReplyFormUI(ui);
        ReplyFormUIModel model = dialogContent.getModel();
        model.setTo(demand.getSender());

        openFrame(dialogContent, t("faxtomail.reply.title", demand.getObject()), new Dimension(800, 600));
    }

    public void archive() {
        ArchiveFromListAction saveAction =
                getContext().getActionFactory().createLogicAction(DemandeListUIHandler.this,
                                                                  ArchiveFromListAction.class);
        getContext().getActionEngine().runAction(saveAction);
    }

}
