package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: EmailService.java 162 2014-06-09 17:14:06Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/EmailService.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.MessagingException;

import com.franciaflex.faxtomail.persistence.entities.MailField;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.DefaultAuthenticator;
import org.apache.commons.mail.EmailConstants;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.MultiPartEmail;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFileImpl;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFileTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.persistence.entities.Client;
import com.franciaflex.faxtomail.persistence.entities.ClientTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.EmailFilter;
import com.franciaflex.faxtomail.persistence.entities.EmailGroup;
import com.franciaflex.faxtomail.persistence.entities.EmailGroupTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EmailTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.History;
import com.franciaflex.faxtomail.persistence.entities.HistoryTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.RangeRow;
import com.franciaflex.faxtomail.persistence.entities.RangeRowTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.Reply;
import com.franciaflex.faxtomail.persistence.entities.ReplyTopiaDao;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since x.x
 */
public class EmailService extends FaxToMailServiceSupport {

    private static final Log log = LogFactory.getLog(EmailService.class);

    protected Binder<Attachment, Attachment> attachmentBinder =
            BinderFactory.newBinder(Attachment.class, Attachment.class);

    public Email getEmailById(String id) {
        Email email = getPersistenceContext().getEmailDao().findByTopiaId(id);
        return email;
    }

    /**
     * Save email with default email client code.
     * 
     * @param email email
     * @param user user
     * @param modifiedFields modified fields
     * @return updated email instance
     * @throws InvalidClientException if client code is not valid
     */
    public Email saveEmail(Email email, FaxToMailUser user, String... modifiedFields) throws InvalidClientException {
        Client client = email.getClient();
        return saveEmail(email, null, client != null ? client.getCode() : null, user, modifiedFields);
    }

    public Email saveEmail(Email email, Collection<Attachment> attachments, String clientCode, FaxToMailUser user, String... modifiedFields) throws InvalidClientException {
        Date now = getNow();
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();

        // if client code is null, do not manage client at all
        Client client = null;
        if (StringUtils.isNotBlank(clientCode)) {
            client = getClientService().getClientForCode(clientCode);
            if (client == null) {
                String message = t("faxtomail.service.email.save.clientCode.error", clientCode);
                throw new InvalidClientException(message);
            }
        }
        email.setClient(client);

        if (attachments != null) {
            Collection<Attachment> currentAttachments = CollectionUtils.emptyIfNull(email.getAttachment());
            Map<String, Attachment> currentAttachmentIndex = new HashMap<>(Maps.uniqueIndex(currentAttachments, TopiaEntities.getTopiaIdFunction())); 

            AttachmentFileTopiaDao attachementFileTopiaDao = getPersistenceContext().getAttachmentFileDao();
            for (Attachment attachment : attachments) {
                // get session attachment from id
                Attachment currentAttachment;
                if (StringUtils.isNoneBlank(attachment.getTopiaId())) {
                    currentAttachment = currentAttachmentIndex.remove(attachment.getTopiaId());
                } else {
                    currentAttachment = new AttachmentImpl();
                }

                // copy new data
                attachmentBinder.copy(attachment, currentAttachment);

                // persist
                if (currentAttachment.getEditedFile() != null) {
                    if (!currentAttachment.getEditedFile().isPersisted()) {
                        attachementFileTopiaDao.create(currentAttachment.getEditedFile());
                    } else {
                        attachementFileTopiaDao.update(currentAttachment.getEditedFile());
                    }
                }
                if (currentAttachment.getOriginalFile() != null) {
                    if (!currentAttachment.getOriginalFile().isPersisted()) {
                        attachementFileTopiaDao.create(currentAttachment.getOriginalFile());
                    } else {
                        attachementFileTopiaDao.update(currentAttachment.getOriginalFile());
                    }
                }

                if (!attachment.isPersisted()) {
                    // persist using cascade
                    email.addAttachment(currentAttachment);
                }
            }
            
            // delete not found attachments
            for (Attachment attachment : currentAttachmentIndex.values()) {
                email.removeAttachment(attachment);
            }
        }

        if (email.getRangeRow() != null) {
            RangeRowTopiaDao rangeRowDao = getPersistenceContext().getRangeRowDao();
            for (RangeRow rangeRow : email.getRangeRow()) {
                if (!rangeRow.isPersisted()) {
                    rangeRowDao.create(rangeRow);
                } else {
                    rangeRowDao.update(rangeRow);
                }
            }
        }

        if (!email.isPersisted()) {
            if (email.getHistory() != null) {
                historyDao.createAll(email.getHistory());
            }

            email = dao.create(email);
        }

        Set<String> fieldSet = Sets.newHashSet(modifiedFields);

        if (StringUtils.isNotBlank(email.getMailFolder().getEdiFolder())) {
            History transmissionToEdi = CollectionUtils.find(email.getHistory(), new Predicate<History>() {
                @Override
                public boolean evaluate(History object) {
                    return object.getType() == HistoryType.TRANSMISSION_TO_EDI;
                }
            });

            // we transmit to EDI if:
            // - it has never been transmitted before
            // - all the required fields are filled
            if (transmissionToEdi == null
                    && email.getClient() != null
                    && email.getDemandType() != null
                    && StringUtils.isNotBlank(email.getProjectReference())
                    && (!email.getDemandType().containsFields(MailField.RANGE_ROW) || CollectionUtils.isNotEmpty(email.getRangeRow()))) {

                email.setDemandStatus(DemandStatus.TRANSMITTED_TO_EDI);
                transmissionToEdi = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION_TO_EDI,
                                                      History.PROPERTY_MODIFICATION_DATE, new Date());
                email.addHistory(transmissionToEdi);
                fieldSet.add(Email.PROPERTY_DEMAND_STATUS);
            }
        }

        History history;
        if (fieldSet.contains(Email.PROPERTY_ARCHIVE_DATE)) {
            history = historyDao.create(History.PROPERTY_TYPE, HistoryType.ARCHIVED,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_MODIFICATION_DATE, now);

        } else if (fieldSet.contains(Email.PROPERTY_MAIL_FOLDER)) {
            history = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_MODIFICATION_DATE, now);

        } else {
            if (email.getTakenBy() == null &&
                    !fieldSet.isEmpty() && !fieldSet.contains(Email.PROPERTY_TAKEN_BY)) {
                email.setTakenBy(user);
                fieldSet.add(Email.PROPERTY_TAKEN_BY);
            }

            history = historyDao.create(History.PROPERTY_TYPE, email.isHistoryEmpty() ? HistoryType.CREATION : HistoryType.MODIFICATION,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_FIELDS, fieldSet,
                                        History.PROPERTY_MODIFICATION_DATE, now);
        }
        email.addHistory(history);

        Email result = dao.update(email);
        getPersistenceContext().commit();

        return result;
    }

    public void transmitPendingDemandsToEdi() {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();

        List<Email> toTransmitToEdi = dao.forDemandStatusEquals(DemandStatus.TRANSMITTED_TO_EDI).findAll();
        for (Email email : toTransmitToEdi) {
            transmitDemandToEdi(email);
        }
    }

    /**
     * Generate txt file to send demand to EDI system.
     * 
     * @param email email to send
     */
    protected void transmitDemandToEdi(Email email) {
        Preconditions.checkArgument(email.getDemandStatus() == DemandStatus.TRANSMITTED_TO_EDI);

        if (CollectionUtils.isEmpty(email.getRangeRow())) {
            return;
        }

        //TODO kmorin 20140521 maybe check if the file is not being read

        // get output file with pattern name depending on date
        Date now = serviceContext.getNow();
        File ediDirectory = getApplicationConfig().getEdiDirectory();
        ediDirectory.mkdirs();

        String fileDate = DateFormatUtils.format(now, "yyMMddhhmmss");
        
        // create a file for each commande
        for (RangeRow rangeRow : email.getRangeRow()) {
            String fileName = "ORD_FAX_" + fileDate + "_" + rangeRow.getTopiaId() + ".txt";
            
            File ediFile = new File(ediDirectory, fileName);
    
            // generate output content
            String separator = ";";
            try (Writer ediWriter = new BufferedWriter(new FileWriter(ediFile))) {
                
                ediWriter.write("%BEGIN_ENTETE_QUOTE\n");
                
                //N° Champ/Champ/Type/Longueur/Observation
                //01/Id Enregistrement/Alpha/1/Cst : E (En-tête)
                ediWriter.write("E" + separator);
                //02/N° d’ordre achat/Numérique/6/Cst : OARFAX
                ediWriter.write("OARFAX" + separator);
                //03/Date traitement/Date/6/JJMMAA dte système
                ediWriter.write(DateFormatUtils.format(now, "ddMMyy") + separator);
                //04/Date livraison prévue/Date/6/JJMMAA dte système
                ediWriter.write(DateFormatUtils.format(now, "ddMMyy") + separator);
                //05/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //06/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //07/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //08/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //09//Référence client FX/Alpha/36/Référence Chantier
                ediWriter.write(Strings.nullToEmpty(email.getProjectReference()) + separator);
                //10/Nom du contact/Alpha/35/Vide
                ediWriter.write(separator);
                //11/Téléphone contact/Alpha/15/Vide
                ediWriter.write(separator);
                //12/Fax contact/Alpha/15/Vide
                ediWriter.write(separator);
                //13/Devise/Alpha/3/Cst : EUR
                ediWriter.write("EUR" + separator);
                //14/Condition de livraison/Alpha/3/Vide
                ediWriter.write(separator);
                //15/Type de commande/Alpha/Type de document
                ediWriter.write(email.getDemandType().getLabel() + separator);
                //16/Adresse livr : Nom/Alpha/35/Vide
                ediWriter.write(separator);
                //18/Adresse livr : adr1/Alpha/30/Vide
                ediWriter.write(separator);
                //19/Adresse livr : adr2/Alpha/30/Vide
                ediWriter.write(separator);
                //20/Adresse livr : cp ville/Alpha/30/Vide
                ediWriter.write(separator);
                //21/Adresse livr : compl./Alpha/30/Vide
                ediWriter.write(separator);
                //22/Adresse livr : Code postal/Alpha/10/Vide
                ediWriter.write(separator);
                //23/Adresse livr : Code Pays/Alpha/3/Vide
                ediWriter.write(separator);
                //24/Code Ean acheteur/Alpha/13/Code client
                ediWriter.write(email.getClient().getCode() + separator);
                //25/Code Ean facturé/Alpha/13/Code client
                ediWriter.write(email.getClient().getCode() + separator);
                //26/Code Ean fournisseur/Alpha/13/Gamme
                ediWriter.write(rangeRow.getRange().getLabel() + separator);
                //27/Identifiant Unique
                ediWriter.write(rangeRow.getRange().getTopiaId() + "\n");

                ediWriter.write("%END_ENTETE_QUOTE\n");

            } catch (IOException ex) {
                throw new RuntimeException("Can't generate EDI file");
            }

        }
    }

    public List<Email> getEmailForFolder(MailFolder folder) {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        List<Email> result = new ArrayList<>(dao.forMailFolderEquals(folder)
                                                     .addNull(Email.PROPERTY_ARCHIVE_DATE)
                                                     .addNotEquals(Email.PROPERTY_DEMAND_STATUS, DemandStatus.ARCHIVED)
                                                     .setOrderByArguments(Email.PROPERTY_RECEPTION_DATE)
                                                     .findAll());
        return result;
    }

    public List<Email> getEmailForFolderAndSubfolders(MailFolder folder) {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        List<MailFolder> folders = new ArrayList<>();
        folders.addAll(getChildrenRecursively(folder));
        return new ArrayList<>(dao.forMailFolderIn(folders)
                                       .findAll());
    }

    protected List<MailFolder> getChildrenRecursively(MailFolder folder) {
        List<MailFolder> folders = new ArrayList<>();
        folders.add(folder);
        Collection<MailFolder> children = folder.getChildren();
        for (MailFolder child : children) {
            folders.addAll(getChildrenRecursively(child));
        }
        return folders;
    }

    public Email addToHistory(String emailId, HistoryType type, FaxToMailUser user, Date date, String... fields) {
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        Email email = emailDao.findByTopiaId(emailId);

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, type,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_FIELDS, Sets.newHashSet(fields),
                                            History.PROPERTY_MODIFICATION_DATE, date);

        email.addHistory(history);

        email = emailDao.update(email);
        getPersistenceContext().commit();

        return email;
    }

    public Email takeEmail(String emailId, FaxToMailUser user) {
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        Email email = emailDao.findByTopiaId(emailId);
        email.setTakenBy(user);

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.MODIFICATION,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_FIELDS, Sets.newHashSet(Email.PROPERTY_TAKEN_BY),
                                            History.PROPERTY_MODIFICATION_DATE, new Date());
        email.addHistory(history);

        email = emailDao.update(email);
        getPersistenceContext().commit();
        return email;
    }

    public List<Email> search(EmailFilter emailFilter) {

        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        List<Email> result = dao.search(emailFilter);
        return result;
    }

    public Email groupEmails(Email email1, Email email2, FaxToMailUser user) {
        EmailGroupTopiaDao groupDao = getPersistenceContext().getEmailGroupDao();

        EmailGroup group1 = email1.getEmailGroup();
        EmailGroup group2 = email2.getEmailGroup();

        // if both groups are null
        if (group1 == null && group2 == null) {
            EmailGroup group = groupDao.create(EmailGroup.PROPERTY_EMAIL, Lists.newArrayList(email1, email2));
            email1.setEmailGroup(group);
            email2.setEmailGroup(group);

        // if only group 1 is null
        } else if (group1 == null) {
            email1.setEmailGroup(group2);
            group2.addEmail(email1);
            groupDao.update(group2);

        // if only group 2 is null
        } else if (group2 == null) {
            email2.setEmailGroup(group1);
            group1.addEmail(email2);
            groupDao.update(group1);

        // if the groups are equals, do nothing
        } else if (group1.equals(group2)) {
           return email1;

        // if both groups exist, merge them
        } else {
            group1.addAllEmail(group2.getEmail());
            email2.setEmailGroup(group1);
            group2.clearEmail();
            groupDao.delete(group2);
            groupDao.update(group1);
        }

        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        Date now = new Date();

        email1.addHistory(historyDao.create(History.PROPERTY_TYPE, HistoryType.GROUP,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_FIELDS, Sets.newHashSet(email2.getObject()),
                                            History.PROPERTY_MODIFICATION_DATE, now));
        Email result = emailDao.update(email1);

        email2.addHistory(historyDao.create(History.PROPERTY_TYPE, HistoryType.GROUP,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_FIELDS, Sets.newHashSet(email1.getObject()),
                                            History.PROPERTY_MODIFICATION_DATE, now));
        emailDao.update(email2);

        getPersistenceContext().commit();

        return result;
    }

    public Email reply(String from, String to, String subject,
                       String content, Collection<AttachmentFile> attachments,
                       String originalEmailId, FaxToMailUser user) throws EmailException, MessagingException, IOException {

        Email email = getEmailById(originalEmailId);

        final String smtpUser = getApplicationConfig().getSmtpUser();
        final String password = getApplicationConfig().getSmtpPassword();
        final boolean useSsl = getApplicationConfig().getSmtpUseSsl();

        MultiPartEmail message = new MultiPartEmail();
        message.setHostName(getApplicationConfig().getSmtpHost());
        message.setSmtpPort(getApplicationConfig().getSmtpPort());
        if (StringUtils.isNotBlank(smtpUser) && password != null) {
            message.setAuthenticator(new DefaultAuthenticator(smtpUser, password));
        }
        message.setSSLOnConnect(useSsl);

        message.setCharset(EmailConstants.UTF_8);
        message.setFrom(from);
        message.addTo(to);
        message.setSubject(subject);
        message.setMsg(content);

        for (AttachmentFile attachmentFile : attachments) {
            // Create the attachment
            File file = attachmentFile.getFile();
            DataSource source = new FileDataSource(file);
            message.attach(source, attachmentFile.getFilename(), null);
        }

        String emailId = message.send();

        ReplyTopiaDao replyTopiaDao = getPersistenceContext().getReplyDao();
        Date now = new Date();

        StringBuilder emailSource = new StringBuilder();
        Enumeration<String> headerLines = message.getMimeMessage().getAllHeaderLines();
        while (headerLines.hasMoreElements()) {
            String headerLine = headerLines.nextElement();
            emailSource.append(headerLine).append("\n");
        }
        emailSource.append("\n").append(IOUtils.toString(message.getMimeMessage().getInputStream()));

        Reply reply = replyTopiaDao.create(Reply.PROPERTY_EMAIL_SOURCE, emailSource.toString(),
                                           Reply.PROPERTY_SENT_DATE, now,
                                           Reply.PROPERTY_SUBJECT, subject);

        email.addReplies(reply);

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.REPLY,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_MODIFICATION_DATE, now);
        email.addHistory(history);

        email = saveEmail(email, user);

        return email;
    }

    /**
     * Save content stream into attachment file content.
     * 
     * @param contentStream content
     * @return attachmentFile filled by content
     */
    public AttachmentFile getAttachmentFileFromStream(InputStream contentStream) {
        AttachmentFile attachmentFile = new AttachmentFileImpl();
        try {
            //Session hibernateSession = getPersistenceContext().getHibernateSupport().getHibernateSession();
            //Blob contentBlob = Hibernate.getLobCreator(hibernateSession).createBlob(contentStream, contentStream.available());
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            IOUtils.copy(contentStream, output);
            attachmentFile.setContent(output.toByteArray());
        } catch (Exception ex) {
            throw new RuntimeException("Can't save content", ex);
        }
        return attachmentFile;
    }
}
