/*******************************************************************************
 *
 * Pentaho Data Integration
 *
 * Copyright (C) 2002-2012 by Pentaho : http://www.pentaho.com
 *
 *******************************************************************************
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ******************************************************************************/

package org.pentaho.di.core.database;

import org.pentaho.di.core.Const;
import org.pentaho.di.core.row.ValueMetaInterface;

/**
 * Contains Generic Database Connection information through static final members 
 * 
 * @author Matt
 * @since  11-mrt-2005
 */

public class DerbyDatabaseMeta extends BaseDatabaseMeta implements DatabaseInterface
{
	@Override
  public int[] getAccessTypeList()
	{
		return new int[] { DatabaseMeta.TYPE_ACCESS_NATIVE, DatabaseMeta.TYPE_ACCESS_ODBC, DatabaseMeta.TYPE_ACCESS_JNDI };
	}
	
	/**
	 * @see DatabaseInterface#getNotFoundTK(boolean)
	 */
	@Override
  public int getNotFoundTK(boolean use_autoinc)
	{
		if ( supportsAutoInc() && use_autoinc)
		{
			return 0;
		}
		return super.getNotFoundTK(use_autoinc);
	}
	
	@Override
  public String getDriverClass()
	{
        if (getAccessType()==DatabaseMeta.TYPE_ACCESS_NATIVE)
        {
        	if (Const.isEmpty(getHostname())) {
        		return "org.apache.derby.jdbc.EmbeddedDriver";
        	}
        	else {
        		return "org.apache.derby.jdbc.ClientDriver";
        	}
        }
        else
        {
            return "sun.jdbc.odbc.JdbcOdbcDriver"; // ODBC bridge
        }

	}
    
    @Override
    public String getURL(String hostname, String port, String databaseName)
    {
        if (getAccessType()==DatabaseMeta.TYPE_ACCESS_NATIVE)
        {
            if (!Const.isEmpty(hostname))
            {
                String url="jdbc:derby://"+hostname;
                if (!Const.isEmpty(port)) url+=":"+port;
                url+="/"+databaseName;
                return url;
            }
            else // Simple format: jdbc:derby:<dbname>
            {
                return "jdbc:derby:"+databaseName;
            }
        }
        else
        {
            return "jdbc:odbc:"+databaseName;
        }
	}

	/**
	 * Checks whether or not the command setFetchSize() is supported by the JDBC driver...
	 * @return true is setFetchSize() is supported!
	 */
	@Override
  public boolean isFetchSizeSupported()
	{
		return true;
	}
	
	/**
	 * @return true if the database supports bitmap indexes
	 */
	@Override
  public boolean supportsBitmapIndex()
	{
		return false;
	}
	
	/**
	 * @return true if Kettle can create a repository on this type of database.
	 */
	@Override
  public boolean supportsRepository()
	{
		return true;
	}
	
	/**
	 * @param tableName The table to be truncated.
	 * @return The SQL statement to truncate a table: remove all rows from it without a transaction
	 */
	@Override
  public String getTruncateTableStatement(String tableName)
	{
	    return "DELETE FROM "+tableName;
	}


	/**
	 * Generates the SQL statement to add a column to the specified table
	 * For this generic type, i set it to the most common possibility.
	 * 
	 * @param tablename The table to add
	 * @param v The column defined as a value
	 * @param tk the name of the technical key field
	 * @param use_autoinc whether or not this field uses auto increment
	 * @param pk the name of the primary key field
	 * @param semicolon whether or not to add a semi-colon behind the statement.
	 * @return the SQL statement to add a column to the specified table
	 */
	@Override
  public String getAddColumnStatement(String tablename, ValueMetaInterface v, String tk, boolean use_autoinc, String pk, boolean semicolon)
	{
		return "ALTER TABLE "+tablename+" ADD "+getFieldDefinition(v, tk, pk, use_autoinc, true, false);
	}

	/**
	 * Generates the SQL statement to modify a column in the specified table
	 * @param tablename The table to add
	 * @param v The column defined as a value
	 * @param tk the name of the technical key field
	 * @param use_autoinc whether or not this field uses auto increment
	 * @param pk the name of the primary key field
	 * @param semicolon whether or not to add a semi-colon behind the statement.
	 * @return the SQL statement to modify a column in the specified table
	 */
	@Override
  public String getModifyColumnStatement(String tablename, ValueMetaInterface v, String tk, boolean use_autoinc, String pk, boolean semicolon)
	{
		return "ALTER TABLE "+tablename+" ALTER "+getFieldDefinition(v, tk, pk, use_autoinc, true, false);
	}

	@Override
  public String getFieldDefinition(ValueMetaInterface v, String tk, String pk, boolean use_autoinc, boolean add_fieldname, boolean add_cr)
	{
		String retval="";
		
		String fieldname = v.getName();
		int    length    = v.getLength();
		int    precision = v.getPrecision();
		
		if (add_fieldname) retval+=fieldname+" ";
		
		int type         = v.getType();
		switch(type)
		{
		case ValueMetaInterface.TYPE_DATE   : retval+="TIMESTAMP"; break;
		case ValueMetaInterface.TYPE_BOOLEAN: retval+="CHAR(1)"; break;
        case ValueMetaInterface.TYPE_NUMBER    :
        case ValueMetaInterface.TYPE_INTEGER   : 
        case ValueMetaInterface.TYPE_BIGNUMBER : 
            if (fieldname.equalsIgnoreCase(tk) || // Technical key
                fieldname.equalsIgnoreCase(pk)    // Primary key
                ) 
            {
                if (use_autoinc)
                {
                    retval+="BIGINT NOT NULL GENERATED ALWAYS AS IDENTITY (START WITH 0, INCREMENT BY 1)";
                }
                else
                {
                    retval+="BIGINT NOT NULL PRIMARY KEY";
                }
            } 
            else
            {
                // Integer values...
                if (precision==0)
                {
                    if (length>9)
                    {
                        retval+="BIGINT";
                    }
                    else
                    {
                        if (length>4)
                        {
                            retval+="INTEGER";
                        }
                        else
                        {
                            retval+="SMALLINT";
                        }
                    }
                }
                // Floating point values...
                else  
                {
                    if (length>18)
                    {
                        retval+="DECIMAL("+length;
                        if (precision>0) retval+=", "+precision;
                        retval+=")";
                    }
                    else
                    {
                        retval+="FLOAT";
                    }
                }
            }
            break;
		case ValueMetaInterface.TYPE_STRING:
			if (length>=DatabaseMeta.CLOB_LENGTH || length>32700)
			{
				retval+="CLOB";
			}
			else
			{
				retval+="VARCHAR";
				if (length>0)
				{
					retval+="("+length;
				}
				else
				{
					retval+="("; // Maybe use some default DB String length?
				}
				retval+=")";
			}
			break;
        case ValueMetaInterface.TYPE_BINARY:
            retval+="BLOB";
            break;
		default:
			retval+="UNKNOWN";
			break;
		}
		
		if (add_cr) retval+=Const.CR;
		
		return retval;
	}

    @Override
    public String[] getUsedLibraries()
    {
        return new String[] { "derbyclient.jar" };
    }
    
    @Override
    public int getDefaultDatabasePort()
    {
        return 1527;
    }
    
    @Override
    public boolean supportsGetBlob()
    {
        return false;
    }
    
    @Override
    public String getExtraOptionsHelpText()
    {
        return "http://db.apache.org/derby/papers/DerbyClientSpec.html";
    }
    
    @Override
    public String[] getReservedWords() {
    	return new String[] {
    		      "ADD"
    		    , "ALL"
    		    , "ALLOCATE"
    		    , "ALTER"
    		    , "AND"
    		    , "ANY"
    		    , "ARE"
    		    , "AS"
    		    , "ASC"
    		    , "ASSERTION"
    		    , "AT"
    		    , "AUTHORIZATION"
    		    , "AVG"
    		    , "BEGIN"
    		    , "BETWEEN"
    		    , "BIT"
    		    , "BOOLEAN"
    		    , "BOTH"
    		    , "BY"
    		    , "CALL"
    		    , "CASCADE"
    		    , "CASCADED"
    		    , "CASE"
    		    , "CAST"
    		    , "CHAR"
    		    , "CHARACTER"
    		    , "CHECK"
    		    , "CLOSE"
    		    , "COLLATE"
    		    , "COLLATION"
    		    , "COLUMN"
    		    , "COMMIT"
    		    , "CONNECT"
    		    , "CONNECTION"
    		    , "CONSTRAINT"
    		    , "CONSTRAINTS"
    		    , "CONTINUE"
    		    , "CONVERT"
    		    , "CORRESPONDING"
    		    , "COUNT"
    		    , "CREATE"
    		    , "CURRENT"
    		    , "CURRENT_DATE"
    		    , "CURRENT_TIME"
    		    , "CURRENT_TIMESTAMP"
    		    , "CURRENT_USER"
    		    , "CURSOR"
    		    , "DEALLOCATE"
    		    , "DEC"
    		    , "DECIMAL"
    		    , "DECLARE"
    		    , "DEFERRABLE"
    		    , "DEFERRED"
    		    , "DELETE"
    		    , "DESC"
    		    , "DESCRIBE"
    		    , "DIAGNOSTICS"
    		    , "DISCONNECT"
    		    , "DISTINCT"
    		    , "DOUBLE"
    		    , "DROP"
    		    , "ELSE"
    		    , "END"
    		    , "ENDEXEC"
    		    , "ESCAPE"
    		    , "EXCEPT"
    		    , "EXCEPTION"
    		    , "EXEC"
    		    , "EXECUTE"
    		    , "EXISTS"
    		    , "EXPLAIN"
    		    , "EXTERNAL"
    		    , "FALSE"
    		    , "FETCH"
    		    , "FIRST"
    		    , "FLOAT"
    		    , "FOR"
    		    , "FOREIGN"
    		    , "FOUND"
    		    , "FROM"
    		    , "FULL"
    		    , "FUNCTION"
    		    , "GET"
    		    , "GET_CURRENT_CONNECTION"
    		    , "GLOBAL"
    		    , "GO"
    		    , "GOTO"
    		    , "GRANT"
    		    , "GROUP"
    		    , "HAVING"
    		    , "HOUR"
    		    , "IDENTITY"
    		    , "IMMEDIATE"
    		    , "IN"
    		    , "INDICATOR"
    		    , "INITIALLY"
    		    , "INNER"
    		    , "INOUT"
    		    , "INPUT"
    		    , "INSENSITIVE"
    		    , "INSERT"
    		    , "INT"
    		    , "INTEGER"
    		    , "INTERSECT"
    		    , "INTO"
    		    , "IS"
    		    , "ISOLATION"
    		    , "JOIN"
    		    , "KEY"
    		    , "LAST"
    		    , "LEFT"
    		    , "LIKE"
    		    , "LONGINT"
    		    , "LOWER"
    		    , "LTRIM"
    		    , "MATCH"
    		    , "MAX"
    		    , "MIN"
    		    , "MINUTE"
    		    , "NATIONAL"
    		    , "NATURAL"
    		    , "NCHAR"
    		    , "NVARCHAR"
    		    , "NEXT"
    		    , "NO"
    		    , "NOT"
    		    , "NULL"
    		    , "NULLIF"
    		    , "NUMERIC"
    		    , "OF"
    		    , "ON"
    		    , "ONLY"
    		    , "OPEN"
    		    , "OPTION"
    		    , "OR"
    		    , "ORDER"
    		    , "OUT"
    		    , "OUTER"
    		    , "OUTPUT"
    		    , "OVERLAPS"
    		    , "PAD"
    		    , "PARTIAL"
    		    , "PREPARE"
    		    , "PRESERVE"
    		    , "PRIMARY"
    		    , "PRIOR"
    		    , "PRIVILEGES"
    		    , "PROCEDURE"
    		    , "PUBLIC"
    		    , "READ"
    		    , "REAL"
    		    , "REFERENCES"
    		    , "RELATIVE"
    		    , "RESTRICT"
    		    , "REVOKE"
    		    , "RIGHT"
    		    , "ROLLBACK"
    		    , "ROWS"
    		    , "RTRIM"
    		    , "SCHEMA"
    		    , "SCROLL"
    		    , "SECOND"
    		    , "SELECT"
    		    , "SESSION_USER"
    		    , "SET"
    		    , "SMALLINT"
    		    , "SOME"
    		    , "SPACE"
    		    , "SQL"
    		    , "SQLCODE"
    		    , "SQLERROR"
    		    , "SQLSTATE"
    		    , "SUBSTR"
    		    , "SUBSTRING"
    		    , "SUM"
    		    , "SYSTEM_USER"
    		    , "TABLE"
    		    , "TEMPORARY"
    		    , "TIMEZONE_HOUR"
    		    , "TIMEZONE_MINUTE"
    		    , "TO"
    		    , "TRAILING"
    		    , "TRANSACTION"
    		    , "TRANSLATE"
    		    , "TRANSLATION"
    		    , "TRUE"
    		    , "UNION"
    		    , "UNIQUE"
    		    , "UNKNOWN"
    		    , "UPDATE"
    		    , "UPPER"
    		    , "USER"
    		    , "USING"
    		    , "VALUES"
    		    , "VARCHAR"
    		    , "VARYING"
    		    , "VIEW"
    		    , "WHENEVER"
    		    , "WHERE"
    		    , "WITH"
    		    , "WORK"
    		    , "WRITE"
    		    , "XML"
    		    , "XMLEXISTS"
    		    , "XMLPARSE"
    		    , "XMLSERIALIZE"
    		    , "YEAR"
    	};
    }

	/**
	 * Get the SQL to insert a new empty unknown record in a dimension.
	 * @param schemaTable the schema-table name to insert into
	 * @param keyField The key field
	 * @param versionField the version field
	 * @return the SQL to insert the unknown record into the SCD.
	 */
	@Override
  public String getSQLInsertAutoIncUnknownDimensionRow(String schemaTable, String keyField, String versionField) {
		return "insert into "+schemaTable+"("+versionField+") values (1)";		
	}

}
