/*
 * *##% 
 * JAXX Action
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.util.config;

import org.nuiton.jaxx.util.DialogUIModel;
import org.nuiton.util.config.Config;

import java.util.EnumSet;

/**
 * Abstract config ui model.
 *
 * @author chemit
 */
public abstract class DialogConfigUIModel<E extends Enum<E>, C extends Config<E>> extends DialogUIModel {

    public static final String CONFIG_PROPERTY_CHANGED = "config";
    public static final String MODIFIED_PROPERTY_CHANGED = "modify";
    public static final String UNVALID_PROPERTY_CHANGED = "unvalid";

    /** @return a empty config */
    protected abstract C newConfig();

    /**
     * @param key   property key
     * @param value value to validate
     * @return <code>true</code> if given value is valid for property, <code>false>/code> otherwise
     */
    protected abstract boolean isValid(E key, Object value);

    /**
     * object used to init model and save model, this is an external object.
     * <p/>
     * The object must have bean read-write properties for each value of E
     */
    protected Object src;

    /** current config used in model */
    protected C current;

    /** set of modified properties */
    protected EnumSet<E> modifieds;

    /** set of unvalid properties */
    protected EnumSet<E> unvalids;

    /** enum class */
    protected Class<E> klass;

    /** set of key not to check */
    protected EnumSet<E> uncheckedKeys;

    /** set of all keys checkable */
    protected EnumSet<E> checkedKeysSet;

    protected DialogConfigUIModel(Class<E> klass) {
        this.klass = klass;
        this.current = newConfig();
        this.modifieds = EnumSet.noneOf(klass);
        this.unvalids = EnumSet.noneOf(klass);
    }

    public EnumSet<E> getCheckedKeysSet() {
        if (checkedKeysSet == null) {
            if (uncheckedKeys != null) {
                checkedKeysSet = EnumSet.complementOf(uncheckedKeys);
            } else {
                checkedKeysSet = EnumSet.allOf(klass);
            }
        }
        return checkedKeysSet;
    }

    public Object getSrc() {
        return src;
    }

    public C getCurrent() {
        return current;
    }

    public EnumSet<E> getUnivserse() {
        return getCurrent().getUniverse();
    }

    public EnumSet<E> getModifieds() {
        return modifieds;
    }

    public EnumSet<E> getUnvalids() {
        return unvalids;
    }

    public boolean isModified() {
        return !modifieds.isEmpty();
    }

    public boolean isConfigValid() {
        return unvalids.isEmpty();
    }

    public void populate(Object src) {
        this.src = src;
        this.current = newConfig();
        if (src != null) {
            this.current.copyFrom(src);
        } 
        this.modifieds.clear();
        this.unvalids.clear();
        setModified(false);
        firePropertyChange(CONFIG_PROPERTY_CHANGED, null, this);
    }

    public void reset() {
        populate(src);
    }

    public void setModified(boolean modified) {
        firePropertyChange(MODIFIED_PROPERTY_CHANGED, null, modified);
    }

    public void setUnvalid(boolean unvalid) {
        firePropertyChange(UNVALID_PROPERTY_CHANGED, null, unvalid);
    }

    public void addModified(E key) {
        if (!modifieds.contains(key)) {
            modifieds.add(key);
            log.debug(key);
        }
        setModified(!modifieds.isEmpty());
    }

    public void removeModified(E key) {

        if (modifieds.contains(key)) {
            modifieds.remove(key);
        }
        setModified(!modifieds.isEmpty());
    }

    public void removeModified(EnumSet<E> keys) {
        for (E key : keys) {
            if (modifieds.contains(key)) {
                modifieds.remove(key);
            }
        }
        setModified(!modifieds.isEmpty());
    }

    public void setUnvalids(EnumSet<E> keys) {
        for (E key : keys) {
            if (!unvalids.contains(key)) {
                unvalids.add(key);
            }
        }
        for (E key : EnumSet.complementOf(keys)) {
            if (unvalids.contains(key)) {
                unvalids.remove(key);
            }
        }
        setUnvalid(!unvalids.isEmpty());
    }


    public void addUnvalid(E key) {
        if (!unvalids.contains(key)) {
            unvalids.add(key);
        }
        setUnvalid(!unvalids.isEmpty());
    }

    public void removeUnvalid(E key) {
        if (unvalids.contains(key)) {
            unvalids.remove(key);
        }
        setUnvalid(!unvalids.isEmpty());
    }

    public void save() {
        current.copyTo(src, modifieds);
        modifieds.clear();
        // redisplay config
        firePropertyChange(CONFIG_PROPERTY_CHANGED, null, this);
    }

    public void changeModifiedState(E key, Object uiValue, Object currentValue) {
        if (uiValue!=null && uiValue.equals(currentValue)) {
            removeModified(key);
        } else {
            addModified(key);
        }
    }

    public void clear(E key) {
        log.info(key);
        modifieds.remove(key);
        unvalids.remove(key);
    }

    protected void validateProperty(E key, Object uiValue) {
        if (isValid(key, uiValue)) {
            removeUnvalid(key);
        } else {
            addUnvalid(key);
        }
    }
}