/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: NavigationTreeTableNode.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.0.2/jaxx-runtime/src/main/java/jaxx/runtime/swing/navigation/treetable/NavigationTreeTableNode.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.navigation.treetable;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.swing.navigation.NavigationNode;
import jaxx.runtime.swing.navigation.NavigationNodeRenderer;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModelBuilder;
import jaxx.runtime.swing.navigation.tree.NavigationTreeNode;
import org.apache.commons.jxpath.JXPathContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.treetable.DefaultMutableTreeTableNode;

import javax.swing.tree.TreeNode;
import java.util.Enumeration;

/**
 * Node of the {@link NavigationTreeTableModel}.
 *
 * @author sletellier
 * @see NavigationTreeNode
 * @since 2.0.0
 */
public class NavigationTreeTableNode extends DefaultMutableTreeTableNode implements NavigationNode<NavigationTreeTableNode> {

    private static final long serialVersionUID = -1L;

    /** Logger */
    private static final Log log =
            LogFactory.getLog(NavigationTreeTableNode.class);

    /**
     * The path separator used to build the {@link #fullPath}.
     *
     * @see #path
     * @see #fullPath
     */
    protected final String pathSeparator;

    /**
     * The node path.
     * <p/>
     * Used to build the {@link #fullPath} which gives a unique identifier of
     * the node.
     *
     * @see #pathSeparator
     * @see #fullPath
     */
    protected String path;

    /**
     * The full path for the node from the rood node.
     * <p/>
     * This path is build by appending nodes {@link #path} from the root node to
     * this node, between each path we introduce the {@link #pathSeparator}.
     * <p/>
     * Exemple :
     * <pre>
     * root
     *   $root
     *    `-- child1
     *         `-- child2
     * </pre>
     * will given {@code $root/child1/child2}.
     *
     * @see #pathSeparator
     * @see #path
     */
    protected String fullPath;

    /**
     * The UI class associated with the node.
     * <p/>
     * This class can be {@code null}, in that case, the {@link
     * NavigationTreeModelBuilder#defaultUIClass} will be used while building
     * the model.
     */
    protected Class<? extends JAXXObject> uIClass;

    /**
     * The UI handler class associated with the node.
     * <p/>
     * This class can be {@code null}, in that case, the {@link
     * NavigationTreeModelBuilder#defaultUIHandlerClass} will be used while
     * building the model.
     */
    protected Class<? extends JAXXAction> uIHandlerClass;

    /**
     * The context entry definition to retrieve the bean.
     * <p/>
     * <b>Note:</b> If not set - the {@code bean} will be retrieve on a ancestor
     * node.
     */
    protected JAXXContextEntryDef<?> jaxxContextEntryDef;

    /**
     * The jxPath to process to obtain real {@code bean} from the data retrieve
     * in the context.
     * <p/>
     * <b>Note:</b> If not set -no jxpath will be apply on the bean from
     * context.
     */
    protected String jaxxContextEntryPath;

    /**
     * The bean associated with the node ( the value can be obtain via the
     * method {@link #getBean(JAXXContext)} or directly set via method {@link
     * #setBean(Object)}.
     * <p/>
     * cache of bean associated with bean to improve performance
     */
    protected transient Object bean;

    /**
     * The type of the related bean associated with the node.
     * <p/>
     * Note: This type is here to override the NodeRenderer internalClass, since
     * we could need to override this data.
     * <p/>
     * If this property is let to null, then we will use the NodeRenderer one
     */
    protected Class<?> internalClass;

    public NavigationTreeTableNode(String pathSeparator,
                                   String navigationPath,
                                   Object jaxxContextEntryDef) {
        this.pathSeparator = pathSeparator;
        path = navigationPath;
        if (jaxxContextEntryDef instanceof JAXXContextEntryDef<?>) {
            this.jaxxContextEntryDef =
                    (JAXXContextEntryDef<?>) jaxxContextEntryDef;
        } else if (jaxxContextEntryDef instanceof String) {
            jaxxContextEntryPath = (String) jaxxContextEntryDef;
        } else if (jaxxContextEntryDef != null) {
            // wrong context definition type
            throw new IllegalArgumentException(
                    "to define a context link, must be a String (jxpath) or" +
                    " a " + JAXXContextEntryDef.class + ", but was " +
                    jaxxContextEntryDef);
        }
    }

    public NavigationTreeTableNode(String pathSeparator,
                                   String navigationPath,
                                   JAXXContextEntryDef<?> jaxxContextEntryDef,
                                   String jaxxContextEntryPath) {
        this.pathSeparator = pathSeparator;
        path = navigationPath;
        this.jaxxContextEntryDef = jaxxContextEntryDef;
        this.jaxxContextEntryPath = jaxxContextEntryPath;
    }

    /** @return the text node renderer (store in {@link #userObject} property. */
    @Override
    public NavigationNodeRenderer getRenderer() {
        NavigationNodeRenderer render = null;
        Object o = getUserObject();
        if (o != null && o instanceof NavigationNodeRenderer) {
            render = (NavigationNodeRenderer) o;
        }
        return render;
    }

    @Override
    public void setRenderer(NavigationNodeRenderer renderer) {
        // clear all cache
        bean = null;
        setUserObject(renderer);
    }

    @Override
    public String getPathSeparator() {
        return pathSeparator;
    }

    @Override
    public String getNodePath() {
        return path;
    }

    @Override
    public void setNodePath(String navigationPath) {
        path = navigationPath;
    }

    @Override
    public Class<? extends JAXXObject> getUIClass() {
        return uIClass;
    }

    @Override
    public void setUIClass(Class<? extends JAXXObject> uIClass) {
        this.uIClass = uIClass;
    }

    @Override
    public void setInternalClass(Class<?> internalClass) {
        this.internalClass = internalClass;
    }

    @Override
    public Class<? extends JAXXAction> getUIHandlerClass() {
        return uIHandlerClass;
    }

    @Override
    public void setUIHandlerClass(Class<? extends JAXXAction> uIHandlerClass) {
        this.uIHandlerClass = uIHandlerClass;
    }

    @Override
    public JAXXContextEntryDef<?> getJaxxContextEntryDef() {
        return jaxxContextEntryDef;
    }

    @Override
    public void setJaxxContextEntryDef(JAXXContextEntryDef<?> jaxxContextEntryDef) {
        this.jaxxContextEntryDef = jaxxContextEntryDef;
    }

    @Override
    public String getJaxxContextEntryPath() {
        return jaxxContextEntryPath;
    }

    @Override
    public void setJaxxContextEntryPath(String jaxxContextEntryPath) {
        this.jaxxContextEntryPath = jaxxContextEntryPath;
    }

    @Override
    public Class<?> getInternalClass() {
        if (internalClass == null && getRenderer() != null) {
            return getRenderer().getInternalClass();
        }
        return internalClass;
    }

    /** @return the fully context path of the node from the root node to this. */
    @Override
    public String getFullPath() {
        if (fullPath == null) {
            StringBuilder sb = new StringBuilder();
            for (TreeNode treeNode : getPath()) {
                NavigationTreeTableNode myNode =
                        (NavigationTreeTableNode) treeNode;
                sb.append(pathSeparator).append(myNode.getNodePath());
            }
            fullPath = sb.substring(1);
        }
        return fullPath;
    }

    /**
     * @return the first ancestor with a none null {@link #jaxxContextEntryDef}
     *         or <code>null</code> if none find..
     */
    protected NavigationTreeTableNode getFirstAncestorWithDef() {
        if (jaxxContextEntryDef != null) {
            // find a node with a direct link with the context
            return this;
        }
        // the node is not linked to context
        // seek in his parent
        NavigationTreeTableNode ancestor = getParent();
        return ancestor == null ? null : ancestor.getFirstAncestorWithDef();
    }

    protected String computeJXPath(String expr,
                                   NavigationTreeTableNode parentNode) {
        if (equals(parentNode)) {
            // reach the parent limit node, return the expr computed
            return expr;
        }
        int firstIndex = expr.indexOf("..");
        int lastIndex = expr.lastIndexOf("..");
        if (firstIndex == -1) {
            // this is a error, since current node is not parent limit node,
            // we must find somewhere a way to go up in nodes
            throw new IllegalArgumentException(
                    expr + " should contains at least one \"..\"");
        }
        if (firstIndex != 0) {
            // this is a error, the ../ must be at the beginning of the
            // expression
            throw new IllegalArgumentException(
                    "\"..\" must be at the beginning but was : " + expr);
        }
        NavigationTreeTableNode ancestor = getParent();
        if (firstIndex == lastIndex) {
            // found only one go up, so must be substitute by the parent node
            // context
            String newExpr = expr.substring(2);
            //String newExpr = expr.substring(expr.startsWith("../") ? 3 : 2);
            if (getParent().equals(parentNode)) {
                // parent node is the final parent node, so no substitution
                // needed
                return newExpr;
            }
            // ancestor must have a jaxxContextEntryPath
            if (ancestor.jaxxContextEntryPath == null) {
                throw new IllegalArgumentException(
                        "with the expression " + expr + ", the ancestor node ("
                        + ancestor + ") must have a jaxxContextEntryPath " +
                        "definition, but was not ");
            }
            newExpr = ancestor.jaxxContextEntryPath + newExpr;
            return ancestor.computeJXPath(newExpr, parentNode);
        }
        // have more than one go up, so the ancestor node can not have a
        // jaxxContextEntryPath
        if (ancestor.jaxxContextEntryPath != null) {
            throw new IllegalArgumentException(
                    "with the expression " + expr + ", the ancestor node can " +
                    "not have a jaxxContextEntryPath definition");
        }
        // substitute the last ..[/] and delegate to ancestor
        String newExpr = expr.substring(0, lastIndex - 1) +
                         expr.substring(
                                 lastIndex +
                                 (expr.charAt(lastIndex + 3) == '/' ? 3 : 2)
                         );
        return ancestor.computeJXPath(newExpr, parentNode);
    }


    @Override
    public Object getBean() {
        return bean;
    }

    @Override
    public void setBean(Object bean) {
        this.bean = bean;
    }

    @Override
    public void reload(JAXXContext context) {

        // clear bean cache
        bean = null;

        // clear context navigation cache
        fullPath = null;

        NavigationNodeRenderer renderer = getRenderer();
        if (renderer == null) {
            // this can't be !
            throw new NullPointerException(
                    "could not find the renderer for node " + this);
        }

        String s = getFullPath();
        Object b = null;
        try {
            b = getBean(context);
        } finally {
            if (log.isInfoEnabled()) {
                log.info("bean for path [" + s + "]  = " + bean);
            }
        }
        renderer.reload(b);
    }

    /**
     * Obtain the associated bean value from context corresponding to node
     *
     * @param context the context to seek
     * @return the value associated in context with the given context path
     */
    @Override
    public Object getBean(JAXXContext context) {
        if (bean != null) {
            // use cached bean
            return bean;
        }
        Object result;
        if (getJaxxContextEntryDef() != null && jaxxContextEntryPath == null) {
            // the node maps directly a value in context, with no jxpath
            // resolving
            result = getJaxxContextEntryDef().getContextValue(context);
            // save in cache
            setBean(result);
            return result;
        }
        // find the first ancestor node with a context def
        NavigationTreeTableNode parentNode = getFirstAncestorWithDef();
        if (parentNode == null) {
            log.warn("could not find a ancestor node with a definition of" +
                     " a context entry from node (" + this + ")");
            // todo must be an error
            // no parent found
            return null;
        }
        Object parentBean =
                parentNode.getJaxxContextEntryDef().getContextValue(context);
        if (parentBean == null) {
            // must be an error no bean found
            log.warn("could not find a bean attached in context from context" +
                     " entry definition " + parentNode.getJaxxContextEntryDef());
            return null;
        }
        if (parentNode.jaxxContextEntryPath != null) {
            // apply the jxpath on parentBean
            JXPathContext jxcontext = JXPathContext.newContext(parentBean);
            parentBean = jxcontext.getValue(parentNode.jaxxContextEntryPath);
        }
        // save in cache
        parentNode.setBean(parentBean);
        if (parentNode.equals(this)) {
            // current node is the node matching the context entry value and
            //  no jxpath is found
            return parentBean;
        }
        if (jaxxContextEntryPath == null) {
            // todo must be an error
            log.warn("must find a jaxxContextEntryPath on node (" + this +
                     ")");
            return null;
        }
        String jxpathExpression = computeJXPath(jaxxContextEntryPath,
                                                parentNode);
        if (jxpathExpression == null) {
            /// todo must be an error
            log.warn("could not build jxpath from node " + parentNode +
                     " to " + this);
            // could not retreave the jxpath...
            return null;
        }
        if (jxpathExpression.startsWith("[")) {
            // special case when we want to access a collection
            jxpathExpression = '.' + jxpathExpression;
        }
        if (log.isDebugEnabled()) {
            log.debug("jxpath : " + jxpathExpression);
        }
        JXPathContext jxcontext = JXPathContext.newContext(parentBean);
        result = jxcontext.getValue(jxpathExpression);
        // save in cache
        setBean(result);
        return result;
    }


    @Override
    public NavigationTreeTableNode getChildAt(int index) {
        return (NavigationTreeTableNode) super.getChildAt(index);
    }

    @Override
    public NavigationTreeTableNode getParent() {
        return (NavigationTreeTableNode) super.getParent();
    }

    @Override
    public NavigationTreeTableNode getChild(String path) {
        Enumeration<?> childs = children();
        while (childs.hasMoreElements()) {
            NavigationTreeTableNode son =
                    (NavigationTreeTableNode) childs.nextElement();
            if (path.equals(son.getNodePath())) {
                return son;
            }
        }
        return null;
    }

    @Override
    public Enumeration<? extends NavigationTreeTableNode> children() {
        return (Enumeration<? extends NavigationTreeTableNode>) super.children();
    }

    @Override
    public void insert(NavigationTreeTableNode child, int index) {
        super.insert(child, index);
    }

    @Override
    public void remove(NavigationTreeTableNode node) {
        super.remove(node);
    }

    @Override
    public void setParent(NavigationTreeTableNode newParent) {
        super.setParent(newParent);
    }

    @Override
    public int getIndex(NavigationTreeTableNode node) {
        return super.getIndex(node);
    }

    @Override
    public void add(NavigationTreeTableNode node) {
        super.add(node);
    }

    @Override
    public NavigationTreeTableNode[] getPath() {
        return getPathToRoot(this, 0);
    }

    @Override
    public NavigationTreeTableNode[] getPathToRoot(
            NavigationTreeTableNode aNode, int depth) {
        NavigationTreeTableNode[] retNodes;

        /* Check for null, in case someone passed in a null node, or
         they passed in an element that isn't rooted at root. */
        if (aNode == null) {
            if (depth == 0) {
                return null;
            } else {
                retNodes = new NavigationTreeTableNode[depth];
            }
        } else {
            depth++;
            retNodes = getPathToRoot(aNode.getParent(), depth);
            retNodes[retNodes.length - depth] = aNode;
        }
        return retNodes;
    }

    @Override
    public boolean isRoot() {
        return getParent() == null;
    }

}
