/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: NavigationMultiTreeHandler.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.0.2/jaxx-runtime/src/main/java/jaxx/runtime/swing/navigation/handler/NavigationMultiTreeHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.navigation.handler;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.navigation.NavigationContextHelper;
import jaxx.runtime.swing.navigation.NavigationMultiContentUI;
import jaxx.runtime.swing.navigation.NavigationNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.event.TreeSelectionEvent;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.util.ArrayList;
import java.util.List;

/**
 * The handler of a navigation tree.
 * <p/>
 * This is also the selection model to use, since we must check before moving
 * from nodes we can not just listen selection model changed, we must control
 * it.
 * <p/>
 * This one is used to enable multi selection
 *
 * @author sletellier
 * @since 2.0.1
 */
public abstract class NavigationMultiTreeHandler<E extends NavigationNode<E>> extends AbstractNavigationHandler<E> {

    /** Logger */
    static private final Log log =
            LogFactory.getLog(NavigationMultiTreeHandler.class);


    public NavigationMultiTreeHandler(String contextPrefix,
                                      JAXXObject context,
                                      Strategy strategy) {
        super(contextPrefix, context, strategy);

        // Enable multi selection
        setSelectionMode(DISCONTIGUOUS_TREE_SELECTION);
    }

    @Override
    public void valueChanged(TreeSelectionEvent event) {
        TreePath[] paths = event.getPaths();

        // TODO : verifier que la selection n'est pas la même
        List<E> nodes = new ArrayList<E>();
        if (paths == null) {
            selectNodeUI(new ArrayList<E>());
            return;
        }
        for (TreePath path : paths) {
            E node = (E) path.getLastPathComponent();
            nodes.add(node);
            if (log.isDebugEnabled()) {
                log.debug("Adding path : " + path);
                log.debug("As node : " + node.getFullPath());
            }
        }
        selectNodeUI(nodes);
    }

    /**
     * Ouvre l'ui associée aux noeuds sélectionnés dans l'arbre de navigation.
     *
     * @param newUI l'ui associé au noeud sélectionné à ouvrir
     * @param nodes les node de l'ui a ouvrir
     * @throws Exception if any
     */
    protected abstract void openUI(Component newUI, List<E> nodes)
            throws Exception;

    /**
     * Instancie une nouvelle ui associé à des noeuds de l'arbre de navigation
     *
     * @param nodes les noeuds associés à l'ui à créer
     * @return la nouvelle ui associée au noeud
     * @throws Exception if any
     */
    protected abstract Component createUI(List<E> nodes)
            throws Exception;

    /**
     * @param nodes les noeuds associés à l'ui à retrouver
     * @return l'ui associés au nouveau noeud sélectionné
     */
    protected abstract Component getUI(List<E> nodes);

    protected void selectNodeUI(List<E> nodes) {
        log.info("select nodes " + nodes);

        try {
//            List<String> paths = new ArrayList<String>();
            List<Object> beans = new ArrayList<Object>();
            for (E node : nodes) {

                String path = node.getFullPath();
//                paths.add(path);

                if (log.isTraceEnabled()) {
                    log.trace(path);
                }
                // now, we are free to open the ui associated with the selected
                // node in navigation

                // get the bean associated with the node
                beans.add(getNavigationTreeModel().getBean(path));
            }

            Component newUI = getUI(nodes);

            // save it in context (must be done before init ui)
            JAXXContext ctxt = getContext();
            NavigationContextHelper<E> helper = getContextHelper();

            // remove previous selected bean
            //TODO-TC-20091004 should have an automatic clean context method
            //TODO-TC-20091004 while
            helper.setSelectedBean(ctxt, null);

//            if (beans != null) {
            helper.setSelectedBeans(ctxt, beans);
//            }

            if (newUI == null) {

                // a new ui instance is required
                newUI = createUI(nodes);
            }

            // save in context current node context path
            helper.setSelectedPaths(ctxt, getPaths(nodes));

            // save in context current node
            helper.setSelectedNodes(ctxt, nodes);

            // save in context current ui
//            helper.setSelectedUI(ctxt, newUI);

            // really open the ui associated with the selected node
            // init ui before to be visible
            if (newUI instanceof NavigationMultiContentUI<?>) {
                ((NavigationMultiContentUI<E>) newUI).openUI(nodes);
            }

            // set ui in content
            openUI(newUI, nodes);

        } catch (Exception e) {
            // remove data from context

            // if any error, go back to previvous node
            treateError(e);
        }
    }

    protected List<String> getPaths(List<E> nodes) {
        List<String> result = new ArrayList<String>();
        for (E node : nodes) {
            result.add(node.getFullPath());
        }
        return result;
    }

}
