package jaxx.runtime.swing.navigation;


import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXContextEntryDef;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.navigation.NavigationUtil.NodeRenderer;
import org.apache.commons.jxpath.JXPathContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Le modele utilisé pour un arbre de navigation.
 * <p/>
 * Il est composé de {@link NavigationTreeModel.NavigationTreeNode}
 *
 * @author chemit
 */
public class NavigationTreeModel extends DefaultTreeModel {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(NavigationTreeModel.class);

    static private final long serialVersionUID = 1L;

    /** the separator char used to produce the navigation path of a node. */
    protected final String navigationPathSeparator;

    public NavigationTreeModel(TreeNode root, String navigationPathSeparator) {
        super(root);
        this.navigationPathSeparator = navigationPathSeparator;
    }

    @Override
    public NavigationTreeNode getRoot() {
        return (NavigationTreeNode) super.getRoot();
    }

    /**
     * Search from the root node a node named by his fully path (concatenation of nodes
     * {@link NavigationTreeNode#navigationPath} valued separated by dot.
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path the fully path of the searched node.
     * @return the node matching the fully context from the root node, or <code>null</code> if not find.
     */
    public NavigationTreeNode findNode(String path) {
        return findNode(getRoot(), path, (Pattern) null);
    }

    /**
     * Apply first the regex pattern to obtain the searched node fi the given <code>regex</code> is not null.
     * <p/>
     * Search then from the root node a node named by his fully path (concatenation of nodes
     * {@link NavigationTreeNode#navigationPath} valued separated by {@link #navigationPathSeparator}.
     * <p/>
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path  the fully path of the searched node.
     * @param regex a optional regex to apply to path before searching
     * @return the node matching the fully context from the root node, or <code>null</code> if not found.
     */
    public NavigationTreeNode findNode(String path, String regex) {
        return findNode(getRoot(), path, regex);
    }

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from the root node a node named by his fully path (concatenation of nodes
     * {@link NavigationTreeNode#navigationPath} valued separated by {@link #navigationPathSeparator}.
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path  the fully path of the searched node.
     * @param regex a optional regex to apply to path before searching
     * @return the node matching the fully context from the root node, or <code>null</code> if not found.
     */
    public NavigationTreeNode findNode(String path, Pattern regex) {
        return findNode(getRoot(), path, regex);
    }


    /**
     * Search from a given root node a node named by his fully path (concatenation of nodes
     * {@link NavigationTreeNode#navigationPath} valued separated by {@link #navigationPathSeparator}.
     *
     * @param root root node to be used
     * @param path the fully path of the searched node.
     * @return the node matching the fully context from the given root node, or <code>null</code> if not found.
     */
    public NavigationTreeNode findNode(NavigationTreeNode root, String path) {
        return findNode(root, path, (Pattern) null);
    }

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from a given root node a node named by his fully path (concatenation of nodes)
     * {@link NavigationTreeNode#navigationPath} valued separated by {@link #navigationPathSeparator}.
     *
     * @param root  root node to be used
     * @param path  the fully path of the searched node.
     * @param regex a previous regex to apply to path : must have a matches
     * @return the node matching the fully context from the given root node, or <code>null</code> if not found.
     */
    public NavigationTreeNode findNode(NavigationTreeNode root, String path, String regex) {
        return findNode(root, path, regex == null ? null : Pattern.compile(regex));
    }

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from a given root node a node named by his fully path (concatenation of nodes
     * {@link NavigationTreeNode#navigationPath} valued separated by {@link #navigationPathSeparator}.
     *
     * @param root  root node to be used
     * @param path  the fully path of the searched node.
     * @param regex a previous regex to apply to path : must have a matches
     * @return the node matching the fully context from the given root node, or <code>null</code> if not found.
     */
    public NavigationTreeNode findNode(NavigationTreeNode root, String path, Pattern regex) {
        if (regex != null) {
            Matcher matcher = regex.matcher(path);
            if (!matcher.matches() || matcher.groupCount() < 1) {
                log.warn("no matching regex " + regex + " to " + path);
                return null;
            }
            path = matcher.group(1);
            if (log.isDebugEnabled()) {
                log.debug("matching regex " + regex + " : " + path);
            }
        }
        StringTokenizer stk = new StringTokenizer(path, navigationPathSeparator);
        NavigationTreeNode result = root;
        // pas the first token (matches the root node)
        if (root.isRoot() && stk.hasMoreTokens()) {
            String rootPath = stk.nextToken();
            if (!rootPath.equals(root.getNavigationPath())) {
                return null;
            }
        }
        while (stk.hasMoreTokens()) {
            result = result.getChild(stk.nextToken());
        }
        return result;
    }


    /**
     * Obtain the associated bean value from context corresponding to node from given navigation path.
     *
     * @param context        the context where to seek value
     * @param navigationPath the current context path of the node
     * @return the value associated in context with the given navigation path
     */
    public Object getJAXXContextValue(JAXXContext context, String navigationPath) {
        Object result;
        NavigationTreeNode node = findNode(navigationPath, (Pattern) null);
        result = getJAXXContextValue(context, node);
        return result;
    }

    /**
     * Obtain the associated bean value from context corresponding to node
     *
     * @param context the context where to seek value
     * @param node    the current node
     * @return the value associated in context with the given node.
     */
    public Object getJAXXContextValue(JAXXContext context, NavigationTreeNode node) {
        if (node == null) {
            return null;
            //fixme should throw a NPE exception
            //throw new NullPointerException("node can not be null");
        }
        return node.getJAXXContextValue(context);
    }


    @Override
    public void nodeChanged(TreeNode node) {
        nodeChanged(node, false);
    }

    public void nodeChanged(TreeNode node, boolean deep) {
        NavigationTreeNode n = (NavigationTreeNode) node;
        n.clearCache(!deep);
        super.nodeChanged(node);
        if (deep) {
            Enumeration<?> childs = node.children();
            while (childs.hasMoreElements()) {
                NavigationTreeNode o = (NavigationTreeNode) childs.nextElement();
                nodeChanged(o, true);
            }
        }
    }

    /**
     * la représentation d'un noeud dans le modele {@link NavigationTreeModel}
     *
     * @author chemit
     */
    public class NavigationTreeNode extends DefaultMutableTreeNode {

        private static final long serialVersionUID = 1L;

        /** pour representer le context du noeud. */
        protected String navigationPath;

        /**
         * the cached complete navigation path from root node
         * used for performance issues.
         */
        protected String cachedNavigationPath;

        /** the JAXXObject class associated with this node (can be null) */
        protected Class<? extends JAXXObject> jaxxClass;

        /** the JAXXAction class associated with this node and will be put in ui context */
        protected Class<? extends JAXXAction> jaxxActionClass;

        /** the definition of the JAXXContext entry associated to this node, if null will seek in parent */
        protected JAXXContextEntryDef<?> jaxxContextEntryDef;

        /** jxPath to process to obtain real value associated from context with the node (can be null) */
        protected String jaxxContextEntryPath;

        /** cache of bean associated with bean to improve performance */
        protected transient Object cachedBean;

        /** renderer of the node */
        protected NodeRenderer renderer;

        /**
         * The type of the related bean associated with the node.
         * <p/>
         * Note: This type is here to override the NodeRenderer internalClass, since
         * we could need to override this data.
         * <p/>
         * If this property is let to null, then we will use the NodeRenderer one
         */
        protected Class<?> internalClass;

        public NavigationTreeNode(Object renderer,
                                  Object jaxxContextEntryDef,
                                  String navigationPath,
                                  Class<? extends JAXXObject> jaxxClass,
                                  Class<? extends JAXXAction> jaxxActionClass) {
            super(renderer);
            if (renderer instanceof NodeRenderer) {
                // the renderer must keep a reference of the node
                this.renderer = (NodeRenderer) renderer;
                this.renderer.setNode(this);
            } else if (renderer instanceof String) {
                // nothing special to be done
            } else if (renderer != null) {
                // wrong renderer type
                throw new IllegalArgumentException("to define a renderer, must be a String (simple libelle) or a  " + NodeRenderer.class + ", but was " + renderer);
            }
            this.navigationPath = navigationPath;
            this.jaxxClass = jaxxClass;
            this.jaxxActionClass = jaxxActionClass;

            if (jaxxContextEntryDef instanceof JAXXContextEntryDef<?>) {
                this.jaxxContextEntryDef = ((JAXXContextEntryDef<?>) jaxxContextEntryDef);
            } else if (jaxxContextEntryDef instanceof String) {
                this.jaxxContextEntryPath = (String) jaxxContextEntryDef;
            } else if (jaxxContextEntryDef != null) {
                // wrong context definition type
                throw new IllegalArgumentException("to define a context link, must be a String (jxpath) or a " + JAXXContextEntryDef.class + ", but was " + jaxxContextEntryDef);
            }
        }

        public NavigationTreeNode(Object renderer,
                                  JAXXContextEntryDef<?> jaxxContextEntryDef,
                                  String jaxxContextEntryPath,
                                  String navigationPath,
                                  Class<? extends JAXXObject> jaxxClass,
                                  Class<? extends JAXXAction> jaxxActionClass) {
            super(renderer);
            if (renderer instanceof NodeRenderer) {
                // the renderer must keep a reference of the node
                this.renderer = (NodeRenderer) renderer;
                this.renderer.setNode(this);
            } else if (renderer instanceof String) {
                // nothing special to be done
            } else if (renderer != null) {
                // wrong renderer type
                throw new IllegalArgumentException("to define a renderer, must be a String (simple libelle) or a  " + NodeRenderer.class + ", but was " + renderer);
            }
            this.navigationPath = navigationPath;
            this.jaxxClass = jaxxClass;
            this.jaxxActionClass = jaxxActionClass;
            this.jaxxContextEntryDef = jaxxContextEntryDef;
            this.jaxxContextEntryPath = jaxxContextEntryPath;
        }

        public String getNavigationPath() {
            return navigationPath;
        }

        public void setNavigationPath(String navigationPath) {
            this.navigationPath = navigationPath;
        }

        public Class<? extends JAXXObject> getJaxxClass() {
            return jaxxClass;
        }

        public void setJaxxClass(Class<? extends JAXXObject> jaxxClass) {
            this.jaxxClass = jaxxClass;
        }

        public void setInternalClass(Class<?> internalClass) {
            this.internalClass = internalClass;
        }

        public Class<? extends JAXXAction> getJaxxActionClass() {
            return jaxxActionClass;
        }

        public void setJaxxActionClass(Class<? extends JAXXAction> jaxxActionClass) {
            this.jaxxActionClass = jaxxActionClass;
        }

        public JAXXContextEntryDef<?> getJaxxContextEntryDef() {
            return jaxxContextEntryDef;
        }

        public void setJaxxContextEntryDef(JAXXContextEntryDef<?> jaxxContextEntryDef) {
            this.jaxxContextEntryDef = jaxxContextEntryDef;
        }

        public String getJaxxContextEntryPath() {
            return jaxxContextEntryPath;
        }

        public void setJaxxContextEntryPath(String jaxxContextEntryPath) {
            this.jaxxContextEntryPath = jaxxContextEntryPath;
        }

        public Class<?> getInternalClass() {
            return internalClass == null ? renderer.getInternalClass() : internalClass;
        }

        /** @return the fully context pathof the node from the root node to this. */
        public String getContextPath() {
            if (cachedNavigationPath == null) {
                TreeNode[] path = getPath();
                StringBuilder sb = new StringBuilder();
                for (TreeNode treeNode : path) {
                    NavigationTreeNode myNode = (NavigationTreeNode) treeNode;
                    sb.append(navigationPathSeparator).append(myNode.getNavigationPath());
                }
                cachedNavigationPath = sb.substring(1);
            }
            return cachedNavigationPath;
        }

        @Override
        public NavigationTreeNode getChildAt(int index) {
            return (NavigationTreeNode) super.getChildAt(index);
        }

        @Override
        public NavigationTreeNode getParent() {
            return (NavigationTreeNode) super.getParent();
        }

        /**
         * @param navigationPath the name of the {@link #navigationPath} to be matched in the cild of this node.
         * @return the child of this node with given {@link # navigationPath} value.
         */
        public NavigationTreeNode getChild(String navigationPath) {
            for (int i = 0, max = getChildCount(); i < max; i++) {
                NavigationTreeNode son = getChildAt(i);
                if (navigationPath.equals(son.getNavigationPath())) {
                    return son;
                }
            }
            return null;
        }

        /**
         * Obtain the associated bean value from context corresponding to node
         *
         * @param context the context to seek
         * @return the value associated in context with the given context path
         */
        public Object getJAXXContextValue(JAXXContext context) {
            Object result;

            if (cachedBean != null) {
                // use cached bean
                return cachedBean;
            }

            if (getJaxxContextEntryDef() != null && jaxxContextEntryPath == null) {
                // the node maps directly a value in context, with no jxpath resolving
                result = getJaxxContextEntryDef().getContextValue(context);
                // save in cache
                setCachedBean(result);
                return result;
            }
            // find the first ancestor node with a context def
            NavigationTreeNode parentNode = getFirstAncestorWithDef();
            if (parentNode == null) {
                log.warn("could not find a ancestor node with a definition of a context entry from node (" + this + ")");
                // todo must be an error
                // no parent found
                return null;
            }

            Object parentBean = parentNode.getJaxxContextEntryDef().getContextValue(context);

            if (parentBean == null) {
                // must be an error no bean found
                log.warn("culd not find a bean attached in context from context entry definition " + parentNode.getJaxxContextEntryDef());
                return null;
            }

            if (parentNode.jaxxContextEntryPath != null) {
                // apply the jxpath on parentBean
                JXPathContext jxcontext = JXPathContext.newContext(parentBean);

                parentBean = jxcontext.getValue(parentNode.jaxxContextEntryPath);
            }

            // save in cache
            parentNode.setCachedBean(parentBean);

            if (this == parentNode) {
                // current node is the node matching the context entry value and no jxpath is found
                return parentBean;
            }

            if (jaxxContextEntryPath == null) {
                // todo must be an error
                log.warn("must find a jaxxContextEntryPath on node (" + this + ")");
                return null;
            }

            String jxpathExpression = computeJXPath(jaxxContextEntryPath, parentNode);

            if (jxpathExpression == null) {
                /// todo must be an error
                log.warn("could not build jxpath from node " + parentNode + " to " + this);
                // could not retreave the jxpath...
                return null;
            }
            if (jxpathExpression.startsWith("[")) {
                // special case when we want to access a collection
                jxpathExpression = '.' + jxpathExpression;
            }
            if (log.isDebugEnabled()) {
                log.debug("jxpath : " + jxpathExpression);
            }

            JXPathContext jxcontext = JXPathContext.newContext(parentBean);

            result = jxcontext.getValue(jxpathExpression);

            // save in cache
            setCachedBean(result);

            return result;
        }

        /**
         * @return the first ancestor with a none null {@link #jaxxContextEntryDef}
         *         or <code>null</code> if none find..
         */
        protected NavigationTreeNode getFirstAncestorWithDef() {
            if (jaxxContextEntryDef != null) {
                return this;
            }
            return getParent() == null ? null : getParent().getFirstAncestorWithDef();
        }

        protected String computeJXPath(String expr, NavigationTreeNode parentNode) {
            if (parentNode == this) {
                // reach the parent limit node, return the expr computed
                return expr;
            }
            int firstIndex = expr.indexOf("..");
            int lastIndex = expr.lastIndexOf("..");

            if (firstIndex == -1) {
                // this is a error, since current node is not parent limit node,
                // we must find somewhere a way to go up in nodes
                throw new IllegalArgumentException(expr + " should contains at least one \"..\"");
            }

            if (firstIndex != 0) {
                // this is a error, the ../ must be at the beginning of the expression
                throw new IllegalArgumentException("\"..\" must be at the beginning but was : " + expr);
            }

            NavigationTreeNode ancestor = getParent();

            if (firstIndex == lastIndex) {
                // found only one go up, so must be substitute by the parent node context

                String newExpr = expr.substring(2);
                //String newExpr = expr.substring(expr.startsWith("../") ? 3 : 2);

                if (getParent().equals(parentNode)) {

                    // parent node is the final parent node, so no substitution needed
                    return newExpr;
                    //return parentNode.computeJXPath(newExpr, parentNode);
                }

                // ancestor must have a jaxxContextEntryPath 
                if (ancestor.jaxxContextEntryPath == null) {
                    throw new IllegalArgumentException("with the expression " + expr + ", the ancestor node (" + ancestor + ") must have a jaxxContextEntryPath definition, but was not ");
                }

                newExpr = ancestor.jaxxContextEntryPath + newExpr;

                return ancestor.computeJXPath(newExpr, parentNode);
            }

            // have more than one go up, so the ancestor node can not have a jaxxContextEntryPath
            if (ancestor.jaxxContextEntryPath != null) {
                throw new IllegalArgumentException("with the expression " + expr + ", the ancestor node can not have a jaxxContextEntryPath definition");
            }

            // substitute the last ..[/] and delegate to ancestor
            String newExpr = expr.substring(0, lastIndex - 1) + expr.substring(lastIndex + (expr.charAt(lastIndex + 3) == '/' ? 3 : 2));

            return ancestor.computeJXPath(newExpr, parentNode);
        }

        public void clearCache() {
            clearCache(false);
        }

        public void clearCache(boolean deep) {

            // clear bean cache
            cachedBean = null;

            // clear context navigation cache
            cachedNavigationPath = null;

            // clear render cache
            if (renderer != null) {
                renderer.setRendererCachedValue(null);
            }

            if (deep) {
                // clear cache in childs
                Enumeration<?> childs = this.children();
                while (childs.hasMoreElements()) {
                    NavigationTreeNode o = (NavigationTreeNode) childs.nextElement();
                    o.clearCache();
                }
            }
        }

        public Object getCachedBean() {
            return cachedBean;
        }

        public void setCachedBean(Object cachedBean) {
            this.cachedBean = cachedBean;
        }
    }

}
