/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.runtime.swing;

import javax.swing.JPanel;
import java.awt.Insets;

/**
 * Panel which uses a {@link VBoxLayout} by default.
 *
 * @author Ethan Nicholas
 */
public class VBox extends JPanel {

    private static final long serialVersionUID = 1L;
    public static final String SPACING_PROPERTY = "spacing";
    public static final String MARGIN_PROPERTY = "margin";
    public static final String HORIZONTAL_ALIGNMENT_PROPERTY = "horizontalAlignment";
    public static final String VERTICAL_ALIGNMENT_PROPERTY = "verticalAlignment";
    private Insets margin;

    public VBox() {
        super(new VBoxLayout());
    }

    /**
     * Returns the spacing between components, in pixels.   Spacing is applied between components only,
     * not to the top or bottom of the container.
     *
     * @return spacing between components
     */
    public int getSpacing() {
        return ((VBoxLayout) getLayout()).getSpacing();
    }

    /**
     * Sets the spacing between components.   Spacing is applied between components only,
     * not to the top or bottom of the container.
     *
     * @param spacing new spacing value
     */
    public void setSpacing(int spacing) {
        int oldValue = getSpacing();
        ((VBoxLayout) getLayout()).setSpacing(spacing);
        firePropertyChange(SPACING_PROPERTY, oldValue, spacing);
        revalidate();
    }

    public int getHorizontalAlignment() {
        return ((VBoxLayout) getLayout()).getHorizontalAlignment();
    }

    public void setHorizontalAlignment(int horizontalAlignment) {
        int oldValue = getHorizontalAlignment();
        ((VBoxLayout) getLayout()).setHorizontalAlignment(horizontalAlignment);
        firePropertyChange(HORIZONTAL_ALIGNMENT_PROPERTY, oldValue, horizontalAlignment);
        revalidate();
    }

    public int getVerticalAlignment() {
        return ((VBoxLayout) getLayout()).getVerticalAlignment();
    }

    public void setVerticalAlignment(int verticalAlignment) {
        int oldValue = getVerticalAlignment();
        ((VBoxLayout) getLayout()).setVerticalAlignment(verticalAlignment);
        firePropertyChange(VERTICAL_ALIGNMENT_PROPERTY, oldValue, verticalAlignment);
        revalidate();
    }

    public Insets getMargin() {
        return margin;
    }

    public void setMargin(Insets margin) {
        Insets oldValue = this.margin;
        this.margin = (Insets) margin.clone();
        firePropertyChange(MARGIN_PROPERTY, oldValue, margin);
    }

    @Override
    public Insets getInsets() {
        Insets result = super.getInsets();
        if (margin != null) {
            result.top += margin.top;
            result.left += margin.left;
            result.right += margin.right;
            result.bottom += margin.bottom;
        }
        return result;
    }
}
