/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.runtime.swing;

import javax.swing.JTree;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.List;

public class JAXXTree extends JTree {

    private static final long serialVersionUID = 1L;
    private static final String SYNTHETIC = "<synthetic root node>";

    public class JAXXTreeModel implements TreeModel {

        private Item root;
        private List<TreeModelListener> listeners = new ArrayList<TreeModelListener>();

        public JAXXTreeModel(List<Item> items) {
            if (items.size() == 1) {
                this.root = items.get(0);
            } else {
                this.root = new Item(null, null, SYNTHETIC, false);
                for (Item item : items) {
                    root.addChild(item);
                }
            }

            PropertyChangeListener listener = new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent e) {
                    if (e.getPropertyName().equals(Item.SELECTED_PROPERTY)) {
                        Item item = (Item) e.getSource();
                        if (item.isSelected()) {
                            addSelectionPath(getTreePath(item));
                        } else {
                            removeSelectionPath(getTreePath(item));
                        }
                    } else {
                        Item item = (Item) e.getSource();
                        boolean root = item.getParent() == null;
                        TreePath path = !root ? getTreePath(item.getParent()) : null;
                        fireTreeNodesChanged(new TreeModelEvent(JAXXTreeModel.this, path,
                                !root ? new int[]{item.getParent().getChildren().indexOf(item)} : null,
                                new Object[]{item.getValue()}));
                    }
                }
            };
            addPropertyChangeListener(root, listener);
        }

        private void addPropertyChangeListener(Item item, PropertyChangeListener listener) {
            item.addPropertyChangeListener(listener);
            List<Item> children = item.getChildren();
            for (Item aChildren : children) {
                addPropertyChangeListener(aChildren, listener);
            }
        }

        @Override
        public void addTreeModelListener(TreeModelListener listener) {
            listeners.add(listener);
        }


        /* This is an inefficient implementation, but hand-coded tree structures are unlikely to contain
        enough nodes for that to really matter.  This could be sped up with caching. */
        private Item findItem(Object value) {
            return findItem(root, value);
        }

        private Item findItem(Item node, Object value) {
            if (node.getValue() == value) {
                return node;
            } else {
                List<Item> children = node.getChildren();
                for (Item aChildren : children) {
                    Item result = findItem(aChildren, value);
                    if (result != null) {
                        return result;
                    }
                }
                return null;
            }
        }

        private TreePath getTreePath(Item node) {
            List<Object> path = new ArrayList<Object>();
            while (node != null) {
                path.add(0, node.getValue());
                node = node.getParent();
            }
            return new TreePath(path.toArray());
        }

        @Override
        public Object getChild(Object parent, int index) {
            Item node = findItem(parent);
            return node.getChildren().get(index).getValue();
        }

        @Override
        public int getChildCount(Object parent) {
            Item node = findItem(parent);
            return node.getChildren().size();
        }

        @Override
        public int getIndexOfChild(Object parent, Object child) {
            Item node = findItem(parent);
            List<Item> children = node.getChildren();
            for (int i = 0, j = children.size(); i < j; i++) {
                if (children.get(i).getValue() == child) {
                    return i;
                }
            }
            return -1;
        }

        @Override
        public Object getRoot() {
            return root.getValue();
        }

        @Override
        public boolean isLeaf(Object node) {
            Item item = findItem(node);
            return item != null && item.getChildren().size() == 0;
        }

        @Override
        public void removeTreeModelListener(TreeModelListener listener) {
            listeners.remove(listener);
        }

        public void fireTreeNodesChanged(TreeModelEvent e) {
            for (TreeModelListener listener : listeners) {
                listener.treeNodesChanged(e);
            }
        }

        @Override
        public void valueForPathChanged(TreePath path, Object newValue) {
        }
    }

    public JAXXTree(TreeModel model) {
        super(model);
    }

    public JAXXTree() {
        setCellRenderer(new DefaultTreeCellRenderer() {

            private static final long serialVersionUID = 1L;

            @Override
            public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
                TreeModel model = tree.getModel();
                if (model instanceof JAXXTreeModel) {
                    Item item = ((JAXXTreeModel) model).findItem(value);
                    if (item != null) {
                        String label = item.getLabel();
                        if (label != null) {
                            value = label;
                        }
                    }
                }
                return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            }
        });

        addTreeSelectionListener(new TreeSelectionListener() {

            @Override
            public void valueChanged(TreeSelectionEvent e) {
                TreeModel model = getModel();
                if (model instanceof JAXXTreeModel) {
                    scan((JAXXTreeModel) model, ((JAXXTreeModel) model).root);
                }
            }

            private void scan(JAXXTreeModel model, Item item) {
                TreePath path = model.getTreePath(item);
                if (item.isSelected() != isPathSelected(path)) {
                    item.setSelected(!item.isSelected());
                }
                List<Item> children = item.getChildren();
                for (Item aChildren : children) {
                    scan(model, aChildren);
                }
            }
        });
    }

    public void setItems(List<Item> items) {
        JAXXTreeModel model = new JAXXTreeModel(items);
        if (model.getRoot() != null) {
            setRootVisible(model.getRoot() != SYNTHETIC);
        }
        setModel(model);
    }

    public Object getSelectionValue() {
        TreePath selectionPath = getSelectionPath();
        return selectionPath != null ? selectionPath.getLastPathComponent() : null;
    }
}
