package jaxx.runtime;

import jaxx.runtime.*;
import jaxx.runtime.validator.BeanValidatorScope;
import jaxx.runtime.validator.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JList;
import javax.swing.JTable;
import java.awt.event.MouseListener;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableMouseListener;
import jaxx.runtime.validator.swing.SwingValidatorMessageListMouseListener;
import jaxx.runtime.validator.swing.SwingValidatorMessage;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import static org.nuiton.i18n.I18n.n_;

/**
 * The helper class for validation module.
 *
 * @author chemit
 */
public class SwingValidatorUtil extends BeanValidatorUtil {

    static ImageIcon errorIcon;
    static ImageIcon warningIcon;
    static ImageIcon infoIcon;
    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(SwingValidatorUtil.class);

    public static ImageIcon getErrorIcon() {
        if (errorIcon == null) {
            errorIcon = jaxx.runtime.Util.createImageIcon("error.png");
        }
        return errorIcon;
    }

    public static ImageIcon getInfoIcon() {
        if (infoIcon == null) {
            infoIcon = jaxx.runtime.Util.createImageIcon("info.png");
        }
        return infoIcon;
    }

    public static ImageIcon getWarningIcon() {
        if (warningIcon == null) {
            warningIcon = jaxx.runtime.Util.createImageIcon("warning.png");
        }
        return warningIcon;
    }

    protected SwingValidatorUtil() {
        // no instance
    }

    /**
     * Prepare the ui where to display the validators messages.
     * 
     * @param errorTable the table where to display validators messages
     */
    public static void installUI(JTable errorTable, SwingValidatorMessageTableRenderer render) {
        errorTable.setDefaultRenderer(Object.class, render);
        errorTable.getRowSorter().setSortKeys(java.util.Arrays.asList(new RowSorter.SortKey(0, SortOrder.ASCENDING)));
        SwingUtil.setI18nTableHeaderRenderer(errorTable,
                n_("validator.scope.header"),
                n_("validator.scope.header.tip"),
                n_("validator.field.header"),
                n_("validator.field.header.tip"),
                n_("validator.message.header"),
                n_("validator.message.header.tip"));
        // register a single 'goto widget error' mouse listener on errorTable
        registerErrorTableMouseListener(errorTable);
        SwingUtil.fixTableColumnWidth(errorTable, 0, 25);
    }

    /**
     * Register for a given validator list ui a validator mouse listener.
     *
     * Note: there is only one listener registred for a given list model, so
     * invoking this method tiwce or more will have no effect.
     *
     * @param list the validation ui list
     * @return the listener instanciate or found
     * @see SwingValidatorMessageListMouseListener
     */
    public static SwingValidatorMessageListMouseListener registerErrorListMouseListener(JList list) {
        SwingValidatorMessageListMouseListener listener = getErrorListMouseListener(list);

        if (listener != null) {
            return listener;
        }
        listener = new SwingValidatorMessageListMouseListener();
        if (log.isDebugEnabled()) {
            log.debug(listener.toString());
        }
        list.addMouseListener(listener);
        return listener;
    }

    /**
     * Register for a given validator table ui a validator mouse listener
     *
     * Note: there is onlt one listener registred for a givne table model, so
     * invokin this method twice or more will have no effect.
     *
     * @param table the validator table ui
     * @return the listener instanciate or found
     * @see SwingValidatorMessageTableMouseListener
     */
    public static SwingValidatorMessageTableMouseListener registerErrorTableMouseListener(JTable table) {
        SwingValidatorMessageTableMouseListener listener = getErrorTableMouseListener(table);

        if (listener != null) {
            return listener;
        }
        listener = new SwingValidatorMessageTableMouseListener();
        if (log.isDebugEnabled()) {
            log.debug(listener.toString());
        }
        table.addMouseListener(listener);
        return listener;
    }

    /**
     * @param list the validator list ui
     * @return the validator list mouse listener, or <code>null</code> if not found
     * @see SwingValidatorMessageListMouseListener
     */
    public static SwingValidatorMessageListMouseListener getErrorListMouseListener(JList list) {
        if (list != null) {
            for (MouseListener listener : list.getMouseListeners()) {
                if (listener instanceof SwingValidatorMessageTableMouseListener) {
                    return (SwingValidatorMessageListMouseListener) listener;
                }
            }
        }
        return null;
    }

    /**
     * @param table the validator table ui
     * @return the validator table mouse listener, or <code>null</code> if not found
     * @see SwingValidatorMessageTableMouseListener
     */
    public static SwingValidatorMessageTableMouseListener getErrorTableMouseListener(JTable table) {
        if (table != null) {
            for (MouseListener listener : table.getMouseListeners()) {
                if (listener instanceof SwingValidatorMessageTableMouseListener) {
                    return (SwingValidatorMessageTableMouseListener) listener;
                }
            }
        }
        return null;
    }

    public static String getMessage(SwingValidatorMessage model) {
        String text = model.getMessage();
        if (model.getField() != null) {
            text = model.getField().getI18nError(text);
        }
        return text;
    }

    public static String getFieldName(SwingValidatorMessage model, String value) {
        String text = null;
        JComponent editor = model.getEditor();
        if (editor != null) {
            text = (String) editor.getClientProperty("validatorLabel");
        /*if (l != null) {
        text = I18n._(l);
        } else {
        // TODO should try the text
        }*/
        }
        if (text == null) {
            text = value;
        }
        return text;
    }

    public static ImageIcon getIcon(BeanValidatorScope scope) {
        ImageIcon icon = null;
        switch (scope) {
            case ERROR:
                icon = getErrorIcon();
                break;
            case WARNING:
                icon = getWarningIcon();
                break;
            case INFO:
                icon = getInfoIcon();
                break;
        }
        return icon;
    }
}