/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.runtime.swing.Item;
import jaxx.runtime.swing.JAXXTree;
import jaxx.tags.DefaultComponentHandler;
import jaxx.types.TypeManager;
import org.w3c.dom.Element;

import javax.swing.event.TreeSelectionListener;
import java.io.IOException;
import java.util.List;

public class JTreeHandler extends DefaultComponentHandler {
    public JTreeHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, JAXXTree.class);
    }

    @Override
    protected void configureProxyEventInfo() {
        super.configureProxyEventInfo();
        addProxyEventInfo("getSelectionCount", TreeSelectionListener.class, "selectionModel");
        addProxyEventInfo("getSelectionPath", TreeSelectionListener.class, "selectionModel");
        addProxyEventInfo("getSelectionPaths", TreeSelectionListener.class, "selectionModel");
        addProxyEventInfo("getSelectionRows", TreeSelectionListener.class, "selectionModel");
        addProxyEventInfo("getSelectionValue", TreeSelectionListener.class, "selectionModel");
    }

    @Override
    public CompiledObject createCompiledObject(String id, JAXXCompiler compiler) throws CompilerException {
        return new CompiledItemContainer(id, getBeanClass(), compiler);
    }

    private void createItems(CompiledObject tree, List<Item> items, String addMethod, JAXXCompiler compiler) throws CompilerException {
        for (Item item : items) {
            String id = item.getId();
            CompiledObject compiledItem = new CompiledObject(id, ClassDescriptorLoader.getClassDescriptor(Item.class), compiler);
            compiledItem.setConstructorParams(TypeManager.getJavaCode(id) + ", " + TypeManager.getJavaCode(item.getLabel()) + ", " + TypeManager.getJavaCode(item.getValue()) + ", " + item.isSelected());
            compiler.registerCompiledObject(compiledItem);
            tree.appendAdditionCode(addMethod + "(" + id + ");");
            createItems(tree, item.getChildren(), id + ".addChild", compiler);
        }
    }

    @Override
    public void compileChildrenSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        super.compileChildrenSecondPass(tag, compiler);
        CompiledItemContainer tree = (CompiledItemContainer) compiler.getOpenComponent();
        List<Item> items = tree.getItems();
        if (items != null && !items.isEmpty()) {
            String listName = tree.getId() + "$items";
            tree.appendAdditionCode("java.util.List<jaxx.runtime.swing.Item> " + listName + " = new java.util.ArrayList<jaxx.runtime.swing.Item>();");
            createItems(tree, items, listName + ".add", compiler);
            tree.appendAdditionCode(tree.getId() + ".setItems(" + listName + ");");
        }
    }
}



