/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: TapestryBundleMojo.java 1839 2011-01-12 16:41:52Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.3/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/bundle/TapestryBundleMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.bundle;

import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.bundle.I18nBundleUtil;
import org.nuiton.io.SortedProperties;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;


/**
 * Generates a unique bundle for a tapestry application.
 * <p/>
 * <b>Note :</b> The mojo has nothing specific to tapestry and should be renamed
 * (or removed since the {@code bundle} mojo do the same with more options)...
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal tapestry-bundle
 * @phase generate-resources
 * @execute goal=collect-i18n-artifacts
 * @requiresProject true
 * @requiresDependencyResolution runtime
 * @since 1.2
 */
public class TapestryBundleMojo extends AbstractMakeI18nBundleMojo {

    @Override
    protected void doAction() throws Exception {
        long t00 = System.nanoTime();

        if (!silent) {
            getLog().info("config - bundle name : " + bundleOutputName);
            getLog().info("config - basedir     : " + bundleOutputDir);
            getLog().info("config - locales     : " + Arrays.toString(locales));
        }

        Map<Locale, String> bundleDico =
                new LinkedHashMap<Locale, String>(locales.length);

        for (Locale locale : locales) {

            long t0 = System.nanoTime();

            File bundleOut = getBundleFile(
                    bundleOutputDir,
                    bundleOutputName,
                    locale,
                    false
            );

            if (!silent) {
                getLog().info("generate bundle for locale " + locale +
                              " in file " + bundleOut.getName());
            }

            SortedProperties propertiesOut =
                    new SortedProperties(encoding, false);
            StringBuilder buffer = new StringBuilder();

            URL[] urls = getCollectI18nResources(locale);
            if (urls.length == 0) {
                getLog().warn("no bundle for locale " + locale);
                continue;
            }

            for (URL url : urls) {
                long t000 = System.nanoTime();
                I18nBundleEntry bundleEntry =
                        new I18nBundleEntry(url, locale, null);
                bundleEntry.load(propertiesOut);
                String strPath = bundleEntry.getPath().toString();
                int index = strPath.indexOf("i18n/");

                buffer.append(',').append(strPath.substring(index));
                if (verbose) {
                    getLog().info(
                            "loaded " + bundleEntry.getPath() + " in " +
                            PluginHelper.convertTime(t000, System.nanoTime()));
                }
            }

            if (buffer.length() > 0) {
                bundleDico.put(locale, buffer.substring(1));
                if (!silent) {
                    getLog().info(
                            "bundles for locale : " + bundleDico.get(locale));
                }
            }
            propertiesOut.store(bundleOut);
            if (!silent && verbose) {
                getLog().info(
                        "bundle created in " +
                        PluginHelper.convertTime(t0, System.nanoTime()) +
                        " (detected sentences : " + propertiesOut.size() + ")");
            }
            if (checkBundle) {
                checkBundle(locale, propertiesOut, showEmpty, unsafeMapping);
            }
        }

        failsIfWarning();

        if (generateDefaultLocale) {
            generateDefaultBundle();
        }

        if (!silent && verbose) {
            getLog().info("done in " +
                          PluginHelper.convertTime(t00, System.nanoTime()));
        }
    }

    /**
     * @param root       le repertoire ou sont stockes les fichiers i18n
     * @param artifactId le nom de l'artifact
     * @param locale     le nom du bundle
     * @param create     {@code true} pour creer le fichier si non present
     * @return le fichier i18n
     * @throws IOException si probleme lors de la creation du fichier
     */
    @Override
    protected File getBundleFile(File root,
                                 String artifactId,
                                 Locale locale,
                                 boolean create) throws IOException {
        String path = root.getAbsolutePath() + File.separatorChar + artifactId;
        if (locale != null) {
            path += "_" + locale.getLanguage();
        }
        path += ".properties";
        File file = new File(
                path);
        if (create && !file.exists()) {
            createNewFile(file);
        }
        return file;
    }

    @Override
    protected URL[] getCollectI18nResources(Locale locale) throws IOException {
        File file = getCollectOutputFile(locale, false);
        if (!file.exists()) {
            return I18nBundleUtil.EMPTY_URL_ARRAY;
        }
        URL[] urls = PluginHelper.getLinesAsURL(file);
        return urls;
    }


}
