package org.nuiton.eugene.plugin;

/*
 * #%L
 * I18n :: Maven Plugin
 * %%
 * Copyright (C) 2007 - 2016 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.antlr.v4.runtime.ANTLRInputStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.TokenStream;
import org.antlr.v4.runtime.misc.NotNull;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.nuiton.eugene.DefaultTemplateConfiguration;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.config.templates.ApplicationConfigTransformer;
import org.nuiton.eugene.java.JavaGeneratorUtil;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelClassImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelImpl;
import org.nuiton.eugene.plugin.parser.java.Java8BaseVisitor;
import org.nuiton.eugene.plugin.parser.java.Java8Lexer;
import org.nuiton.eugene.plugin.parser.java.Java8Parser;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginHelper;
import org.nuiton.plugin.PluginWithEncoding;
import org.nuiton.version.Version;

import javax.swing.KeyStroke;
import java.awt.Color;
import java.io.File;
import java.net.URL;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

/**
 * Generate a i18n enum class helper.
 *
 * Created on 28/08/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
@Mojo(name = "generate-config", defaultPhase = LifecyclePhase.GENERATE_SOURCES, requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class GenerateApplicationConfigMojo extends AbstractPlugin implements PluginWithEncoding {

    /**
     * To set the package fully qualified name of the generated class.
     *
     * By default, will use groupId.artifactId (with {@code -} replaced by {@code .}) plus {@code .config}.
     */
    @Parameter(property = "eugene.packageName")
    private String packageName;

    /**
     * Model name (will prefix the generated files names with it).
     */
    @Parameter(property = "eugene.modelName", required = true)
    private String modelName;

    /**
     * The fully qualified name of the enum class of options to scan to generate the configuration java files.
     *
     * By default, will use groupId.artifactId (with {@code -} replaced by {@code .}) plus {@code .config.modelNameOption}.
     */
    @Parameter(property = "eugene.optionsClassName")
    private String optionsClassName;

    /**
     * The fully qualified name of the enum class of actions to scan to generate the configuration java files.
     *
     * By default, will use groupId.artifactId (with {@code -} replaced by {@code .}) plus {@code .config.modelNameAction}.
     */
    @Parameter(property = "eugene.actionsClassName")
    private String actionsClassName;

    /**
     * The source directory where to scan options java file.
     */
    @Parameter(property = "eugene.sourceDirectory", defaultValue = "${basedir}/src/main/java", required = true)
    private File sourceDirectory;

    /**
     * The root directory where to generated.
     */
    @Parameter(property = "eugene.outputdirectory", defaultValue = "${basedir}/target/generated-sources/java", required = true)
    private File outputdirectory;

    /**
     * Pour activer le mode verbeux.
     */
    @Parameter(property = "eugene.verbose", defaultValue = "${maven.verbose}")
    protected boolean verbose;

    /**
     * Encoding to be used for generation of files.
     *
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     */
    @Parameter(property = "eugene.encoding", defaultValue = "${project.build.sourceEncoding}")
    protected String encoding;

    /**
     *
     */
    @Parameter(property = "eugene.prefix", defaultValue = "Config")
    protected String prefix;

    /**
     * Maven project.
     */
    @Parameter(defaultValue = "${project}", readonly = true)
    protected MavenProject project;

    protected ObjectModelImpl model;

    protected ApplicationConfigTransformer template;

    @Override
    protected void init() throws Exception {

        if (packageName == null) {

            packageName = getProject().getGroupId() + "." + getProject().getArtifactId().replaceAll("-", ".");
            getLog().info("Use package name: " + packageName);
        }

        if (optionsClassName == null) {

            optionsClassName = packageName + "." + modelName + prefix + "Option";

            File file = PluginHelper.getFile(sourceDirectory, optionsClassName.split("\\."));
            File javaFile = new File(file.getParentFile(), file.getName() + ".java");

            if (!javaFile.exists()) {

                prefix = "Configuration";

                optionsClassName = packageName + "." + modelName + prefix + "Option";

                file = PluginHelper.getFile(sourceDirectory, optionsClassName.split("\\."));
                javaFile = new File(file.getParentFile(), file.getName() + ".java");

                if (!javaFile.exists()) {
                    throw new MojoExecutionException("Can't find option file at: " + javaFile);
                }
            }
            getLog().info("Detected configuration options class: " + optionsClassName);

        }

        if (actionsClassName == null) {

            actionsClassName = packageName + "." + modelName + prefix + "Action";

            File file = PluginHelper.getFile(sourceDirectory, actionsClassName.split("\\."));
            File javaFile = new File(file.getParentFile(), file.getName() + ".java");

            if (javaFile.exists()) {

                getLog().info("Detected configuration actions class: " + actionsClassName);
            } else {

                // won't be used
                actionsClassName = null;
            }

        }

        ClassLoader loader = this.initClassLoader(getProject(), sourceDirectory, true, false, false, true, true);

        // get options

        model = new ObjectModelImpl();
        model.setName(modelName);



        String className = modelName + prefix;
        getLog().info("Config class name: " + className);

        File file = PluginHelper.getFile(sourceDirectory, optionsClassName.split("\\."));
        File javaFile = new File(file.getParentFile(), file.getName() + ".java");

        String content = FileUtils.readFileToString(javaFile, encoding);
        TokenStream tokenStream = new CommonTokenStream(new Java8Lexer(new ANTLRInputStream(content)));
        Java8Parser parser = new Java8Parser(tokenStream);


        // see http://stackoverflow.com/a/32918434/2038100
        //parser.setErrorHandler(new BailErrorStrategy());
        //parser.getInterpreter().setPredictionMode(PredictionMode.SLL);
        //parser.getInterpreter().tail_call_preserves_sll = false;
        parser.getInterpreter().enable_global_context_dfa = true;

        JavaParserVisitor visitor = new JavaParserVisitor(loader, javaFile);
        parser.compilationUnit().accept(visitor);

        Map<String, String> names = visitor.getNames();

        ObjectModelClassImpl aClass = new ObjectModelClassImpl();
        aClass.setName(className);
        aClass.setPackage(packageName);
        model.addClass(aClass);

        for (Map.Entry<String, String> entry : names.entrySet()) {
            ObjectModelAttributeImpl attribute = new ObjectModelAttributeImpl();
            attribute.setName(JavaGeneratorUtil.convertConstantNameToVariableName(entry.getKey()));
            attribute.setType(entry.getValue());
            aClass.addAttribute(attribute);
        }

        Properties templateProperties = new Properties();

        templateProperties.put(Template.PROP_ENCODING, getEncoding());
        templateProperties.put(Template.PROP_VERBOSE, verbose);
        templateProperties.put(Template.PROP_OVERWRITE, true);
        templateProperties.put(Template.PROP_CLASS_LOADER, loader);
        templateProperties.put(ApplicationConfigTransformer.PROP_OPTION_CLASS_NAME, optionsClassName);
        if (actionsClassName != null) {
            templateProperties.put(ApplicationConfigTransformer.PROP_ACTION_CLASS_NAME, actionsClassName);
        }

        template = new ApplicationConfigTransformer();
        template.setConfiguration(new DefaultTemplateConfiguration(templateProperties));

        if (!project.getCompileSourceRoots().contains(outputdirectory.getPath())) {
            if (isVerbose()) {
                getLog().info("Add compile source root : " + outputdirectory);
            }
            project.addCompileSourceRoot(outputdirectory.getPath());
        }
    }

    @Override
    protected void doAction() throws Exception {

        getLog().info("Generate file(s) to: " + outputdirectory);

        template.applyTemplate(model, outputdirectory);

    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public String getEncoding() {
        return encoding;
    }

    @Override
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    protected class JavaParserVisitor extends Java8BaseVisitor<Void> {

        protected final Map<String, String> names;

        private final ClassLoader loader;
        protected final File file;

        private JavaParserVisitor(ClassLoader loader, File file) {
            this.loader = loader;
            this.file = file;
            names = new LinkedHashMap<>();
        }

        public Map<String, String> getNames() {
            return names;
        }

        String name;
        boolean first;

        @Override
        public Void visitEnumConstant(Java8Parser.EnumConstantContext ctx) {
            name = ctx.getChild(0).getText();
            first = false;
            return super.visitEnumConstant(ctx);
        }

        @Override
        public Void visitArgumentList(@NotNull Java8Parser.ArgumentListContext ctx) {
            if (!first) {

                String type = StringUtils.removeEnd(ctx.getChild(0).getText(), ".class");
                switch (type) {
                    case "File":
                        type = File.class.getName();
                        break;
                    case "Color":
                        type = Color.class.getName();
                        break;
                    case "KeyStroke":
                        type = KeyStroke.class.getName();
                        break;
                    case "URL":
                        type = URL.class.getName();
                        break;
                    case "Class":
                        type = Class.class.getName();
                        break;
                    case "Date":
                        type = Date.class.getName();
                        break;
                    case "Time":
                        type = Time.class.getName();
                        break;
                    case "Timestamp":
                        type = Timestamp.class.getName();
                        break;
                    case "Locale":
                        type = Locale.class.getName();
                        break;
                    case "Version":
                        type = Version.class.getName();
                        break;
                    case "String":
                    case "int":
                    case "Integer":
                    case "long":
                    case "Long":
                    case "float":
                    case "Float":
                    case "boolean":
                    case "Boolean":
                    case "byte":
                    case "Byte":
                    case "char":
                    case "Character":
                    case "double":
                    case "Double":
                        break;
                    default:
                        try {
                            loader.loadClass(type);
                        } catch (ClassNotFoundException e) {

                            // try to see if class found in source path

                            File sourceFile = PluginHelper.getFile(sourceDirectory, type.split("\\."));
                            sourceFile = new File(sourceFile.getParentFile(), sourceFile.getName() + ".java");
                            if (!sourceFile.exists()) {

                                throw new IllegalStateException("Can't find real type of " + type + " for option: " + name + ", please use the fully qualified name in code");
                            }
                        }
                }

                names.put(name, type);
                first = true;
            }
            return super.visitArgumentList(ctx);
        }

    }

}
