/*
 * #%L
 * Pollen :: VoteCounting Api
 * $Id: VoteCountingResult.java 3712 2012-09-30 12:57:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-votecounting-api/src/main/java/org/chorem/pollen/votecounting/model/VoteCountingResult.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.model;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimaps;

import java.io.Serializable;
import java.util.List;

/**
 * Contains results for a vote.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class VoteCountingResult implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * Results for each choice.
     * <p/>
     * <strong>Note:</strong> Natural order is used to describe choice scores
     * (first is winner,...).
     */
    private List<ChoiceScore> scores;

    /**
     * Result for each choice order by their winning rank.
     *
     * @see ChoiceScore#getScoreOrder()
     */
    private transient ArrayListMultimap<Integer, ChoiceScore> scoresByRank;

    public static VoteCountingResult newResult(List<ChoiceScore> scores) {
        VoteCountingResult result = new VoteCountingResult();
        result.setScores(scores);
        return result;
    }

    public List<ChoiceScore> getScores() {
        return scores;
    }

    public ChoiceScore getScore(String choiceId) {
        ChoiceScore result = null;
        for (ChoiceScore score : scores) {
            if (choiceId.equals(score.getChoiceId())) {
                result = score;
                break;
            }
        }
        return result;
    }

    public void setScores(List<ChoiceScore> scores) {
        this.scores = Lists.newArrayList(scores);
    }

    public List<ChoiceScore> getTopRanking() {
        return getScoresByRank().get(0);
    }

    public ArrayListMultimap<Integer, ChoiceScore> getScoresByRank() {
        if (scoresByRank == null) {
            scoresByRank = ArrayListMultimap.create(
                    Multimaps.index(scores, ChoiceScore.SCORE_BY_ORDER));
        }
        return scoresByRank;
    }
}
