/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.user;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.fileupload.FileUploadException;
import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.Retain;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.chorem.pollen.business.dto.PersonListDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.services.ServiceList;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.EvenOdd;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.utils.CSVAccountUtil;
import org.chorem.pollen.ui.utils.LDAPAccountUtil;

/**
 * Classe de la page d'affichage des listes de votants favorites de
 * l'utilisateur. Si l'utilisateur n'est pas identifié, on lui demande d'abord
 * de s'identifier via un formulaire.
 *
 * @author rannou
 * @version $Id: UserLists.java 2703 2009-08-13 10:47:20Z nrannou $
 */
@IncludeStylesheet("context:css/lists.css")
public class UserLists {

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /**
     * Listes de favoris de l'utilisateur.
     */
    @Property
    private List<PersonListDTO> lists;

    /**
     * Liste de favoris sélectionnée.
     */
    @Persist
    @Property
    private PersonListDTO selectedList;

    /**
     * Nouvelle liste de favoris.
     */
    @Property
    private PersonListDTO newList;

    /**
     * Compte courant (pour l'itération des votants de la liste).
     */
    @SuppressWarnings("unused")
    @Property
    private PollAccountDTO account;

    /**
     * Nouveau compte.
     */
    @Property
    private PollAccountDTO newAccount;

    /**
     * Fichier CSV contenant une liste de votants.
     */
    @Property
    private UploadedFile accountsFile;

    /**
     * URL de recherche LDAP pour une liste de votants.
     */
    @Property
    private String accountsUrl;

    /**
     * Zone à rafraîchir.
     */
    @InjectComponent
    private Zone listsZone;

    /**
     * Formulaire de création de liste.
     */
    @Component
    private Form createListForm;

    /**
     * Formulaire de création de compte.
     */
    @Component
    private Form createAccountForm;

    /**
     * Messages.
     */
    @Inject
    private Messages messages;

    /**
     * Objet servant à changer la couleur à chaque ligne de la liste.
     */
    @SuppressWarnings("unused")
    @Property
    private EvenOdd evenOdd;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /**
     * Sert à créer listModel.
     */
    @Inject
    private BeanModelSource beanModelSource;

    /**
     * Sert à passer les messages en paramètre de la création de listModel.
     */
    @Inject
    private ComponentResources componentResources;

    /**
     * Modèle pour l'affichage de la liste des favoris.
     */
    @SuppressWarnings("unchecked")
    @Property
    @Retain
    private BeanModel listModel;

    /** Injection des services */
    @Inject
    private ServiceList serviceList;
    @Inject
    private ServicePollAccount servicePollAccount;

    /**
     * Méthode appelée à la création d'une liste.
     */
    Object onSuccessFromCreateListForm() {

        // Contrôle du nom de la liste
        for (PersonListDTO list : lists) {
            if (list.getName().equals(newList.getName())) {
                createListForm.recordError(messages.format("listExists",
                        newList.getName()));
                return this;
            }
        }

        // Import CVS des comptes
        if (accountsFile != null) {
            if (!accountsFile.getContentType().equals("text/csv")) {
                createListForm.recordError(messages.format("invalidCsv",
                        accountsFile.getFileName()));
                return this;
            }
            List<PollAccountDTO> accounts = CSVAccountUtil
                    .importList(accountsFile);
            if (accounts.size() == 0) {
                createListForm.recordError(messages.format("noAccountCsv",
                        accountsFile.getFileName()));
                return this;
            }
            newList.getPollAccountDTOs().addAll(accounts);
        }

        // Import LDAP des comptes
        if (accountsUrl != null) {
            List<PollAccountDTO> accounts = LDAPAccountUtil
                    .importList(accountsUrl);
            if (accounts.size() == 0) {
                createListForm.recordError(messages.format("noAccountLdap",
                        accountsUrl));
                return this;
            }
            newList.getPollAccountDTOs().addAll(accounts);
        }

        // Création de la nouvelle liste
        newList.setUserId(user.getId());
        newList.setId(serviceList.createPersonList(newList));

        // Sélection de la liste courante
        lists = serviceList.findPersonListByUser(user.getId());
        for (PersonListDTO list : lists) {
            if (list.getId().equals(newList.getId())) {
                selectedList = list;
            }
        }

        return this;
    }

    /**
     * Méthode appelée à la suppression d'une liste.
     */
    Object onActionFromDeleteList() {
        if (selectedList != null) {
            for (PersonListDTO dto : lists) {
                if (dto.getId().equals(selectedList.getId())) {
                    if (serviceList.deletePersonList(selectedList.getId())) {
                        feedback.addInfo(messages.format("listDeleted",
                                selectedList.getName()));
                    } else {
                        feedback.addError(messages.format("listNotDeleted",
                                selectedList.getName()));
                    }
                }
            }
        }
        selectedList = null;
        return this;
    }

    /**
     * Méthode appelée à la création d'un compte.
     */
    Object onSuccessFromCreateAccountForm() {
        for (PollAccountDTO dto : selectedList.getPollAccountDTOs()) {
            if (dto.getVotingId().equals(newAccount.getVotingId())) {
                createAccountForm.recordError(messages.format("accountExists",
                        newAccount.getVotingId()));
            }
        }
        if (!createAccountForm.getHasErrors()) {
            newAccount.setPersonListId(selectedList.getId());
            selectedList.getPollAccountDTOs().add(newAccount);
            serviceList.updatePersonList(selectedList);
            selectedList = serviceList.findPersonListById(selectedList.getId());
        }
        return listsZone.getBody();
    }

    /**
     * Méthode appelée à la suppression d'un compte.
     */
    Object onActionFromDeleteAccount(String accountId) {
        Iterator<PollAccountDTO> it = selectedList.getPollAccountDTOs()
                .iterator();
        while (it.hasNext()) {
            if (accountId.equals(it.next().getId())) {
                it.remove();
            }
        }
        serviceList.updatePersonList(selectedList);
        servicePollAccount.deletePollAccount(accountId);
        selectedList = serviceList.findPersonListById(selectedList.getId());
        return listsZone.getBody();
    }

    /**
     * Récupération des exceptions du champs d'upload de fichier.
     */
    Object onUploadException(FileUploadException ex) {
        createListForm.recordError("Upload exception: " + ex.getMessage());
        return this;
    }

    /** Retourne vrai s'il n'existe aucune liste */
    public boolean isListsNull() {
        return lists == null || lists.size() == 0;
    }

    /** Retourne vrai si la liste sélectionnée est vide */
    public boolean isSelectedListNull() {
        return selectedList == null;
    }

    /**
     * Méthode d'initialisation des listes de favoris.
     */
    void onActivate() {
        evenOdd = new EvenOdd();
        if (userExists) {
            lists = serviceList.findPersonListByUser(user.getId());
        }
        newList = new PersonListDTO();
        newAccount = new PollAccountDTO();
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };

        listModel = beanModelSource.createDisplayModel(PollAccountDTO.class,
                componentResources.getMessages());
        listModel.add("functions", null);
        listModel.include("votingId", "email", "functions");
    }
}