/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.fileupload.FileUploadException;
import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Mixins;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Select;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.annotations.Symbol;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.json.JSONObject;
import org.apache.tapestry5.upload.services.UploadSymbols;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PersonListDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.services.ServiceList;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.ui.base.ContextLink;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.components.FeedContextLink;
import org.chorem.pollen.ui.components.ImageContextLink;
import org.chorem.pollen.ui.data.GenericSelectModel;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollHelper;
import org.chorem.pollen.ui.data.PollStep;
import org.chorem.pollen.ui.data.uio.DateChoiceUIO;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.ui.utils.FeedUtil;
import org.chorem.pollen.ui.utils.ImageUtil;
import org.chorem.pollen.ui.utils.UnitConverter;
import org.slf4j.Logger;

/**
 * Classe de la page de création d'un sondage.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: PollCreation.java 2798 2009-11-13 17:41:42Z fdesbois $
 */
@IncludeStylesheet("context:css/pollCreation.css")
public class PollCreation {

    @Inject
    private Logger log;

    /** Étape courante du formulaire */
    @Persist
    private PollStep step;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    @Property
    private String siteURL;

    /** Date de début du sondage (utilisée pour la validation). */
    private Date beginDateValidation;

    /** Date de début d'ajout des choix (utilisée pour la validation). */
    private Date beginChoiceDateValidation;

    /** Mixin de selection de liste de favoris */
    @SuppressWarnings("unused")
    @Component(parameters = { "event=change",
            "onCompleteCallback=literal:onCompleteZoneUpdate" })
    @Mixins( { "ck/OnEvent" })
    private Select listSelect;

    @InjectComponent
    private Zone pollCreationZone;

    /**
     * Sondage créé par le formulaire
     */
    @Property
    @Persist
    private PollDTO poll;

    /**
     * Sondage copié pour créer un nouveau sondage
     */
    @Persist
    private PollDTO oldPoll;

    /**
     * Sondage copié existe.
     */
    private boolean oldPollExists = false;

    /**
     * variable utilisée pour déterminer si le bouton cliqué mène à l'étape
     * suivante
     */
    private boolean noStepSubmitSelected = false;

    /**
     * variable utilisée pour déterminer si le bouton d'ajout de choix a été
     * cliqué
     */
    @Persist
    private boolean addChoiceSelected;

    /**
     * variable utilisée pour déterminer le groupe à supprimer
     */
    private int deleteGroupId = -1;

    /**
     * variable utilisée pour déterminer s'il y a eu une exception lors de
     * l'upload des images
     */
    @Persist
    private boolean uploadExceptionCatched;

    /** variable utilisée pour l'affichage de la choiceNbCheckBox */
    @Property
    @Persist
    private boolean choiceNbCheckBox;

    /** variable utilisée pour l'affichage de la notificationCheckBox */
    @Property
    @Persist
    private boolean notificationCheckBox;

    /** variable utilisée pour l'affichage de la reminderCheckBox */
    @Property
    @Persist
    private boolean reminderCheckBox;

    /** Règle de notification de vote */
    @Property
    @Persist
    private PreventRuleDTO notificationPreventRule;

    /** Règle de notification pour le rappel des votants */
    @Property
    @Persist
    private PreventRuleDTO reminderPreventRule;

    /**
     * Groupes de votants à créer
     */
    @Property
    @Persist
    private List<VotingListDTO> votingLists;

    /**
     * Objet utilisé dans la boucle de parcours des listes de votants
     */
    @Property
    private VotingListDTO votingList;

    /**
     * Objet utilisé dans la boucle de parcours de la liste de votants courante
     */
    @SuppressWarnings("unused")
    @Property
    private PollAccountDTO votingListPerson;

    /**
     * Nombre de votants affichés initialement
     */
    @Property
    private int nbVotingListPersons = 5;

    /**
     * Liste modifiée actuellement
     */
    @Persist
    private int currentList;

    /**
     * Listes de favoris de l'utilisateur.
     */
    @Property
    @Persist
    private GenericSelectModel<PersonListDTO> personLists;

    /**
     * Liste de favoris sélectionnée.
     */
    @Property
    private PersonListDTO personList;

    /**
     * Nombre de choix affichés initialement
     */
    @Property
    private int nbChoices = 5;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private ChoiceDTO choice;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private DateChoiceUIO dateTypeChoice;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private ImageChoiceUIO imgTypeChoice;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<ChoiceDTO> choices;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<DateChoiceUIO> dateTypeChoices;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<ImageChoiceUIO> imgTypeChoices;

    /**
     * Contexte pour l'upload des images (parametres definis dans .tml)
     */
    @InjectComponent
    private ImageContextLink imgContext;

    /**
     * Contexte pour la gestion du flux RSS
     */
    @InjectComponent
    private ContextLink feedContext;

    /**
     * Formulaire de création de sondage
     */
    @Component(id = "pollCreationForm")
    private Form pollCreationForm;
    @Component(id = "choicesCreationForm")
    private Form choicesCreationForm;

    @InjectPage
    private CreationValidation creationValidation;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    @Inject
    private PropertyAccess _propertyAccess;

    @Inject
    private ComponentResources resources;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /** Tailles maximales des fichiers uploadés */
    @Inject
    @Symbol(UploadSymbols.FILESIZE_MAX)
    private int fileSizeMax;
    @Inject
    @Symbol(UploadSymbols.REQUESTSIZE_MAX)
    private int requestSizeMax;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    @Inject
    private Logger logger;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceList serviceList;

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape suivante de la
     * création de sondage.
     */
    @Log
    Object onSuccessFromPollCreationForm() {
        if (log.isDebugEnabled()) {
            log.debug("Step : " + step);
        }
        switch (step) {
        case POLL:
            adaptStepPoll();
            step = PollStep.OPTIONS;
            break;
        case OPTIONS:
            adaptStepOptions();
            if (isFreePoll()) {
                step = PollStep.CHOICES;
            } else {
                step = PollStep.LISTS;
            }
            break;
        case LISTS:
            if (deleteGroupId >= 0) {
                votingLists.remove(deleteGroupId);
            }
            if (!noStepSubmitSelected) {
                step = PollStep.CHOICES;
            }
            break;
        case CHOICES:
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        // Cas particulier, autre formulaire pour les choix
        if (step.equals(PollStep.CHOICES)) {
            return choicesCreationForm;
        }
        return pollCreationForm;
    }

    /**
     * Méthode appelée que le formulaire soit valide ou non. Il est nécessaire
     * de la redéfinir pour qu'en cas d'erreur de validation, la zone soit tout
     * de même mise à jour pour afficher l'erreur.
     */
    @Log
    Object onSubmitFromPollCreationForm() {
        return pollCreationForm;
    }

    /**
     * Méthode appelée lorsqu'on souhaite valider la création du sondage.
     */
    Object onSuccessFromChoicesCreationForm() {
        if (!addChoiceSelected) {

            // Préparation et création du sondage
            if (!preparePoll()) {
                return this;
            }
            createPoll();

            creationValidation.setPoll(poll);
            return creationValidation;
        }

        return this;
    }

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape précédente de la
     * création de sondage.
     */
    Object onPrevious() {
        switch (step) {
        case OPTIONS:
            step = PollStep.POLL;
            break;
        case LISTS:
            step = PollStep.OPTIONS;
            break;
        case CHOICES:
            if (isFreePoll()) {
                step = PollStep.OPTIONS;
            } else {
                step = PollStep.LISTS;
            }
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationForm; // Pas de gestion du formulaire des choix car derniere etape
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginDate.
     *
     * @throws ValidationException
     */
    void onValidateFromBeginDate(Date value) throws ValidationException {
        beginDateValidation = value;
        if (value != null && value.before(new Date())) {
            throw new ValidationException(messages.get("beginDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * endDate.
     *
     * @throws ValidationException
     */
    void onValidateFromEndDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.before(beginDateValidation)) {
            throw new ValidationException(messages.get("endDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginChoiceDate.
     *
     * @throws ValidationException
     */
    void onValidateFromBeginChoiceDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }
        beginChoiceDateValidation = value;
        if (value != null && value.after(beginDateValidation)) {
            throw new ValidationException(messages
                    .get("beginChoiceDate-validate"));
        }
    }
    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * endChoiceDate.
     *
     * @throws ValidationException
     */
    void onValidateFromEndChoiceDate(Date value) throws ValidationException {
        if (beginChoiceDateValidation == null) {
            beginChoiceDateValidation = new Date();
        }

        if (value != null && value.before(beginChoiceDateValidation)) {
            throw new ValidationException(messages.get("endChoiceDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation des
     * listes de votants.
     *
     * @throws ValidationException
     */
    void onValidateFormFromPollCreationForm() throws ValidationException {
        if (noStepSubmitSelected) {
            return;
        }

        // Validation des votants
        if (step == PollStep.LISTS) {
            int nbListEqual = 0;
            int nbEqual = 0;
            int nbNotNull = 0;

            // Repérage des doublons (listes)
            for (VotingListDTO list1 : votingLists) {
                for (VotingListDTO list2 : votingLists) {
                    if (list1.getName().equals(list2.getName())) {
                        nbListEqual++;
                    }
                }
            }

            if (nbListEqual > votingLists.size()) {
                throw new ValidationException(messages.get("lists-validate"));
            }

            // Repérage des doublons (votants)
            for (VotingListDTO list1 : votingLists) {
                int nbLocalNotNull = 0;
                for (PollAccountDTO account1 : list1.getPollAccountDTOs()) {
                    if (account1.getVotingId() != null
                            && account1.getVotingId() != "") {
                        nbNotNull++;
                        nbLocalNotNull++;

                        // comparaison avec les autres votants
                        for (VotingListDTO list2 : votingLists) {
                            for (PollAccountDTO account2 : list2
                                    .getPollAccountDTOs()) {
                                if (account2.getVotingId() != null
                                        && account1.getVotingId() != "") {
                                    if (account1.getVotingId().equals(
                                            account2.getVotingId())) {
                                        nbEqual++;
                                    }
                                }
                            }
                        }
                    }
                }
                if (nbLocalNotNull == 0) {
                    throw new ValidationException(messages
                            .get("noList-validate"));
                }
            }

            if (logger.isDebugEnabled()) {
                logger.debug("Votants (equal/notNull) : " + nbEqual + "/"
                        + nbNotNull);
            }
            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("list-validate"));
            }
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation de la
     * liste de choix.
     *
     * @throws ValidationException
     */
    void onValidateFormFromChoicesCreationForm() throws ValidationException {
        int nbEqual = 0;
        int nbNotNull = 0;

        // Validation des choix
        if (step == PollStep.CHOICES) {

            // Repérage des doublons (type Texte)
            if (isTextChoices()) {
                for (ChoiceDTO choice1 : choices) {
                    if (choice1.getName() != null) {
                        nbNotNull++;
                        for (ChoiceDTO choice2 : choices) {
                            if (choice2.getName() != null) {
                                if (choice1.getName().equals(choice2.getName())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
                // Repérage des doublons (type Date)
            } else if (isDateChoices()) {
                for (DateChoiceUIO choice1 : dateTypeChoices) {
                    if (choice1.getDate() != null) {
                        nbNotNull++;
                        for (DateChoiceUIO choice2 : dateTypeChoices) {
                            if (choice2.getDate() != null) {
                                if (choice1.getDate().equals(choice2.getDate())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
                // Repérage des doublons (type Image)
            } else if (isImgChoices()) {
                for (ImageChoiceUIO choice1 : imgTypeChoices) {
                    if (choice1.getImg() != null) {
                        nbNotNull++;
                        for (ImageChoiceUIO choice2 : imgTypeChoices) {
                            if (choice2.getImg() != null) {
                                if (choice1.getImg().getFileName().equals(
                                        choice2.getImg().getFileName())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
            }

            if (nbNotNull == 0) {
                throw new ValidationException(messages.get("noChoice-validate"));
            }
            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("choice-validate"));
            }
        }
    }

    /** Echec lors du téléchargement */
    Object onUploadException(FileUploadException e) {
        logger.error(e.getMessage());
        String fSize = UnitConverter.getFormattedFileSize(fileSizeMax);
        String rSize = UnitConverter.getFormattedFileSize(requestSizeMax);
        feedback.addError(messages.format("uploadError", fSize, rSize));
        uploadExceptionCatched = true;
        return this;
    }

    /**
     * Réinitialisation des options en fonction des autres options.
     */
    private void adaptStepOptions() {
        if (poll.isAnonymous()) {
            poll.setAnonymousVoteAllowed(true);
        }
        if (poll.isContinuousResults()) {
            poll.setPublicResults(true);
        }
    }

    /**
     * Réinitialisation des listes de votants en fonction du type de sondage.
     */
    @Log
    private void adaptStepPoll() {

        // Sondage libre : suppression de toutes les listes
        if (isFreePoll()) {
            votingLists.clear();
        }

        // Sondage restreint : suppression des listes supplémentaires
        else if (isRestrictedPoll() && votingLists.size() > 1) {
            for (int i = 1; i < votingLists.size(); i++) {
                votingLists.remove(i);
            }
            currentList = 0;
        }

        // Sondage non libre : création d'une liste initiale
        else if (votingLists.isEmpty()) {
            votingList = new VotingListDTO();
            for (int i = 0; i < nbVotingListPersons; i++) {
                votingList.getPollAccountDTOs().add(new PollAccountDTO());
            }
            votingLists.add(votingList);
            currentList = getVotingListIndex();
        }

        if (poll.getBeginDate() != null) {
            poll.setEndChoiceDate(poll.getBeginDate());
        }
    }

    /**
     * Préparation du sondage.
     */
    @Log
    private boolean preparePoll() {

        // Ajout de l'identifiant du créateur
        if (userExists) {
            poll.setUserId(user.getId());
        }

        // Ajout des règles de notification au sondage
        if (notificationCheckBox) {
            poll.getPreventRuleDTOs().add(notificationPreventRule);
        }
        if (reminderCheckBox) {
            poll.getPreventRuleDTOs().add(reminderPreventRule);
        }

        // Ajout des groupes de votants au sondage
        // après suppression des entrées vides
        if (!isFreePoll()) {
            for (VotingListDTO list : votingLists) {
                Iterator<PollAccountDTO> it = list.getPollAccountDTOs()
                        .iterator();
                while (it.hasNext()) {
                    if (it.next().getVotingId() == null) {
                        it.remove();
                    }
                }
            }
            poll.setVotingListDTOs(votingLists);
        }

        // Ajout des choix au sondage
        if (isTextChoices()) {
            for (ChoiceDTO choice : choices)
                if (choice.getName() != null) {
                    choice.setValidate(true);
                    poll.getChoiceDTOs().add(choice);
                }
        } else if (isDateChoices()) {
            for (DateChoiceUIO choice : dateTypeChoices)
                if (choice.getDate() != null) {
                    choice.setValidate(true);
                    choice.setName(String.valueOf(choice.getDate().getTime()));
                    poll.getChoiceDTOs().add(choice);
                }
        } else if (isImgChoices()) {
            for (ImageChoiceUIO imgChoice : imgTypeChoices) {
                if (imgChoice.getImg() != null) {
                    logger.debug("Image: " + imgChoice.getImg().getFileName()
                            + ", type: " + imgChoice.getImg().getContentType());
                    if (imgChoice.getImg().getContentType().contains("image")
                            || imgChoice.getImg().getContentType().contains(
                                    "IMAGE")) {
                        imgChoice.setValidate(true);
                        imgChoice.setName(imgChoice.getImg().getFileName()
                                .replace(' ', '_'));
                        poll.getChoiceDTOs().add(imgChoice);
                    } else {
                        return false;
                    }
                }
            }
        }

        // Retouche des attributs dépendants l'un de l'autre
        if (poll.getBeginDate() == null) {
            poll.setBeginDate(new Date());
        }
        if (poll.isAnonymous()) {
            poll.setAnonymousVoteAllowed(true);
        }
        if (poll.isContinuousResults()) {
            poll.setPublicResults(true);
        }
        if (poll.getMaxChoiceNb() < 1
                || poll.getMaxChoiceNb() > poll.getChoiceDTOs().size()) {
            poll.setMaxChoiceNb(poll.getChoiceDTOs().size());
        }

        return true;
    }

    /**
     * Création du sondage.
     */
    private void createPoll() {

        // Création du sondage
        poll.setId(servicePoll.createPoll(poll));

        if (poll.getId() != null) {

            // Création des images
            if (poll.getChoiceType() == ChoiceType.IMAGE) {
                File dir = imgContext.getImageDir();
                ImageUtil.saveImages(imgTypeChoices, dir);
            }

            // Mise à jour du sondage
            poll = servicePoll.findPollById(poll.getId());

            // Mise à jour du flux Atom et envoi d'un mail de confirmation
            addFeedEntry();
            sendMailNotification();
        }
    }

    /** Ajout d'une entrée dans le flux de syndication */
    private void addFeedEntry() {
        PollAccountDTO creator = servicePollAccount.findPollAccountById(poll
                .getCreatorId());
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        File feedFile = feedContext.getFile(poll.getPollId());

        FeedUtil.createFeed(feedFile, "atom_1.0", messages.format(
                "pollFeed_title", poll.getTitle()), siteURL, messages.format(
                "pollFeed_desc", poll.getDescription()));
        FeedUtil.feedFeed(feedFile, messages.format("pollFeed_createTitle",
                creator.getVotingId()), voteURL, messages
                .get("pollFeed_createContent"));
    }

    /** Envoi du mail de notification */
    private void sendMailNotification() {
        PollAccountDTO creator = servicePollAccount.findPollAccountById(poll
                .getCreatorId());
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        String modifURL = siteURL + "poll/Modification/" + poll.getPollId()
                + ":" + creator.getAccountId();
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mail au créateur
        if (poll.getCreatorEmail() != null) {
            data.put("to", poll.getCreatorEmail());
            data.put("title", messages.format("creatorEmail_subject", poll
                    .getTitle()));
            data.put("msg", messages.format("creatorEmail_msg",
                    poll.getTitle(), voteURL, modifURL));

            PreventRuleManager.emailAction(data);
        }

        // Mails aux votants
        for (VotingListDTO list : poll.getVotingListDTOs()) {
            for (PollAccountDTO account : list.getPollAccountDTOs()) {
                if (account.getEmail() != null) {
                    String accountVoteURL = voteURL + ":"
                            + account.getAccountId();

                    data.put("to", account.getEmail());
                    data.put("title", messages.format("votingEmail_subject",
                            poll.getTitle()));
                    data
                            .put("msg", messages.format("votingEmail_msg", poll
                                    .getTitle(), account.getVotingId(),
                                    accountVoteURL));

                    PreventRuleManager.emailAction(data);
                }
            }
        }
    }

    public String getChoiceDateDisplay() {
        return poll.isChoiceAddAllowed() ? "display: block;" : "display: none;";
    }

    public String getChoiceNbDisplay() {
        return choiceNbCheckBox ? "display: block;" : "display: none;";
    }

    public String getNotificationDisplay() {
        return notificationCheckBox ? "display: block;" : "display: none;";
    }

    public String getReminderDisplay() {
        return reminderCheckBox ? "display: block;" : "display: none;";
    }

    /** Retourne l'index de la liste courante */
    public int getVotingListIndex() {
        return votingLists.indexOf(votingList);
    }

    /** Retourne le numéro de la liste courante (index+1) */
    public int getVotingListNumber() {
        return votingLists.indexOf(votingList) + 1;
    }

    /** Retourne la chaîne correspondant à l'étape courante */
    public String getStepLegend() {
        Integer index = step.getIndex();

        // corrections selon le type de sondage
        if (isFreePoll()) {
            if (step == PollStep.CHOICES) {
                index--;
            }
        }

        // mise en forme du message
        switch (step) {
        case POLL:
            return messages.format("pollLegend", index);
        case OPTIONS:
            return messages.format("optionsLegend", index);
        case LISTS:
            return messages.format("listsLegend", index);
        case CHOICES:
            return messages.format("choicesLegend", index);
        default:
            return "";
        }
    }

    /** Retourne la classe CSS correspondant au groupe courant */
    public String getCurrentListClass() {
        if (votingLists.size() > 1 && getVotingListIndex() == currentList) {
            return "currentGroupDiv";
        }
        return "groupDiv";
    }

    /**
     * Retourne s'il existe plusieurs groupes.
     */
    public boolean isSeveralGroups() {
        return votingLists.size() > 1;
    }

    public boolean isInPoll() {
        return step == PollStep.POLL;
    }

    public boolean isInLists() {
        return step == PollStep.LISTS;
    }

    public boolean isInChoices() {
        return step == PollStep.CHOICES;
    }

    public boolean isInOptions() {
        return step == PollStep.OPTIONS;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.CONDORCET;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextChoices() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateChoices() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImgChoices() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    /**
     * Méthode appelée à la sélection d'une liste de votants. Le mixin
     * ck/OnEvent ne permet pas de retourner le contenu d'une zone. Il faut donc
     * passer par une fonction JavaScript pour activer un event link. Un lien
     * est créé. Il sera retourné à la fonction JavaScript onCompleteCallback
     * pour mettre à jour la zone.
     *
     * @return un JSONObject contenant l'url de l'évènement mettant à jour la
     *         zone.
     */
    public JSONObject onChangeFromListSelect(String value) {
        if (!"".equals(value)) {
            personList = serviceList.findPersonListById(value);

            // Copie des personnes de la liste de favoris dans la liste de votants
            for (PollAccountDTO account : personList.getPollAccountDTOs()) {
                account.setId("");
                account.setPersonListId("");
            }

            votingLists.get(currentList).setPollAccountDTOs(
                    personList.getPollAccountDTOs());
        }
        return createParamsForCallback();
    }

    /**
     * Méthode appelée à la sélection d'un type de choix. Le mixin ck/OnEvent ne
     * permet pas de retourner le contenu d'une zone. Il faut donc passer par
     * une fonction JavaScript pour activer un event link. Un lien est créé. Il
     * sera retourné à la fonction JavaScript onCompleteCallback pour mettre à
     * jour la zone.
     *
     * @return un JSONObject contenant l'url de l'évènement mettant à jour la
     *         zone.
     */
    public JSONObject onChangeFromChoiceType(String value) {
        poll.setChoiceType(ChoiceType.valueOf(value));
        return createParamsForCallback();
    }

    /**
     * Création d'un JSONObject contenant un identifiant de zone et une url pour
     * un évènement.
     *
     * @return un JSONObject contenant un zoneId et une url.
     */
    private JSONObject createParamsForCallback() {
        JSONObject json = new JSONObject();
        Link link = resources.createEventLink("updatePollCreationZone");
        json.put("link", link.toAbsoluteURI());
        json.put("zoneId", "pollCreationZone");
        return json;
    }

    /**
     * Méthode appelée par le callback JavaScript pour mettre à jour la zone.
     *
     * @return le contenu mis à jour de la zone.
     */
    public Object onUpdatePollCreationZone() {
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée pour l'ajout d'une ligne supplémentaire dans le
     * formulaire des listes de votants.
     */
    void onSelectedFromAddPerson(int i) {
        votingLists.get(i).getPollAccountDTOs().add(new PollAccountDTO());
        noStepSubmitSelected = true;
    }

    /**
     * Méthode appelée pour l'ajout d'un groupe dans le formulaire des listes de
     * votants.
     */
    void onSelectedFromAddGroup() {
        votingList = new VotingListDTO();
        for (int i = 0; i < nbVotingListPersons; i++) {
            votingList.getPollAccountDTOs().add(new PollAccountDTO());
        }
        votingLists.add(votingList);
        currentList = getVotingListIndex();
        noStepSubmitSelected = true;
    }

    /**
     * Méthode appelée lors de la suppression d'un groupe dans le formulaire des
     * listes de votants.
     */
    void onSelectedFromDeleteGroup(int i) {
        //votingLists.remove(i);
        if (currentList == i) {
            currentList = votingLists.size() - 1;
        } else if (currentList > i) {
            currentList--;
        }
        deleteGroupId = i;
        noStepSubmitSelected = true;
    }

    /**
     * Méthode appelée à la selection d'un groupe dans le formulaire des listes
     * de votants.
     */
    void onSelectedFromEditGroup(int i) {
        currentList = i;
        noStepSubmitSelected = true;
    }

    /**
     * Méthode appelée pour l'ajout d'une ligne supplémentaire dans le
     * formulaire des choix.
     */
    void onSelectedFromAddChoice() {
        if (poll.getChoiceType() == ChoiceType.DATE) {
            dateTypeChoices.add(new DateChoiceUIO());
        } else if (poll.getChoiceType() == ChoiceType.IMAGE) {
            imgTypeChoices.add(new ImageChoiceUIO());
        } else {
            choices.add(new ChoiceDTO());
        }
        addChoiceSelected = true;
    }

    /** Retourne vrai si des listes de favoris existent */
    public boolean isPersonListsExists() {
        return personLists != null && !personLists.getList().isEmpty();
    }

    /**
     * Activation de la page
     */
    void onActivate(String id) {

        // Si un sondage est fourni (copie de sondage)
        if (id != null && !"".equals(id)) {

            // Réinitialisation des variables de session
            // Si l'ancien sondage n'existe pas ou est différent de celui fourni
            if (oldPoll == null || !id.equals(oldPoll.getPollId())) {
                oldPoll = servicePoll.findPollByPollId(id);
                if (oldPoll != null) {
                    initWithExistingPoll(oldPoll);
                    oldPoll = null;
                    oldPollExists = true;
                }
            }
        }
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };

        if (!addChoiceSelected && !uploadExceptionCatched
                && !choicesCreationForm.getHasErrors()) {
            step = PollStep.POLL;
            if (!oldPollExists) {
                oldPoll = null;
                initPoll();
            }
            initPersonLists();
        }

        addChoiceSelected = false;
        uploadExceptionCatched = false;
    }

    /**
     * Initialisation du sondage.
     */
    private void initPoll() {

        // Initialisation du sondage
        poll = new PollDTO();
        if (userExists) {
            poll.setCreatorId(user.getLogin());
            poll.setCreatorEmail(user.getEmail());
        }

        // Initialisation des règles de notification
        notificationPreventRule = new PreventRuleDTO("vote", 0, true,
                PreventRuleManager.EMAIL_ACTION);
        reminderPreventRule = new PreventRuleDTO("rappel", 0, false,
                PreventRuleManager.EMAIL_ACTION);

        notificationCheckBox = false;
        reminderCheckBox = false;
        choiceNbCheckBox = false;

        // Initialisation des choix
        choices = new ArrayList<ChoiceDTO>();
        dateTypeChoices = new ArrayList<DateChoiceUIO>();
        imgTypeChoices = new ArrayList<ImageChoiceUIO>();
        for (int i = 0; i < nbChoices; i++) {
            choices.add(new ChoiceDTO());
            dateTypeChoices.add(new DateChoiceUIO());
            imgTypeChoices.add(new ImageChoiceUIO());
        }

        // Initialisation des listes de votants
        votingLists = new ArrayList<VotingListDTO>();
    }

    /**
     * Initialisation du sondage à partir d'un sondage existant.
     */
    private void initWithExistingPoll(PollDTO oldPoll) {

        // Initialisation du sondage
        poll = PollHelper.getPoll(oldPoll);

        // Initialisation du créateur du sondage
        PollAccountDTO creator = servicePollAccount.findPollAccountById(oldPoll
                .getCreatorId());
        poll.setCreatorId(creator.getVotingId());
        poll.setCreatorEmail(creator.getEmail());

        // Initialisation des règles de notification
        notificationPreventRule = PollHelper
                .getNotificationPreventRule(oldPoll);
        notificationCheckBox = !"".equals(notificationPreventRule.getScope());
        reminderPreventRule = PollHelper.getReminderPreventRule(oldPoll);
        reminderCheckBox = !"".equals(reminderPreventRule.getScope());

        // Initialisation des choix
        choices = PollHelper.getTextChoices(oldPoll);
        dateTypeChoices = PollHelper.getDateChoices(oldPoll);
        imgTypeChoices = PollHelper.getImageChoices(oldPoll);

        // Initialisation des listes de votants
        votingLists = PollHelper.getVotingLists(oldPoll);
    }

    /**
     * Initialisation de la liste de favoris.
     */
    private void initPersonLists() {
        if (userExists) {
            List<PersonListDTO> _personLists = serviceList
                    .findPersonListByUser(user.getId());
            personLists = new GenericSelectModel<PersonListDTO>(_personLists,
                    PersonListDTO.class, "name", "id", _propertyAccess);
        }
    }
}
