package org.chorem.lima.ui.common;

/*
 * #%L
 * Lima :: Swing
 * $Id: FinancialTransactionTableModel.java 3768 2014-04-11 13:38:02Z dcosse $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/ui/common/FinancialTransactionTableModel.java $
 * %%
 * Copyright (C) 2008 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.utils.EntryComparator;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FinancialTransactionImpl;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.financialtransaction.AccountColumn;
import org.chorem.lima.ui.financialtransaction.CreditColumn;
import org.chorem.lima.ui.financialtransaction.DateColumn;
import org.chorem.lima.ui.financialtransaction.DebitColumn;
import org.chorem.lima.ui.financialtransaction.DescriptionColumn;
import org.chorem.lima.ui.financialtransaction.VoucherColumn;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;

/**
 * @author Sylvain Bavencoff <bavencoff@codelutin.com>
 */
public class FinancialTransactionTableModel extends TableModelWithGroup<Entry> {

    /** serialVersionUID. */
    private static final long serialVersionUID = -7495388454688562991L;

    protected static final Log log = LogFactory.getLog(FinancialTransactionTableModel.class);

    /** Service (just to update setValueAt(). */
    protected FinancialTransactionService financialTransactionService;

    public FinancialTransactionTableModel() {

        setComparator(new EntryComparator());

        financialTransactionService =
                LimaServiceFactory.getService(FinancialTransactionService.class);
    }

    @Override
    protected void initColumn() {

        addColumn(new DateColumn());
        addColumn(new VoucherColumn());
        addColumn(new AccountColumn());
        addColumn(new DescriptionColumn());
        addColumn(new DebitColumn());
        addColumn(new CreditColumn());


    }

    public void setTransactions(List<FinancialTransaction> transactions) {
        clear();
        for (FinancialTransaction transaction : transactions) {
            addAll(transaction.getEntry());
        }
    }

    public FinancialTransaction getTransactionAt(int row) {
        Entry entry = get(row);
        return entry.getFinancialTransaction();
    }

    public int indexOf(FinancialTransaction transaction) {
        int index = 0;
        Collection<Entry> entriesTransaction = transaction.getEntry();
        for (Entry entry : values) {
            if (!entriesTransaction.contains(entry)) {
                index++;
            } else {
                break;
            }
        }
        return index;
    }

    public boolean contains(FinancialTransaction transaction) {
        boolean result = false;
        for (Entry entry : transaction.getEntry()) {
            if (contains(entry)) {
                result = true;
                break;
            }
        }
        return  result;

    }

    /**
     * Insert new entry.
     *
     * @param entry entry to insert
     */
    public Entry addEntry(Entry entry) {
        Entry newEntry = null;
        FinancialTransaction transaction = entry.getFinancialTransaction();
        if (contains(transaction)) {
            newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(transaction);
            newEntry.setVoucher(entry.getVoucher());
            newEntry.setAccount(entry.getAccount());
            newEntry.setDescription(entry.getDescription());
            newEntry.setAmount(entry.getAmount());
            newEntry.setDebit(entry.getDebit());
            newEntry = financialTransactionService.createEntry(newEntry);

            transaction.addEntry(newEntry);
            financialTransactionService.updateFinancialTransaction(transaction);
            addValue(newEntry);
        }
        return newEntry;
    }

    /**
     * Delete selected row in table (could be transaction or entry).
     * <p/>
     * Called by model.
     *
     * @param row
     */
    public void removeTransaction(int row) {
        FinancialTransaction transaction = getTransactionAt(row);
        removeAll(transaction.getEntry());
        financialTransactionService.removeFinancialTransaction(transaction);
    }

    public void removeEntry(int row) {
        Entry entry = get(row);
        FinancialTransaction transaction = entry.getFinancialTransaction();
        if (transaction.sizeEntry() > 1) {
            financialTransactionService.removeEntry(entry);
            transaction.removeEntry(entry);
            remove(entry);
        } else {
            financialTransactionService.removeFinancialTransaction(transaction);
            removeAll(transaction.getEntry());
        }
    }

    public FinancialTransaction addTransaction(FinancialTransaction transaction) {
        FinancialTransaction newTransaction = new FinancialTransactionImpl();
        newTransaction.setEntryBook(transaction.getEntryBook());
        newTransaction.setTransactionDate(transaction.getTransactionDate());
        newTransaction = financialTransactionService.createFinancialTransaction(newTransaction);

        if (transaction.getEntry() == null || transaction.getEntry().isEmpty()) {
            Entry newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(newTransaction);
            newEntry = financialTransactionService.createEntry(newEntry);
            newTransaction.addEntry(newEntry);

            newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(newTransaction);
            newEntry = financialTransactionService.createEntry(newEntry);
            newTransaction.addEntry(newEntry);
        } else {
            for (Entry entry : transaction.getEntry()) {
                Entry newEntry = new EntryImpl();
                newEntry.setFinancialTransaction(newTransaction);
                newEntry.setVoucher(entry.getVoucher());
                newEntry.setAccount(entry.getAccount());
                newEntry.setDescription(entry.getDescription());
                newEntry.setAmount(entry.getAmount());
                newEntry.setDebit(entry.getDebit());
                newEntry = financialTransactionService.createEntry(newEntry);
                newTransaction.addEntry(newEntry);
            }
            financialTransactionService.updateFinancialTransaction(newTransaction);
        }

        addAll(newTransaction.getEntry());
        return newTransaction;
    }

    public BigDecimal getBalanceTransactionInRow(int row) {
        FinancialTransaction transaction = getTransactionAt(row);
        BigDecimal debit = transaction.getAmountDebit();
        BigDecimal credit = transaction.getAmountCredit();
        BigDecimal balance = debit.subtract(credit);
        return balance;
    }

    @Override
    public boolean sameGroup(Entry e1, Entry e2) {
        return e1.getFinancialTransaction().equals(e2.getFinancialTransaction());
    }

    public void updateEntry(Entry entry) {
        if (log.isDebugEnabled()) {
            log.debug("Update Entry");
        }
        financialTransactionService.updateEntry(entry);
    }

    public void updateTransaction(FinancialTransaction transaction) {
        if (log.isDebugEnabled()) {
            log.debug("Update transaction");
        }
        financialTransactionService.updateFinancialTransaction(transaction);
    }

    public void fireTransaction(FinancialTransaction transaction) {
        int firstRow = indexOf(transaction);
        int lastRow = firstRow + transaction.sizeEntry() - 1;
        fireTableRowsUpdated(firstRow, lastRow);
    }
}
