/*
 * #%L
 * Coser :: UI
 * 
 * $Id: Coser.java 924 2011-12-22 13:56:04Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/coser/tags/coser-1.2.2/coser-ui/src/main/java/fr/ifremer/coser/Coser.java $
 * %%
 * Copyright (C) 2010 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package fr.ifremer.coser;

import java.io.File;

import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import jaxx.runtime.context.DefaultApplicationContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.widget.SwingSession;

import com.bbn.openmap.MapBean;

import fr.ifremer.coser.services.CommandService;
import fr.ifremer.coser.services.ControlService;
import fr.ifremer.coser.services.ImportService;
import fr.ifremer.coser.services.ProjectService;
import fr.ifremer.coser.services.PublicationService;
import fr.ifremer.coser.services.WebService;
import fr.ifremer.coser.ui.CoserFrame;
import fr.ifremer.coser.ui.util.ErrorHelper;

/**
 * Coser main class.
 * 
 * @author chatellier
 * @version $Revision: 924 $
 * 
 * Last update : $Date: 2011-12-22 14:56:04 +0100 (Thu, 22 Dec 2011) $
 * By : $Author: echatellier $
 */
public class Coser {

    private static final Log log = LogFactory.getLog(Coser.class);

    /**
     * Coser main method.
     * 
     * @param args main args
     */
    public static void main(String... args) {

        // init config
        final CoserConfig coserConfig = new CoserConfig();
        try {
            coserConfig.parse(args);
        } catch (ArgumentsParserException ex) {
            if (log.isErrorEnabled()) {
                log.error("Cant' parse configuration", ex);
            }
        }

        // OpenMap sysout
        MapBean.suppressCopyright = true;

        // catch wall application exception
        launch(coserConfig);

        Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
            public void uncaughtException(Thread t, Throwable ex) {
                if (log.isErrorEnabled()) {
                    log.error("Global application exception", ex);
                }
                ErrorHelper errorHelper = new ErrorHelper(coserConfig);
                errorHelper.showErrorDialog(null, ex.getMessage(), ex);
            }
        });
    }

    /**
     * Display main UI.
     * 
     * @param coserConfig coser configuration
     */
    public static void launch(final CoserConfig coserConfig) {

        // to work in java webstart
        I18n.init(new DefaultI18nInitializer("coser-ui-i18n"), coserConfig.getLocale());

        // launch UI
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {

                // ne to be done in Swing EDT (otherwize, don't work on javawebstart)
                // declare new classloader
                ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
                CoserClassLoader coserClassLoader = new CoserClassLoader(currentClassLoader);
                Thread.currentThread().setContextClassLoader(coserClassLoader);
                // set coser xwork validation file directory in classloader
                coserClassLoader.setValidatorsDirectory(coserConfig.getValidatorsDirectory());

                // init LAF (from configuration)
                try {
                    String lafClassName = coserConfig.getLookAndFeel();
                    for (UIManager.LookAndFeelInfo laf : UIManager.getInstalledLookAndFeels()) {
                        if (laf.getClassName().equalsIgnoreCase(lafClassName)) {
                            UIManager.setLookAndFeel(laf.getClassName());
                        }
                    }
                } catch (Exception ex) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't init look and feel", ex);
                    }
                }

                // update font size (after laf)
                coserConfig.updateSwingFont();

                File coserConfigDirectory = new File(coserConfig.getUserConfigDirectory(), "coser");
                coserConfigDirectory.mkdirs();
                File mainFrameFile = new File(coserConfigDirectory, "session.xml");
                SwingSession session = new SwingSession(mainFrameFile, true);

                // define unique context global values
                DefaultApplicationContext context = new DefaultApplicationContext();
                context.setContextValue(coserConfig);
                context.setContextValue(session);
                context.setContextValue(new ProjectService(coserConfig));
                context.setContextValue(new ImportService(coserConfig));
                context.setContextValue(new CommandService(coserConfig));
                context.setContextValue(new ControlService(coserConfig));
                context.setContextValue(new PublicationService(coserConfig));
                context.setContextValue(new WebService(coserConfig));

                // init frame with session reloading
                CoserFrame frame = new CoserFrame(context);
                frame.setLocationRelativeTo(null);
                session.add(frame);

                frame.setVisible(true);
            }
        });
    }
}
