/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: FormManager.java 1589 2010-10-01 10:53:19Z sletellier $
 * $HeadURL: svn+ssh://chatellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.2.0/vradi-services/src/main/java/com/jurismarches/vradi/services/managers/FormManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.managers;

import static org.nuiton.i18n.I18n._;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.SessionStatus;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * Class containing the methods to manage the forms :
 * - creation, update, deletion, retrieving
 * - binding forms with clients whose queries are returning the forms
 *
 * @author schorlet
 * @version $Revision: 1589 $ $Date: 2010-10-01 12:53:19 +0200 (ven., 01 oct. 2010) $
 */
public class FormManager {
    
    private static final Log log = LogFactory.getLog(FormManager.class);

    protected WikittyProxy wikittyProxy;
    protected ThesaurusManager thesaurusManager;
    protected SearchManager searchManager;

    public FormManager(WikittyProxy wikittyProxy,
            ThesaurusManager thesaurusManager, SearchManager searchManager) {
        this.wikittyProxy = wikittyProxy;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    /**
     * Update forms and attach forms to specified thesaurus.
     * 
     * @param forms forms to save
     * @param thesaurus thesaurus to attach form to (set empty set to clear thesaurus, null = no thesaurus management)
     * @return forms saved
     * @throws VradiException
     */
    public List<Form> updateForms(List<Form> forms, Set<String> thesaurus) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateForms(forms)");
        }

        if (forms == null || forms.isEmpty()) {
            return forms;
        }

        List<Thesaurus> entitiesToSave = new ArrayList<Thesaurus>();
        for (Form form : forms) {
            if (log.isDebugEnabled()) {
                log.debug("updating form: " + form.getId());
            }

            Date now = new Date();
            if (form.getCreationDate() == null) {
                form.setCreationDate(now);
            } else {
                ModificationTag modificationTag = wikittyProxy.cast(form, ModificationTag.class);
                modificationTag.setLastModified(now);
            }

            // add specified thesaurus to form
            if (thesaurus != null) {

                // search for thesaurus containing current form
                // to remove it
                List<Thesaurus> attachedThesauruses = thesaurusManager.getThesaurusAttachedToForm(form);
                for (Thesaurus attachedThesaurus : attachedThesauruses) {
                    attachedThesaurus.removeAttachment(form.getId());
                    if (!entitiesToSave.contains(attachedThesaurus)) {
                        entitiesToSave.add(attachedThesaurus);
                    }
                }

                for (String thesaurusId : thesaurus) {
                    Thesaurus node = thesaurusManager.getThesaurus(thesaurusId);

                    if (node.getAttachment() == null
                            || !node.getAttachment().contains(form.getWikittyId())) {

                        node.addAttachment(form.getWikittyId());

                        if (!entitiesToSave.contains(node)) {
                            entitiesToSave.add(node);
                        }
                    }
                }
            }
        }

        // also save form
        List<Form> result = wikittyProxy.store(forms);

        if (!entitiesToSave.isEmpty()) {
            wikittyProxy.store(entitiesToSave);
        }

        return result;
    }

    public Session getLastCloseSession() {

        // Get last close session to search after this one
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);
        search.eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(VradiConstants.SessionStatus.SENT.getValue()))
                .criteria();

        Criteria criteria = search.criteria();
        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = wikittyProxy.findAllByCriteria(Session.class, criteria);
        if (result == null || result.size() == 0) {
            return null;
        }
        return result.getFirst();
    }

    public Session getLastOpenSession() throws VradiException {

        Session lastClose = getLastCloseSession();

        String formatedLastCloseDate = null;
        if (lastClose != null) {

            if (log.isDebugEnabled()) {
                log.debug("Last closed session : " + lastClose.getSessionDate());
            }
            try {
                formatedLastCloseDate = WikittyUtil.formatDate(lastClose.getSessionDate());
            } catch (ParseException e) {
                log.error("Cant parse date");
            }
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        if (formatedLastCloseDate != null) {
            search.ge(Session.FQ_FIELD_SESSION_SESSIONDATE,
                        formatedLastCloseDate);
        }
        
        search.or().eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.ACTIVE.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.STOPPED.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.ERROR.getValue()));

        Criteria criteria = search.criteria();

        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = wikittyProxy.findAllByCriteria(Session.class, criteria);
        Session session = null;
        if (result != null && result.size() > 0) {
            session = result.getFirst();
        }

        if (log.isDebugEnabled() && session != null){
            log.debug("Last sessions found : " + session.getWikittyId() + " - "
                    + session.getSessionDate() +" with status : "
                    + session.getStatus());
        }

        return session;
    }

    /**
     * Find all session for specific date.
     * 
     * @param sessionDate session date
     * @throws VradiException
     * @return all session for date
     */
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getSessions for date " + sessionDate.toString());
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        search.bw(Session.FQ_FIELD_SESSION_SESSIONDATE,
                getBeginOfDaySolrDate(sessionDate),
                getEndOfDaySolrDate(sessionDate));

        // Get result
        Criteria criteria = search.criteria();
        List<Session> allSession = wikittyProxy.findAllByCriteria(Session.class, criteria).getAll();

        if (log.isDebugEnabled()) {
            log.debug(allSession.size() + " sessions found");
        }

        return allSession;
    }

    /**
     * Retourne la date a minuit au format solr.
     * 
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getBeginOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtil.setMinTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Retourne la date a 23h59:59.999 au format solr.
     * 
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getEndOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtil.setMaxTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Crée une nouvelle session initialisée.
     * 
     * @return la session créée
     * @throws VradiException
     */
    public Session createNewSession() throws VradiException {

        // TODO EC20100909 autorisé la création de plusieurs session
        Session lastSession = getLastOpenSession();
        if (lastSession != null) {
            throw new VradiException(_("Already opened session on %tF", lastSession.getSessionDate()));
        }

        // Get all session for this date
        Date now = new Date();

        // TODO EC201009089 ca sert a rien de toutes les récuperer pour avoir le dernier numéro
        List<Session> sessionList = getSessions(now);

        // Calculate number af session for day
        int num = sessionList.size() + 1;

        // create new session
        if (log.isDebugEnabled()) {
            log.debug("No session found, creating a new one with num : " + num);
        }

        Session currentSession = new SessionImpl();
        currentSession.setSessionDate(now);
        currentSession.setNum(num);
        currentSession.setStatus(SessionStatus.ACTIVE.getValue());
        currentSession.setParagraph(VradiServiceConfiguration.getInstance().getSessionDefaultParagraph());

        currentSession = wikittyProxy.store(currentSession);

        return currentSession;
    }

    /**
     * Recherche tous les formulaires non lié et les lie à la session
     * demandée.
     * 
     * @param currentSession session a remplir
     * @return la session en argument
     * @throws VradiException
     */
    public Session bindForms(Session currentSession) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("bindForms " + currentSession.getSessionDate());
        }

        try {
            // un sending = un utilsateur et ses formulaires
            List<Sending> toSend = new ArrayList<Sending>(attachForms(currentSession));

            // Reload session to avoid getting wikitty obsolete error
            Session reloadedSession = wikittyProxy.restore(Session.class,
                        currentSession.getWikittyId());

            if (reloadedSession != null) {
                currentSession = reloadedSession;
            }

            // Throw exception if session is not active
            if (currentSession.getStatus() != VradiConstants.SessionStatus.ACTIVE.getValue()) {
                throw new VradiException("Session is not active, abording");
            }

            for (Sending s : toSend) {
                currentSession.addSending(s.getWikittyId());
            }

            // Store the session and sending
            currentSession = wikittyProxy.store(currentSession);
            wikittyProxy.store(toSend);

            if (log.isDebugEnabled()) {
                Set<String> sending = currentSession.getSending();
                int size = sending == null ? 0 : sending.size();

                log.debug("Saving session '" + currentSession.getSessionDate()
                        + "' number '" + currentSession.getNum()
                        + "' status '" + SessionStatus.getStatus(currentSession.getStatus()).getDescription()
                        + "' with '" + size
                        + "' sending");
            }

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't bind form : ", e);
            }
            throw new VradiException("Can't bind form : ", e);
        }
        return currentSession;
    }

    /**
     * Construit la liste des sending (un utilisateur et ses formulaires, soit
     * un mail).
     * 
     * @param currentSession session
     * @return les sending a traiter
     * @throws VradiException
     */
    protected Set<Sending> attachForms(Session currentSession) throws VradiException {

        // Only selected forms
        String status = getSelectionneStatus().getWikittyId();

        // List of the forms to bind ( = not yet bound)
        Set<Sending> toSend = new HashSet<Sending>();

        // recherche de tous les executeurs de requette (client, groupes...)
        // was all query make before
        //List<QueryMaker> queryMakers = searchManager.findQueryMakersWithQueries();
        List<Group> groups = searchManager.findGroupsWithQueries();

        // iterate on the group
        for (Group group : groups) {

            // List of the forms to bind ( = not yet bound)
            List<Form> formsToBind = new ArrayList<Form>();

            // normalement c'est pas null, sinon la requette findGroupsWithQueries
            // marche pas
            Set<String> queries = group.getQueries();

            for (String queryEntry : queries) {
                
                try {
                    // Get query
                    QueryBean queryBean = new QueryBean(queryEntry);
                    String query = queryBean.getQuery();

                    // Create filter
                    Search filter = Search.query();
                    filter.eq(Form.FQ_FIELD_INFOGENE_STATUS, status);

                    // Limited on older inscription date
                    Search filterDatePublication = filter.or();
                    Date olderInscriptionDate = getOlderInscriptionDate(group.getUser());
                    filterDatePublication.isNull(Form.FQ_FIELD_FORM_DATEPUB);
                    filterDatePublication.ge(Form.FQ_FIELD_FORM_DATEPUB, WikittyUtil.formatDate(olderInscriptionDate));

                    // Limited on date peremption
                    Search filterDatePeremption = filter.or();
                    filterDatePeremption.isNull(Form.FQ_FIELD_FORM_DATEPEREMPTION);
                    filterDatePeremption.ge(Form.FQ_FIELD_FORM_DATEPEREMPTION, WikittyUtil.formatDate(new Date()));

                    Criteria criteria = filter.criteria();

                    // create a new VradiFormPage to find all the forms corresponding to the query
                    FormPagedResult vradiFormPage = new FormPagedResult();

                    // Execute request
                    vradiFormPage = searchManager.findForms(query, criteria, vradiFormPage);

                    // Extract result
                    List<Form> resultForms = vradiFormPage.getFormsToShow();

                    if (log.isDebugEnabled()) {
                        log.debug("Query " + query + " return " + resultForms.size() + " forms");
                    }

                    formsToBind.addAll(resultForms);
                } catch (IOException ex) {
                    // pas grave on zape ?
                    if (log.isErrorEnabled()) {
                        log.error("Can't parse group query", ex);
                    }
                } catch (UnsupportedQueryException ex) {
                    // pas grave on zape ?
                    if (log.isErrorEnabled()) {
                        log.error("Can't execute group query", ex);
                    }
                } catch (ParseException eee) {
                    throw new VradiException("Cant format date : ", eee);
                }
            }

            // if some forms are to be bound, bind them
            if (!formsToBind.isEmpty()) {
                List<Sending> sendings = createAllSending(currentSession, group, formsToBind, true, false);
                toSend.addAll(sendings);
            }
        }

        return toSend;
    }

    /**
     * Retourne la plus ancienne date du query maker si c'est une date
     * ou des user gu groupe si c'est un groupe.
     * 
     * @param queryMaker queryMaker
     * @return
     */
    protected Date getOlderInscriptionDate(Set<String> usersIds) {
        Date result = new Date();

        for (String userId : usersIds) {
            User user = wikittyProxy.restore(User.class, userId);

            Date beginSearchDate = user.getBeginSearchDate();
            if (beginSearchDate != null && result.after(DateUtil.setMinTimeOfDay(beginSearchDate))){
                result = beginSearchDate;
            }
            if (log.isDebugEnabled()){
                log.debug("The older inscription date for user " + user.getName() + " is " + result) ;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("The older inscription date " + result) ;
        }
        return result;
    }

    public List<Sending> createAllSending(Session currentSession, Group group, List<Form> formsToBind, boolean removeAlreadyBind) {
        return createAllSending(currentSession, group, formsToBind, removeAlreadyBind, true); 
    }

    protected List<Sending> createAllSending(Session currentSession, Group group, List<Form> formsToBind, boolean removeAlreadyBind, boolean save) {
        List<Sending> result = new ArrayList<Sending>();

        List<User> users = wikittyProxy.restore(User.class, new ArrayList<String>(group.getUser()));

        for (User user : users) {
            Sending sending = createUserSending(currentSession, user, formsToBind, removeAlreadyBind, save);
            if (sending != null) {
                result.add(sending);
            }
        }
        return result;
    }

    public Sending createUserSending(Session currentSession, User user, List<Form> formsToBind, boolean removeAlreadyBind) {
        return createUserSending(currentSession, user, formsToBind, removeAlreadyBind, true);
    }

    protected Sending createUserSending(Session currentSession, User user, List<Form> formsToBind, boolean removeAlreadyBind, boolean save) {

        currentSession = wikittyProxy.restore(Session.class, currentSession.getWikittyId());

        // Create or find current sending for queryMaker
        Sending sending = findExistingOrCreateSending(currentSession, user);

        sending.setUser(user.getWikittyId());
        if (log.isDebugEnabled()) {
            log.debug("Add user " + user.getWikittyId());
        }

        List<Form> finalFormsToBind = new ArrayList<Form>(formsToBind);

        if (removeAlreadyBind) {

            // Remove already bound forms with queryMaker
            finalFormsToBind = removeAlreadyBound(currentSession, finalFormsToBind, user);

            // Get user begin search date
            Date beginSearchDate = user.getBeginSearchDate();

            // Check date
            for (Form f : new ArrayList<Form>(finalFormsToBind)) {

                // Using last modification date
                Date formCreationDate = f.getCreationDate();

                // This test never happens, new date is set by default
                // If not enable, not attach forms
                if (!user.getEnable() || formCreationDate.before(beginSearchDate)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Removing form '" +
                                f.getObjet() +
                                "' because date of creation (" +
                                formCreationDate +
                                ") is before user begin search date (" +
                                beginSearchDate + ") for user : " +
                                user.getName());
                    }

                    finalFormsToBind.remove(f);
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("Adding " + finalFormsToBind.size() + " forms to sending");
            }

        }

        // Attach forms to sending.
        for (Form f : finalFormsToBind) {
            String fid = f.getWikittyId();
            sending.removeForm(fid);
            sending.addForm(fid);
        }

        Set<String> formsAttached = sending.getForm();

        // If not empty
        if (formsAttached != null && !formsAttached.isEmpty()) {
            //restore session before adding sendings and storing to avoid
            //obsolete wikitty exception
            currentSession = wikittyProxy.restore(Session.class,currentSession.getWikittyId());

            currentSession.addSending(sending.getWikittyId());

            // store datas
            if (save) {
                wikittyProxy.store(currentSession);
                sending = wikittyProxy.store(sending);
            }
            return sending;
        }
        return null;
    }

    public List<Sending> removeAllSending(Session session, Form form, Group group) throws VradiException {
        List<Sending> result = new ArrayList<Sending>();

        List<User> users = wikittyProxy.restore(User.class, new ArrayList<String>(group.getUser()));
        for (User user : users) {
            result.add(removeSending(session, form, user));
        }
        return result;
    }

    public Sending removeSending(Session session, Form form, User user) throws VradiException {

        try {
            Set<String> sendingSet= session.getSending();
            List<String> sendingList = new ArrayList<String>();
            if (sendingSet != null) {
                sendingList = new ArrayList<String>(sendingSet);
            }

            List<Sending> sendings = wikittyProxy.restore(Sending.class, sendingList);

            for (Sending sending : sendings) {
                if (sending != null) {
                    if (user.getWikittyId().equals(sending.getUser())) {
                        String formId = form.getWikittyId();
                        Set<String> forms = sending.getForm();
                        if (forms != null && forms.contains(formId)) {

                            // Remove
                            sending.removeForm(form.getWikittyId());

                            // Add deleted
                            sending.addDeletedForms(form.getWikittyId());
                        }

                        // If is empty, set deleted
                        Set<String> formsIds = sending.getForm();
                        if (formsIds == null || formsIds.isEmpty()) {
                            sending.setStatus(VradiConstants.SendingStatus.DELETED.getValue());
                        }
                        wikittyProxy.store(sending);
                        return sending;
                    }
                }
            }
            return null;
        } catch (Exception eee) {
            throw new VradiException("Cant remove forms for user " + user.getName(), eee);
        }
    }

    /**
     * Recherche dans la session l'object sending qui correspond au query maker
     * donné ou en crée une nouvelle s'il n'a pas été trouvé.
     * 
     * Le sending retourné n'est pas affecter au query maker.
     * 
     * @param currentSession
     * @param queryMaker
     * @return
     */
    protected Sending findExistingOrCreateSending(Session currentSession, User user) {

        // Restore existing sending in session
        Set<String> sendingsId = currentSession.getSending();

        // restore sendings
        List<Sending> sendings;
        if (sendingsId != null) {
            sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsId));
        }
        else {
            sendings = new ArrayList<Sending>();
        }

        // chargement, si il existe, du sending concernant
        // ce client / user / group
        for (Sending sending : sendings) {
            String userId = user.getWikittyId();
            if (sending != null && userId.equals(sending.getUser()) &&
                    !(sending.getStatus() == VradiConstants.SendingStatus.DELETED.getValue())) {

                if (log.isDebugEnabled()) {
                    log.debug("Sending is existing for queryMaker : " + user.getName());
                }

                return sending;
            }
        }

        // Set properties in default
        SendingImpl newSending = new SendingImpl();
        newSending.setSentDate(null);
        newSending.setReceptionDate(null);
        newSending.setParagraph(VradiConstants.DEFAULT_SENDING_PARAGRAPH);
        newSending.setReceptionProof(false);
        newSending.setStatus(VradiConstants.SendingStatus.TO_SEND.getValue());
        newSending.setUser(user.getWikittyId());

        if (log.isDebugEnabled()) {
            log.debug("Sending dont exist for queryMaker : " + user + ", creating one");
        }
        return newSending;
    }

    /**
     * Retourne la liste {@code forms} d'origine en ayant filtré les formulaires
     * qui ont déja été lié.
     * 
     * @param currentSession
     * @param forms
     * @param queryMaker
     * @return
     */
    protected List<Form> removeAlreadyBound(Session currentSession, List<Form> forms, User user) {
        List<String> formIds = new ArrayList<String>();

        // Extract formIds
        for (Form form : forms){
            formIds.add(form.getWikittyId());
        }

        // Recherche d'un sending existant
        String userId = user.getWikittyId();

        Search search = Search.query();

        Set<String> sendingIds = currentSession.getSending();

        // Except current
        if (sendingIds != null) {
            for (String currentSendingId : sendingIds) {
                search.neq(Element.ELT_ID, currentSendingId);
            }
        }

        // If its attached or deleted
        search.or().contains(Sending.FQ_FIELD_SENDING_FORM, formIds)
                      .contains(Sending.FQ_FIELD_SENDING_DELETEDFORMS, formIds);

        // And attached to client / user / group
        search.eq(Sending.FQ_FIELD_SENDING_USER, userId);

        // And not deleted
        search.neq(Sending.FQ_FIELD_SENDING_STATUS,
                         String.valueOf(VradiConstants.SendingStatus.DELETED.getValue()));

        // Find
        PagedResult<Sending> pagedResult = wikittyProxy.findAllByCriteria(Sending.class, search.criteria());

        if (log.isDebugEnabled()){
            log.debug("Remove " + pagedResult.size() + " already bounds forms");
        }
        for (Sending send : pagedResult.getAll()){
            Set<String> formsIds = send.getForm();
            if (formsIds != null) {
                formIds.removeAll(formsIds);
            }
        }
        if (log.isDebugEnabled()){
            log.debug("Rest " + formIds.size() + " forms");
        }

        List<Form> result = wikittyProxy.restore(Form.class, formIds);
        return result;
    }

    /*
     * Status
     */

    public Status getStatus(String statusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatus(" + statusId + ")");
        }
        Status status = wikittyProxy.restore(Status.class, statusId);
        return status;
    }

    public List<Status> getStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatuses(statusIds)");
        }
        List<Status> statuses = wikittyProxy.restore(Status.class, statusIds);
        return new ArrayList<Status>(statuses);
    }

    public List<Status> createDefaultStatuses() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("createDefaultStatuses()");
        }

        List<Status> all = new ArrayList<Status>();
        all.add(getNonTraiteStatus());
        all.add(getNonSelectionneStatus());
        all.add(getPreselectionneStatus());
        all.add(getSelectionneStatus());

        return new ArrayList<Status>(all);
    }

    public Status getNonTraiteStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.NON_TRAITE);
    }

    public Status getNonSelectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.NON_SELECTIONNE);
    }

    public Status getPreselectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.PRESELECTIONNE);
    }

    public Status getSelectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.SELECTIONNE);
    }

    protected Status getOrCreateStatus(VradiConstants.FormStatus type){
        if (log.isDebugEnabled()) {
            log.debug("getOrCreateStatus(" + type.getName() + ")");
        }
        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, Status.EXT_STATUS)
                .eq(Status.FQ_FIELD_STATUS_NAME, type.getName());

        Criteria criteria = search.criteria();

        Status status = wikittyProxy.findByCriteria(Status.class,
                criteria);

        if (status == null) {
            log.warn("Status not found creating : " + type.getName());

            status = new StatusImpl();
            status.setName(type.getName());
            status.setValue(type.getValue());
            status.setDescription(type.getDescription());
            status.setModifiable(false);

            status = wikittyProxy.store(status);
        }
        return status;
    }

    public List<Status> updateStatuses(List<Status> statuses)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatuses(statuses)");
        }

        try {
            List<Status> result = wikittyProxy.store(statuses);
            return result;

        } catch (Exception eee) {
            log.error("Cant update statuses : ", eee);
            throw new VradiException("Cant update statuses : ", eee);
        }
    }

    public Status updateStatus(Status status) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatus(status)");
        }

        try {
            status = wikittyProxy.store(status);
            return status;

        } catch (Exception eee) {
            log.error("Cant update status : ", eee);
            throw new VradiException("Cant update status : ", eee);
        }
    }

    public boolean deleteStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteStatus(" + statusIds + ")");
        }
        boolean result = true;

        for (String id : statusIds) {
            // Check if status is in use
            Search search = Search.query().eq(Element.ELT_EXTENSION, Form.EXT_FORM)
                    .eq(Form.FQ_FIELD_INFOGENE_STATUS, id);
            Criteria criteria = search.criteria();
            PagedResult<Form> forms = wikittyProxy.findAllByCriteria(Form.class, criteria);
            if (forms.size() > 0) {
                result = false;
            } else {
                try {
                    wikittyProxy.delete(id);
                } catch (Exception e) {
                    log.error("Cant delete status : ", e);
                    throw new VradiException("Cant delete status : ", e);
                }
            }
        }
        return result;
    }

    protected List<User> getUsersOfClient(String clientId) {

        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER)
                .eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();

        PagedResult<User> usersPageResult = wikittyProxy.findAllByCriteria(User.class, criteria);
        return new ArrayList<User>(usersPageResult.getAll());

    }

    public void deleteSession(String sessionId) throws VradiException {

        if (sessionId == null) {
            return;
        }

        // Restore session
        Session sessionToDelete = wikittyProxy.restore(Session.class, sessionId);

        if (sessionToDelete.getStatus() != VradiConstants.SessionStatus.ACTIVE.getValue()) {
            throw new VradiException("Session is not active, abording");
        }

        // Delete all sending asssociated
        Set<String> sendings = sessionToDelete.getSending();
        if (sendings != null) {
            wikittyProxy.delete(sendings);
        }

        // Delete session
        wikittyProxy.delete(sessionId);
    }

    public Session getSessionOfSending(String sendingId) throws VradiException {
        try {
            Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION)
                    .contains(Session.FQ_FIELD_SESSION_SENDING, sendingId).criteria();

            return wikittyProxy.findByCriteria(Session.class, criteria);
        } catch (Exception eee) {
            log.error("Cant restore session for sending '" + sendingId + "' : ", eee);
            throw new VradiException("Cant restore session for sending '" + sendingId + "' : ", eee);
        }
    }

    /**
     * Check if template file form extension name in param
     *
     * @param forms to remove
     * @return false if no session is opened
     * @throws VradiException
     */
    public boolean removeFormsFromSession(List<Form> forms) throws VradiException {
        Session session = getLastOpenSession();
        if (session == null) {
            return false;
        }

        // Extract forms ids
        List<String> formsIds = new ArrayList<String>();
        for (Form form : forms) {
            formsIds.add(form.getWikittyId());
        }

        for (String sendingId : session.getSending()) {
            Sending sending = wikittyProxy.restore(Sending.class, sendingId);

            if (sending != null) {
                // Remove forms
                for (String formId : formsIds) {
                    sending.removeForm(formId);
                    sending.addDeletedForms(formId);
                }

                if (sending.getForm() == null || sending.getForm().isEmpty()) {
                    sending.setStatus(VradiConstants.SendingStatus.DELETED.getValue());
                }
                // Store
                wikittyProxy.store(sending);
            }
        }
        return true;
    }
}
