/*
 * #%L
 * SGQ :: Business
 * $Id: BatchService.java 53 2012-09-11 16:11:13Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.2/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/BatchService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import java.io.File;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;
import org.xBaseJ.DBF;
import org.xBaseJ.fields.CharField;
import org.xBaseJ.fields.DateField;
import org.xBaseJ.fields.NumField;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.business.model.BatchModel;
import com.herbocailleau.sgq.business.model.SearchModel;
import com.herbocailleau.sgq.entities.Analyze;
import com.herbocailleau.sgq.entities.AnalyzeDAO;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.Presentation;
import com.herbocailleau.sgq.entities.PresentationCode;
import com.herbocailleau.sgq.entities.PresentationDAO;
import com.herbocailleau.sgq.entities.Product;
import com.herbocailleau.sgq.entities.ProductDAO;
import com.herbocailleau.sgq.entities.Supplier;

public class BatchService extends SgqService {

    private static final Log log = LogFactory.getLog(BatchService.class);

    public long getBatchCount() {
        long count = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            count = batchDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return count;
    }

    public List<BatchModel> searchBatch(SearchModel search) {
        List<BatchModel> result = null;
        
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findAllWithQuantity(search);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't search batches", ex);
        }
        
        return result;
    }

    public List<Batch> findAllBetweenNumbers(int startNumber, int endNumber) {
        List<Batch> result = null;
        
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findAllBetweenNumbers(startNumber, endNumber);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't search batches", ex);
        }
        
        return result;
    }

    public Batch getBatchById(String batchId) {
        Batch result = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findByTopiaId(batchId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        
        return result;
    }

    /**
     * Find batch by number.
     * 
     * Only usefull for test, for main code prefer {@link #getBatchById(String)}.
     * 
     * @param batchNumer batch number
     * @return batch instance
     */
    protected Batch getBatchByNumber(int batchNumer) {
        Batch result = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findByNumber(batchNumer);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by number", ex);
        }

        return result;
    }

    public List<Presentation> findAllPresentationsByBatch(Batch batch) {
        List<Presentation> presentations = null;
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            presentations = presentationDAO.findAllByBatch(batch);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return presentations;
    }

    public Presentation getPresentationById(String presentationId) {
        Presentation result = null;
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            result = presentationDAO.findByTopiaId(presentationId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }

        return result;
    }
    
    public void updatePresentation(Presentation presentation) {
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            presentationDAO.update(presentation);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't update", ex);
        }
    }

    public List<Analyze> findAllAnalyzesByBatch(Batch batch) {
        List<Analyze> analyzes = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyzes = analyzeDAO.findAllByBatch(batch);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return analyzes;
    }
    
    public int getNextBioBatchNumber() {
        int result = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.getNextRangeIdBetween(config.getBatchRangeBioStart(),
                    config.getBatchRangeBioEnd());
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }
    
    public int getNextNonBioBatchNumber() {
        int result = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.getNextRangeIdBetween(config.getBatchRangeNonBioStart(),
                    config.getBatchRangeNonBioEnd());
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }

    public Analyze getAnalyzeById(String analyzeId) {
        Analyze result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findByTopiaId(analyzeId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        
        return result;
    }

    public void addAnalyze(Analyze analyze) {
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyzeDAO.create(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't create analyze", ex);
        }
    }

    public List<Analyze> getAnalysisToSend(AnalyzeType analyzeType) {
        List<Analyze> result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findAllAnalyzeToSend(analyzeType);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analysis", ex);
        }
        return result;
    }

    public List<Analyze> getAnalysisToReceive(AnalyzeType analyzeType) {
        List<Analyze> result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findAllAnalyzeToReceive(analyzeType);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analysis", ex);
        }
        return result;
    }

    /**
     * Import du fichier dbf des numéro de lots.
     * 
     * @param batchFile dbf file
     */
    public void importBatchFile(File batchFile) {
        
        if (log.isInfoEnabled()) {
            log.info("Importing batch from file " + batchFile);
        }

        DBF fileDB = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ProductDAO productDAO = daoHelper.getProductDAO();
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();

            // get import bounds
            int nonBioStart = getNextNonBioBatchNumber();
            int nonBioEnd = config.getBatchRangeNonBioEnd();
            int bioStart = getNextBioBatchNumber();
            int bioEnd = config.getBatchRangeBioEnd();

            // get date format
            DateFormat df = new SimpleDateFormat("dd/MM/yy");

            // Open dbf file
            fileDB = new DBF(batchFile.getAbsolutePath(), DBF.READ_ONLY, "UTF-8");

            // code,nom
            DateField dateEntreeField  = (DateField) fileDB.getField("DATE_R");
            NumField colisField  = (NumField) fileDB.getField("COLIS_R");
            NumField poidsField  = (NumField) fileDB.getField("POIDS_R");
            CharField codeProduitField  = (CharField) fileDB.getField("PROD_R");
            NumField lotField  = (NumField) fileDB.getField("LOT_R");
            CharField dluoField  = (CharField) fileDB.getField("DLUO_R");

            for (int i = 0; i < fileDB.getRecordCount(); i++) {
                fileDB.read();

                //int num = Integer.parseInt(lotField.get().trim());
                int num = (int)Double.parseDouble(lotField.get().trim()); // pb jeu de test
                if (nonBioStart <= num && num <= nonBioEnd ||
                        bioStart <= num && num <= bioEnd) {

                    String codeProduit = codeProduitField.get().trim();
                    PresentationCode codePres = null;
                    if (codeProduit.length() == 6) {
                        codePres = PresentationCode.getPresentationCodeFor(codeProduit.charAt(0));
                        codeProduit = codeProduit.substring(1);
                    } else {
                        if (log.isWarnEnabled()) {
                            log.warn("Invalid product code " + codeProduit);
                        }
                        continue;
                    }

                    String dluo = dluoField.get();
                    if (!dluo.matches("\\d\\d/\\d\\d/\\d\\d")) { continue;}

                    double quantity = Double.parseDouble(poidsField.get());
                    Product product = productDAO.findByCode(codeProduit);
                    if (product == null) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find product " + codeProduit);
                        }
                        continue;
                    }

                    Batch batch = batchDAO.createByNaturalId(num);
                    batch.setProduct(product);
                    batch.setDluo(df.parse(dluo));
                    batch.setEntryDate(dateEntreeField.getCalendar().getTime());
                    batch.setQuantity(quantity);
                    //batch.setPackageCount(Integer.parseInt(colisField.get().trim()));
                    batch.setPackageCount((int)Double.parseDouble(colisField.get().trim())); // pb jeu de test
                    batch = batchDAO.update(batch);

                    Presentation presentation = presentationDAO.create();
                    presentation.setQuantity(quantity);
                    presentation.setBatch(batch);
                    presentation.setName(product.getName());
                    presentation.setPresentationCode(codePres);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Skipping product " + num + " (out of range)");
                    }
                }
            }
            
            daoHelper.commit();
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't read dbf file", ex);
        } finally {
            SgqUtils.closeFile(fileDB);
        }
    }

    public void sendAnalyzeToSupplier(Analyze analyze, Supplier supplier) {

        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyze.setSupplier(supplier);
            analyze.setSentDate(new Date());
            analyzeDAO.update(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't send analyze to supplier");
            }
        }
        
    }

    public void receiveAnalyzeFromSupplier(Analyze analyze, Supplier supplier) {

        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyze.setSupplier(supplier);
            analyze.setReceiptDate(new Date());
            analyzeDAO.update(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't send analyze to supplier");
            }
        }
    }
}
