/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.util.Collection;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Basic extension migration use to manage rename field name. Order to detect a
 * rename, not creation a new field, you should use tag value "renameFrom" with
 * name of use in before extension.
 *
 * Since 1.5, this class also take care about extension order.
 * 
 * @author ruchaud
 */
public class WikittyExtensionMigrationRename implements WikittyExtensionMigration {

    static private Log log = LogFactory.getLog(WikittyExtensionMigrationRename.class);

    public static String TAG_RENAME = "renameFrom";

    /**
     * Recursively add extension with requires ones (orderer).
     * 
     * @param service service
     * @param transaction current transaction to use
     * @param wikitty wikitty to add extension to
     * @param newExt extension to add
     */
    protected void addExtension(AbstractWikittyService service, WikittyTransaction transaction, Wikitty wikitty, WikittyExtension newExt) {

        // manage requires before current
        String requires = newExt.getRequires();
        if (StringUtils.isNotEmpty(requires)) {
            for (String require : requires.split(",")) {
                WikittyExtension newRequireExt = service.restoreExtensionLastVersion(transaction, require);
                addExtension(service, transaction, wikitty, newRequireExt);
            }
        }

        // add current extension
        wikitty.addExtension(newExt);
    }

    @Override
    public Wikitty migrate(AbstractWikittyService service, WikittyTransaction transaction, Wikitty wikitty, WikittyExtension oldExt, WikittyExtension newExt) {

        String wikittyId = wikitty.getId();
        String wikittyVersion = wikitty.getVersion();

        Wikitty result = new Wikitty(wikittyId);
        result.setVersion(wikittyVersion);

        // Add all extension and replace old by the new
        for (WikittyExtension extension : wikitty.getExtensions()) {
            // add one by one to manage require
            
            // during loop add new when old one is found
            if (extension.equals(oldExt)) {
                addExtension(service, transaction, result, newExt);
            }
            else {
                addExtension(service, transaction, result, extension);
            }
        }

        // Copy other values
        Set<String> fqFieldNames = wikitty.fieldNames();
        String extName = newExt.getName();
        for (String fieldName : fqFieldNames) {
            if(!fieldName.startsWith(extName + ".")) {
                Object value = wikitty.getFqField(fieldName);
                log.debug("Value " + fieldName + "=" + value);
                result.setFqField(fieldName, value);
            }
        }

        // Migrate field name
        Collection<String> oldFieldNames = oldExt.getFieldNames();
        Collection<String> newFieldNames = newExt.getFieldNames();
        for (String fieldName : newFieldNames) {
            FieldType fieldType = newExt.getFieldType(fieldName);

            String renameFrom = fieldType.getTagValue(TAG_RENAME);
            log.debug("Scan rename migration on " + extName + "." + fieldName + ":" + TAG_RENAME + "=" + renameFrom);

            if(renameFrom != null && !renameFrom.isEmpty()) {
                Object value = wikitty.getFieldAsObject(extName, renameFrom);
                log.debug("Rename " + extName + "." + fieldName + "=" + value);
                result.setField(extName, fieldName, value);
            } else {
                if(oldFieldNames.contains(fieldName)) {
                    Object value = wikitty.getFieldAsObject(extName, fieldName);
                    log.debug("Copy " + extName + "." + fieldName + "=" + value);
                    result.setField(extName, fieldName, value);
                }
            }
        }

        return result;
    }

}
