/*
 * #%L
 * Wikitty :: wikitty-solr-impl
 * 
 * $Id: WikittySearchEngineSolr.java 704 2011-02-15 14:42:04Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.4/wikitty-solr-impl/src/main/java/org/nuiton/wikitty/storage/solr/WikittySearchEngineSolr.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin POUSSIN
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.storage.solr;

import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.embedded.EmbeddedSolrServer;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.core.CoreContainer;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.FacetTopic;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.FieldType.TYPE;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.storage.WikittyExtensionStorage;
import org.nuiton.wikitty.storage.WikittySearchEngine;
import org.nuiton.wikitty.services.WikittyTransaction;

import java.io.File;
import java.util.Collections;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyTreeNodeHelper;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.TreeNodeResult;

/**
 *
 * @author poussin
 * @version $Revision: 704 $
 *
 * Last update: $Date: 2011-02-15 15:42:04 +0100 (mar., 15 févr. 2011) $
 * by : $Author: bpoussin $
 */
public class WikittySearchEngineSolr implements WikittySearchEngine {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittySearchEngineSolr.class);

    /** solr server */
    protected SolrServer solrServer;

    /** Field modifier use to transform to solr format */
    protected TypeFieldModifier fieldModifier;

    /** JTA resource */
    protected SolrResource solrResource;

    /**
     * Init wikitty search engine on solr embedded server.
     * 
     * @param extensionStorage extension storage
     * @param properties properties (can be null)
     */
    public WikittySearchEngineSolr(
            ApplicationConfig config, WikittyExtensionStorage extensionStorage) {

        // init system env solr.data.dir
        if (config != null) {
            // choix du storage (file or Ram)
            String solrDirFactoryKey =
                    WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_FACTORY.getKey();
            String solrDirFactory = config.getOption(solrDirFactoryKey);
            if (solrDirFactory != null) {
                System.setProperty(solrDirFactoryKey, solrDirFactory);
            }

            // on utilise le directory que si on est pas en Ram
            if (solrDirFactory != null && !solrDirFactory.contains("RAMDirectoryFactory")) {
                String solrDataDirKey =
                        WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_DATA.getKey();
                String solrDataDir = config.getOption(solrDataDirKey);
                // make sure that dir exists
                if (solrDataDir != null) {
                    File file = new File(solrDataDir);
                    if (!file.exists() && !file.mkdirs()) {
                        throw new WikittyException(String.format(
                                "Can't create directory '%s'", solrDataDir));
                    }
                    log.info(String.format("Use SolR directory '%s'", solrDataDir));
                    System.setProperty(solrDataDirKey, solrDataDir);
                }
            }
        }

        try {
            CoreContainer.Initializer initializer = new CoreContainer.Initializer();
            CoreContainer coreContainer = initializer.initialize();
            solrServer = new EmbeddedSolrServer(coreContainer, "");

            fieldModifier = new TypeFieldModifier(extensionStorage);
            solrResource = new SolrResource(solrServer);
            
        } catch (Exception eee) {
            throw new WikittyException("SolR initialization error", eee);
        }
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        try {
            // FIXME poussin 20100618 pourquoi n'est pas fait dans la transaction ?
            solrResource.init();
            solrServer.deleteByQuery("*:*");
        } catch (Exception eee) {
            throw new WikittyException("Error during clearing SolR data", eee);
        }
    }

    @Override
    public void store(WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean force) {
        try {
            solrResource.init();

            // tous les wikitties passes en parametre
            Map<String, Wikitty> allWikitties = new HashMap<String, Wikitty>();
            // les ids des wikitties en parametre reellement modifier (a reindexer)
            Set<String> dirtyObject = new HashSet<String>();
            // les ids des TreeNodes dont le champs parent a change (est aussi
            // contenu dans dirtyObject
            Set<String> dirtyParent = new HashSet<String>();
            // les valeur du champs parent des TreeNodes dont le champs parent
            // a change (sauf si parent = null)
            Set<String> dirtyParentParentId = new HashSet<String>();

            // remplissage des collections
            for(Wikitty w : wikitties) {
                allWikitties.put(w.getId(), w);
                if (force || !w.getDirty().isEmpty() ||
                        WikittyUtil.versionGreaterThan("1", w.getVersion())) {
                    // s'il y a au moins un champs a reindexer ou que l'objet
                    // n'a jamais ete sauve (1 > version)
                    dirtyObject.add(w.getId());
                    if (WikittyTreeNodeHelper.hasExtension(w) &&
                            (w.getDirty().contains(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT)
                            || null == WikittyTreeNodeHelper.getParent(w))) {
                            // si le pere a change
                            // ou qu'il est null (creation d'un nouvel arbre)
                            // il faut indexer le noeud
                        dirtyParent.add(w.getId());
                        String parent = WikittyTreeNodeHelper.getParent(w);
                        if (parent != null) {
                            dirtyParentParentId.add(parent);
                        }
                    }
                }
            }

            // recuperation des documents Solr deja indexes, pour minimiser la reindexation
            Map<String, SolrDocument> dirtyObjectDoc =
                    SolrUtil.findAllById(solrServer, dirtyObject);
            Map<String, SolrDocument> dirtyParentDoc =
                    SolrUtil.findAllByParents(solrServer, dirtyParent);
            Map<String, SolrDocument> parents =
                    SolrUtil.findAllById(solrServer, dirtyParentParentId);

            // On genere en meme temps la liste des attachments qui doivent
            // etre reindexe
            AttachmentInTree attachmentInTree = new AttachmentInTree();
            
            //
            // Phase 1: on indexe les objets passe en paremetre, on copie si
            //          besoin #tree.attached et #tree.* des TreeNode dont
            //          leur champs parent n'a pas ete modifie, et dans ce cas
            //          on collecte les modif d'attachments des TreeNode
            //

            for(String id : dirtyObject) {
                Wikitty w = allWikitties.get(id);
                SolrDocument oldDoc = dirtyObjectDoc.get(id);
                SolrInputDocument doc = createIndexDocument(w);
                if (oldDoc != null) {
                    // copy des champs #tree.attached
                    SolrUtil.copySolrDocument(oldDoc, doc, TREENODE_ATTACHED + ".*");
                    if (WikittyTreeNodeHelper.hasExtension(w)
                            && !dirtyParentDoc.containsKey(id)) {
                        // si c'est un TreeNode, mais qu'aucun pere n'a change
                        // on recopie l'ancienne indexation d'arbre
                        // si elle existe
                        SolrUtil.copySolrDocument(oldDoc, doc, TREENODE_PREFIX + ".*");

                        // il faut verifier les objets attaches
                        // attaches ajoute/supprime
                        // on ne traite ici que les TreeNode sans modif d'indexation
                        // pour les autres les attachments seront traites dans
                        // la phase suivante
                        Set<String> newAtt = WikittyTreeNodeHelper.getAttachment(w);
                        Collection oldAtt = oldDoc.getFieldValues(SolrUtil.getSolrFieldName(
                                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT,
                                TYPE.WIKITTY));
                        // il faut supprimer l'indexation arbre des noeuds
                        // qui sont dans old, mais pas dans new
                        Set<String> toRemove = new HashSet<String>();
                        if (oldAtt != null) {
                            toRemove.addAll(oldAtt);
                        }
                        if (newAtt != null) {
                            toRemove.removeAll(newAtt);
                        }
                        attachmentInTree.remove(id, toRemove);
                        // il faut ajouter l'indexation arbre des noeuds
                        // qui sont dans new, mais pas dans old
                        Set<String> toAdd = new HashSet<String>();
                        if (newAtt != null) {
                            toAdd.addAll(newAtt);
                        }
                        if (oldAtt != null) {
                            toAdd.removeAll(oldAtt);
                        }
                        attachmentInTree.add(id, toAdd);
                    }
                }
                solrResource.addDoc(id, doc);
            }

            //
            // Phase 2: on reindexe tous les TreeNode qui en ont besoin
            //          nouveau TreeNode ou TreeNode ayant un parent modifie
            //

            // on ajoute tous les TreeNode qui doivent aussi etre reindexe
            // noeud du sous arbre d'un noeud dont le pere a ete modifie
            dirtyParent.addAll(dirtyParentDoc.keySet());

            for (String id : dirtyParent) {
                // w et oldDoc peuvent etre null, mais pas en meme temps
                // w est null si c'est un noeud dont la reindexation est force
                // parce que un de ces peres a change de parent
                // oldDoc est null, si l'objet n'a jamais ete indexe (nouveau)
                Wikitty w = allWikitties.get(id);
                SolrDocument oldDoc = dirtyParentDoc.get(id);
                SolrInputDocument doc = solrResource.getAddedDoc(id);
                if (w == null) {
                    // on reindexe un ancien objet
                    // normalement doc doit etre null
                    doc = new SolrInputDocument();
                    // on recopie tous les champs, sauf l'indexation arbre
                    SolrUtil.copySolrDocumentExcludeSomeField(
                            oldDoc, doc, TREENODE_PREFIX + ".*");

                    // modifie les champs root, parents
                    addTreeIndexField(solrResource, doc, parents);

                    attachmentInTree.remove(oldDoc);
                    attachmentInTree.add(oldDoc);
                } else if (oldDoc == null) {
                    // ajoute les champs root, parents
                    addTreeIndexField(solrResource, doc, parents);

                    // on indexe un nouvel objet, il faut ajouter tous les
                    // attachment pour indexation
                    attachmentInTree.add(w);
                } else {
                    // ni w, ni oldDoc ne sont pas null, c'est une modification
                    // dans la phase precendente on a deja indexe les champs
                    // normaux

                    // ajoute les champs root, parents
                    addTreeIndexField(solrResource, doc, parents);

                    // il faut supprimer tous les anciens attaches
                    // et ajouter tous nouveaux pour la reindexation
                    attachmentInTree.remove(oldDoc);
                    attachmentInTree.add(w);
                }
                solrResource.addDoc(id, doc);
            }

            //
            // Phase 3: on reindexe les attachments qui en ont besoin
            //

            // on passe null pour tree, car tous les noeuds doivent etre dans
            // solrResource
            addTreeIndexField(solrResource, null, attachmentInTree);
        } catch (Exception eee) {
            throw new WikittyException("Can't store wikitty", eee);
        }
    }

    /**
     * Plusieurs actions possibles en fontion du type d'objet:
     *
     * <li> suppression d'un objet NON noeud
     *   <li> suppression de cet objets
     *   <li> suppression de cet objets dans les attachments des noeuds qui le contiennent
     * </li>
     * <li> suppression d'un noeud d'arbre
     *   <li> suppression du noeud
     *   <li> reindexation des noeuds qui le contenait comme parent
     *   <li> suppression des attached sur les objets contenus dans les attachments de ce noeud
     *   <li> reindexation des objets qui le contenait comme parent dans un champs attached
     * </li>
     *
     * @param transaction
     * @param ids
     * @throws WikittyException
     */
    @Override
    public void delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            solrResource.init();

            AttachmentInTree attachmentInTree = new AttachmentInTree();

            // list de tous les objets a supprimer
            Map<String, SolrDocument> removed = 
                    SolrUtil.findAllById(solrServer, ids);
            // list des TreeNode supprimer
            Map<String, SolrDocument> treeNodeRemoved =
                    new HashMap<String, SolrDocument>();

            // list des TreeNode dont des attachments ont ete supprime
            Set<String> treeNodeAttachmentRemovedId = new HashSet<String>();

            //
            // Phase 1: Suppression des objets demandes en parametre
            //
            for (Map.Entry<String, SolrDocument> e : removed.entrySet()) {
                String id = e.getKey();
                SolrDocument doc = e.getValue();
                if (doc.containsKey(TREENODE_ROOT)) {
                    treeNodeRemoved.put(id, doc);
                    attachmentInTree.remove(doc);
                }
                // recherche tous les noeuds sur lequel est attache cet objet
                Set<String> treeNodeIds = SolrUtil.getAttachedTreeNode(doc);
                treeNodeAttachmentRemovedId.addAll(treeNodeIds);
                
                solrResource.deleteDoc(id);
            }

            //
            // Phase 2: Suppression dans les listes d'attachment des TreeNode supprimes
            //

            // on essaie pas de modifier un noeud que l'on vient de supprimer
            treeNodeAttachmentRemovedId.removeAll(ids);
            if (treeNodeAttachmentRemovedId.size() > 0) {

                // recuperation des noeuds dont il faut modifier les attachments
                Map<String, SolrDocument> treeNodeAttachmentRemoved =
                        SolrUtil.findAllById(solrServer, treeNodeAttachmentRemovedId);
                for (Map.Entry<String, SolrDocument> e : treeNodeAttachmentRemoved.entrySet()) {
                    String id = e.getKey();

                    // le noeud n'a pas ete supprime, donc il faut le mettre a jour
                    SolrDocument doc = e.getValue();
                    SolrInputDocument newDoc = new SolrInputDocument();
                    String field = SolrUtil.getSolrFieldName(
                            WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT,
                            TYPE.WIKITTY);
                    SolrUtil.copySolrDocumentExcludeSomeField(doc, newDoc, field);
                    Collection atts = doc.getFieldValues(field);
                    // remove deleted attachment
                    Set<String> newAtts = new HashSet<String>(atts);
                    newAtts.removeAll(ids);
                    // si apres effacement des attachments, il n'y a plus
                    // d'attachment sur le node, cela revient a avoir
                    // le field attachment null
                    if (newAtts.isEmpty()) {
                        newAtts = null;
                    }

                    addToIndexDocument(newDoc, TYPE.WIKITTY, field, newAtts);
                    solrResource.addDoc(id, newDoc);
                }
            }

            // les phases suivantes ne sont a faire que s'il y a des TreeNode
            // supprimes
            Map<String, SolrDocument> treeNodeParentRemoved = Collections.emptyMap();
            if (treeNodeRemoved != null && treeNodeRemoved.size() > 0) {
                //
                // Phase 3: reindexation des noeuds dont un pere a disparu
                //

                // list des TreeNode dont un des pere a ete supprime
                treeNodeParentRemoved =
                        SolrUtil.findAllByParents(solrServer, ids);

                // on commence par supprimer des noeuds a reindexer les noeuds supprimes
                treeNodeParentRemoved.keySet().removeAll(treeNodeRemoved.keySet());

                for (Map.Entry<String, SolrDocument> e : treeNodeParentRemoved.entrySet()) {
                    SolrDocument oldDoc = e.getValue();

                    SolrInputDocument doc = new SolrInputDocument();
                    // on recopie tous les champs, sauf l'indexation arbre
                    SolrUtil.copySolrDocumentExcludeSomeField(
                            oldDoc, doc, TREENODE_PREFIX + ".*");

                    // modifie les champs root, parents
                    addTreeIndexField(solrResource, doc, treeNodeParentRemoved);

                    // il faut reindexer tous les attachments de ce noeud
                    attachmentInTree.remove(oldDoc);
                    attachmentInTree.add(doc);
                }

                //
                // Phase 4: on reindexe les attachments qui en ont besoin
                //
                attachmentInTree.clean(ids);
                addTreeIndexField(solrResource, treeNodeParentRemoved, attachmentInTree);
            }
        } catch (Exception eee) {
            throw new WikittyException("Can't delete wikitty in index", eee);
        }
    }

    /**
     * Modifie/Ajoute les champs specifique a l'indexation des arbres sur les
     * TreeNode.
     *
     * On se base sur le fait que si un TreeNode est dans SolrResource il ne
     * peut etre que dans deux etats. Soit il a ete reindexe pour les arbres
     * et il a les champs d'indexation arbre. Soit il a pas encore ete reindexe
     * pour les arbres et dans ce cas il ne doit pas avoir les champs d'indexation
     * d'arbre. (il est donc interdit d'avoir des champs d'indexation arbre
     * obsolete si le document est dans SolrResource)
     *
     * @param doc les documents representant le TreeNode
     * @param tree tous les autres noeuds d'arbre dont on pourrait avoir
     * besoin pour l'indexation
     */
    protected void addTreeIndexField(SolrResource solrResource,
            SolrInputDocument doc, Map<String, SolrDocument> tree) {
        Set<String> parents = new HashSet<String>();
        String root = null;
        String parentId = (String)doc.getFieldValue(SOLR_ID);
        parents.add(parentId);
        while (root == null) {
            String nextParentId = null;
            SolrInputDocument parentDoc = solrResource.getAddedDoc(parentId);
            if (parentDoc != null) {
                if (parentDoc != null) {
                    // si parentDoc a deja ete indexe pour l'arbre, on peut reutiliser
                    // directement les valeurs et sortir de la boucle
                    if (parentDoc.containsKey(TREENODE_ROOT)) {
                        root = (String) parentDoc.getFieldValue(TREENODE_ROOT);
                        Collection p = parentDoc.getFieldValues(TREENODE_PARENTS);
                        parents.addAll(p);
                        break;
                    } else {
                        nextParentId = (String) parentDoc.getFieldValue(SolrUtil.getSolrFieldName(
                                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT, TYPE.WIKITTY));
                    }
                }
            } else {
                SolrDocument oldParentDoc = tree.get(parentId);
                if (oldParentDoc != null) {
                    // si parentDoc a deja ete indexe pour l'arbre, on peut reutiliser
                    // directement les valeurs et sortir de la boucle
                    if (oldParentDoc.containsKey(TREENODE_ROOT)) {
                        root = (String) oldParentDoc.getFieldValue(TREENODE_ROOT);
                        Collection p = oldParentDoc.getFieldValues(TREENODE_PARENTS);
                        parents.addAll(p);
                        break;
                    } else {
                        nextParentId = (String) oldParentDoc.getFieldValue(SolrUtil.getSolrFieldName(
                                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT, TYPE.WIKITTY));
                    }
                }
            }
            if (nextParentId != null) {
                parents.add(nextParentId);
                parentId = nextParentId;
            } else {
                root = parentId;
            }
        }

        doc.removeField(TREENODE_ROOT);
        doc.removeField(TREENODE_PARENTS);
        doc.removeField(TREENODE_DEPTH);

        doc.addField(TREENODE_ROOT, root);
        doc.addField(TREENODE_DEPTH, parents.size());
        for (String id : parents) {
            doc.addField(TREENODE_PARENTS, id);
        }
    }

    /**
     * Update attaced extra field on all objects passed in argument allAttachmentToIndex
     *
     * @param solrResource must contains reindexed TreeNode, that contains attachment
     * @param allAttachmentToIndex id of object to update
     * @param attachmentRemovedInTree index to remove
     * @param attachmentAddedInTree index to add
     */
    protected void addTreeIndexField(SolrResource solrResource,
            Map<String, SolrDocument> tree, AttachmentInTree attachmentInTree) {

        if (attachmentInTree.size() > 0) {
            Map<String, SolrDocument> attachments =
                    SolrUtil.findAllById(solrServer, attachmentInTree.getAll());

            for (String treeNodeId : attachmentInTree.getRemoved().keySet()) {
                for (String attId : attachmentInTree.getRemoved().get(treeNodeId)) {
                    SolrDocument oldDoc = attachments.get(attId);
                    SolrInputDocument doc = solrResource.getAddedDoc(attId);
                    if (doc == null) {
                        doc = new SolrInputDocument();
                        SolrUtil.copySolrDocument(oldDoc, doc);
                        solrResource.addDoc(attId, doc);
                    }
                    doc.remove(TREENODE_ATTACHED + treeNodeId);
                }
            }
            for (String treeNodeId : attachmentInTree.getAdded().keySet()) {
                Collection treeNodeParents = null;
                SolrInputDocument treeNodeDoc = solrResource.getAddedDoc(treeNodeId);
                if (treeNodeDoc != null) {
                    treeNodeParents = treeNodeDoc.getFieldValues(TREENODE_PARENTS);
                } else if (tree != null) {
                    SolrDocument doc = tree.get(treeNodeId);
                    treeNodeParents = doc.getFieldValues(TREENODE_PARENTS);
                } else {
                    log.error("SolR doc not found in Transaction or in tree."
                            + "This is a bug !!!");
                }
                for (String attId : attachmentInTree.getAdded().get(treeNodeId)) {
                    SolrDocument oldDoc = attachments.get(attId);
                    SolrInputDocument doc = solrResource.getAddedDoc(attId);
                    if (doc == null) {
                        doc = new SolrInputDocument();
                        SolrUtil.copySolrDocument(oldDoc, doc);
                        solrResource.addDoc(attId, doc);
                    }
                    doc.removeField(TREENODE_ATTACHED + treeNodeId);
                    for (Object id : treeNodeParents) {
                        doc.addField(TREENODE_ATTACHED + treeNodeId, id);
                    }
                }
            }
        }
    }

    @Override
    public PagedResult<String> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        try {
            // Create query with restriction
            Restriction2Solr restriction2Solr = new Restriction2Solr(transaction, fieldModifier);
            String queryString = restriction2Solr.toSolr(criteria.getRestriction(), solrServer);
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + queryString);

            // Add paged
            int firstIndex = criteria.getFirstIndex();
            int endIndex = criteria.getEndIndex();

            query.setStart(firstIndex);
            int nbRows;
            if (endIndex == -1) {
                // WARNING It is necessary to substract 'start' otherwise,
                // there is a capacity overlow in solR
                nbRows = Integer.MAX_VALUE - firstIndex;
            } else {
                nbRows = endIndex - firstIndex + 1;
            }
            query.setRows(nbRows);

            // Add sorting
            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                for (String sort : sortAscending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.asc);
                }
            }
            
            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                for (String sort : sortDescending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.desc);
                }
            }

            // Add faceting
            List<String> facetField = criteria.getFacetField();
            log.debug("facetField : " + facetField);
            List<Criteria> facetCriteria = criteria.getFacetCriteria();

            // use to map query string to criteria facet name
            Map<String, String> facetQueryToName = new HashMap<String, String>();

            if ((facetField != null && !facetField.isEmpty())
                    || (facetCriteria != null && !facetCriteria.isEmpty())) {
                query.setFacet(true);
                query.setFacetMinCount(1);
                // query.setFacetLimit(8); // no limit actualy

                // field facetisation
                if (facetField != null) {
                    for (String fqfieldName : facetField) {
                        String tranform = fieldModifier.convertToSolr(transaction, fqfieldName);
                        query.addFacetField(tranform);
                    }
                }

                // query facetisation
                if (facetCriteria != null) {
                    for (Criteria facet : facetCriteria) {
                        String queryFacet =
                                restriction2Solr.toSolr(facet.getRestriction());
                        facetQueryToName.put(queryFacet, facet.getName());
                        query.addFacetQuery(queryFacet);
                    }
                }
            }

            if(log.isDebugEnabled()) {
                log.debug(String.format("Try to execute query %s", query));
            }
            QueryResponse resp = solrServer.query(query);
            SolrDocumentList solrResults = resp.getResults();

            Map<String, List<FacetTopic>> facets = new HashMap<String, List<FacetTopic>>();
            if (facetField != null && !facetField.isEmpty()) {
                for (FacetField facet : resp.getFacetFields()) {
                    String facetName = fieldModifier.convertToField(transaction, facet.getName());
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    if (facet.getValues() != null) {
                        for (FacetField.Count value : facet.getValues()) {
                            String topicName = value.getName();
                            int topicCount = (int) value.getCount();
                            FacetTopic topic = new FacetTopic(facetName, topicName, topicCount);
                            topics.add(topic);
                        }
                    }
                    facets.put(facetName, topics);
                }
            }
            if (facetCriteria != null && !facetCriteria.isEmpty()) {
                for (Map.Entry<String, Integer> facet : resp.getFacetQuery().entrySet()) {
                    String facetName = facet.getKey();
                    // don't use contains because, map can have key with null value
                    if (null != facetQueryToName.get(facetName)) {
                        facetName = facetQueryToName.get(facetName);
                    }
                    Integer count = facet.getValue();
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    FacetTopic topic = new FacetTopic(facetName, facetName, count);
                    topics.add(topic);
                    facets.put(facetName, topics);
                }
            }

            List<String> ids = new ArrayList<String>(solrResults.size());
            for (SolrDocument doc : solrResults) {
                String id = (String) doc.getFieldValue(SOLR_ID);
                ids.add(id);
            }

            int numFound = (int)resp.getResults().getNumFound();
            PagedResult<String> result = new PagedResult<String>(
                    firstIndex, numFound, queryString, facets, ids);

            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }
    }

    /**
     * Si l'argument n'est pas un TreeNode, une exception est levee
     *
     * @param transaction
     * @param w l'objet root du resultat
     * @param depth profondeur souhaite pour la recherche des fils
     * @param count vrai si l'on souhaite avoir le nombre d'attachment associe
     * au noeud retourne
     * @param filter filtre utilise pour compter le nombre d'attachment
     * @return
     */
    @Override
    public TreeNodeResult<String> findAllChildrenCount(
            WikittyTransaction transaction,
            String wikittyId, int depth, boolean count, Criteria filter) {
        try {
            TreeNodeResult<String> result = null;

            SolrDocument doc = SolrUtil.findById(solrServer, wikittyId);
            if (doc != null) {
                // on verifie que l'argument est bien un TreeNode
                if (doc.containsKey(TREENODE_DEPTH)) {
                    
                    Search treeSearch = Search.query().and().eq(TREENODE_PARENTS, wikittyId);
                    if (depth >= 0) {
                        Integer d = (Integer) doc.getFieldValue(TREENODE_DEPTH);
                        int startDepth = d.intValue();
                        treeSearch = treeSearch.bw(TREENODE_DEPTH,
                                String.valueOf(startDepth),
                                String.valueOf(startDepth + depth));
                    }
                    Criteria treeCriteria = treeSearch.criteria();

                    // on a dans treeSearch uniquement le noeud passe en parametre
                    // et ses enfants jusqu'a la profondeur demandee
                    Restriction2Solr restriction2Solr =
                            new Restriction2Solr(transaction, fieldModifier);
                    String queryString = restriction2Solr.toSolr(
                            treeCriteria.getRestriction(), solrServer);
                    SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + queryString);
                    QueryResponse resp = solrServer.query(query);
                    SolrDocumentList solrResults = resp.getResults();

                    Map<String, Integer> counts = new HashMap<String, Integer>();
                    // recuperation si demande du nombre d'attachment par noeud
                    if (count) {
                        // TODO poussin 20110128 regarder si on ne peut pas
                        // restreindre les facettes aux noeuds trouve dans la recherche
                        // precedente
                        Criteria attCriteria = Search.query(filter).eq(
                                TREENODE_ATTACHED_ALL, wikittyId).criteria()
                                .setFirstIndex(0).setEndIndex(0)
                                .addFacetField(TREENODE_ATTACHED_ALL);
                        PagedResult<String> attSearch =
                                findAllByCriteria(transaction, attCriteria);
                        List<FacetTopic> topics = attSearch.getTopic(TREENODE_ATTACHED_ALL);
                        if (topics != null) {
                            for (FacetTopic topic : topics) {
                                String topicName = topic.getTopicName();
                                int topicCount = topic.getCount();
                                counts.put(topicName, topicCount);
                            }
                        }
                    }

                    // construction du resultat, il proceder en 2 phases car
                    // sinon si on construit un fils avant son pere, il ne sera
                    // jamais associe
                    Map<String, TreeNodeResult<String>> allTreeNodeResult =
                            new HashMap<String, TreeNodeResult<String>>();
                    // key: id de l'enfant, value: l'id du parent
                    Map<String, String> childParent = new HashMap<String, String>();
                    // construction de tous les TreeNodeResult qui permettront
                    // de construire l'arbre
                    for (SolrDocument d : solrResults) {
                        String id = (String) d.getFieldValue(SOLR_ID);
                        String parentId = (String) d.getFieldValue(SolrUtil.getSolrFieldName(
                                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT,
                                TYPE.WIKITTY));
                        int nb = counts.containsKey(id) ? counts.get(id) : 0;
                        TreeNodeResult<String> child = new TreeNodeResult<String>(id, nb);
                        allTreeNodeResult.put(id, child);
                        childParent.put(id, parentId);
                    }
                    // construction de l'arbre avant de le retourner
                    for(Map.Entry<String, TreeNodeResult<String>> e : allTreeNodeResult.entrySet()) {
                        String id = e.getKey();
                        String parentId = childParent.get(id);
                        if (allTreeNodeResult.containsKey(parentId)) {
                            TreeNodeResult<String> child = e.getValue();
                            TreeNodeResult<String> parent = allTreeNodeResult.get(parentId);
                            parent.add(child);
                        }
                    }
                    result = allTreeNodeResult.get(wikittyId);
                } else {
                    throw new WikittyException(String.format(
                            "Wikitty '%s' do not handle extension %s",
                            wikittyId, WikittyTreeNode.EXT_WIKITTYTREENODE));
                }
            }
            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }

    }

    protected void addToIndexDocument(SolrInputDocument doc,
            TYPE type, String fqfieldName, Object fieldValue) {
        if (fqfieldName.startsWith(SOLR_WIKITTY_PREFIX)) {
            doc.remove(fqfieldName);
            doc.addField(fqfieldName, fieldValue);
        } else {
            String solrFqFieldName = SolrUtil.getSolrFieldName(fqfieldName, type);

            // #all.<fieldname>
            // permet de faire des recherches inter extension sur un champs ayant
            // le meme nom. ex:Person.name et User.name
            String solrAllFieldName = SOLR_ALL_EXTENSIONS
                    + WikittyUtil.FQ_FIELD_NAME_SEPARATOR
                    + WikittyUtil.getFieldNameFromFQFieldName(solrFqFieldName);

            String solrNullFieldFqFieldName = SOLR_NULL_FIELD + fqfieldName;

            doc.remove(solrFqFieldName);
            doc.remove(solrNullFieldFqFieldName);
            doc.remove(solrAllFieldName);

            String solrNullFieldFqFieldNameValue = "true";
            if(fieldValue != null) {
                doc.addField(solrFqFieldName, fieldValue);
                doc.addField(solrAllFieldName, fieldValue);
                solrNullFieldFqFieldNameValue = "false";
                if (log.isDebugEnabled()) {
                    log.debug("index field " + solrFqFieldName
                            + " with value '" + fieldValue + "'");
                }
            }
            doc.addField(solrNullFieldFqFieldName, solrNullFieldFqFieldNameValue);
        }
    }

    /**
     * modify one field in SolrInputDocument
     * 
     * @param doc SolrInputDocument to modify
     * @param w wikitty used to find field value
     * @param fqfieldName field to index
     */
    protected void addToIndexDocument(SolrInputDocument doc, Wikitty w, String fqfieldName) {
        TYPE type = null;
        Object fieldValue;
        if (SOLR_ID.equals(fqfieldName)) {
            // extra field #id
            fieldValue = w.getId();
        } else if (SOLR_EXTENSIONS.equals(fqfieldName)) {
            // extra field #extensions
            fieldValue= w.getExtensionNames();
        } else {
            // un champs normal
            FieldType fieldType = w.getFieldType(fqfieldName);
            type = fieldType.getType();

            fieldValue = w.getFqField(fqfieldName);

        }
        addToIndexDocument(doc, type, fqfieldName, fieldValue);
    }

    /**
     * Create all index document to used to modify indexation.
     * this method don't modify index.
     * 
     * The document looks like :
     * #id : wikittyId
     * #extensions : extensionNames
     * [fieldName] : [fieldValue]
     * #null_field-[fieldname] : [true|false]
     *
     * @param w all wikitties object to index
     * @return solrInputDocument used to modify index
     */
    protected SolrInputDocument createIndexDocument(Wikitty w) {
        if (log.isDebugEnabled()) {
            log.debug("index wikitty " + w.getId());
        }

        SolrInputDocument doc = new SolrInputDocument();
        addToIndexDocument(doc, w, SOLR_ID);
        addToIndexDocument(doc, w, SOLR_EXTENSIONS);
        // iter sur tous les champs et pas seulement sont qui ont une valeur
        // pour pouvoir les indexer comme des champs a null
        for (String fqfieldName : w.getAllFieldNames()) {
            addToIndexDocument(doc, w, fqfieldName);
        }
        return doc;
    }

}
