/*
 * #%L
 * Wikitty :: wikitty-solr-impl
 * 
 * $Id: SolrUtil.java 699 2011-01-31 12:13:07Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.4/wikitty-solr-impl/src/main/java/org/nuiton/wikitty/storage/solr/SolrUtil.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.storage.solr;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.entities.FieldType.TYPE;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.nuiton.wikitty.entities.WikittyTreeNode;

import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SOLR_ID;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.TREENODE_PARENTS;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.TREENODE_ATTACHED;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_BINARY;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_BOOLEAN;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_DATE;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_NUMERIC;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_STRING;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_WIKITTY;

/**
 *
 * @author poussin
 * @version $Revision: 699 $
 *
 * Last update: $Date: 2011-01-31 13:13:07 +0100 (lun., 31 janv. 2011) $
 * by : $Author: bpoussin $
 */
public class SolrUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(SolrUtil.class);

    /**
     * Recherche tous les TreeNode auquel appartient en Attachment l'objet passe
     * en parametre
     * 
     * @param doc le document representant l'objet
     * @since 3.1
     */
    static public Set<String> getAttachedTreeNode(SolrDocument doc) {
        Set<String> result = new HashSet<String>();
        for (String field : doc.getFieldNames()) {
            if (field.startsWith(TREENODE_ATTACHED)) {
                String id = field.substring(TREENODE_ATTACHED.length());
                result.add(id);
            }
        }
        return result;
    }

    /**
     * Find solr document by id
     */
    static public SolrDocument findById(SolrServer solrServer, String id) {
        id = quoteForSolr(id);
        SolrQuery query = new SolrQuery(SOLR_ID + ":" + id);
        QueryResponse response;
        try {
            response = solrServer.query(query);
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }

        SolrDocumentList results = response.getResults();
        long numFound = results.getNumFound();
        if(numFound == 1) {
            return results.get(0);
        }

        return null;
    }

    /**
     * Find solr document by id
     * @since 3.1
     */
    static public Map<String, SolrDocument> findAllById(
            SolrServer solrServer, Collection<String> ids) {
        String solrField = SOLR_ID;
        Map<String, SolrDocument> result = findAllByField(solrServer, solrField, ids);
        return result;
    }

    /**
     * Find solr document by TreeNode parents extra field
     * @param ids id that must be find in parents list
     * @return Map key:TreeNode id, value; solr document associate with id
     * @since 3.1
     */
    static public Map<String, SolrDocument> findAllByParents(
            SolrServer solrServer, Collection<String> ids) {
        String solrField = TREENODE_PARENTS;
        Map<String, SolrDocument> result = findAllByField(solrServer, solrField, ids);
        return result;
    }

    /**
     * Find solr document by TreeNode attachment field
     * @param ids id that must be find in attachment list
     * @return Map key:TreeNode id, value; solr document associate with id
     * @since 3.1
     */
    static public Map<String, SolrDocument> findAllByAttachment(
            SolrServer solrServer, Collection<String> ids) {
        String solrField = SolrUtil.getSolrFieldName(
                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT, TYPE.WIKITTY);
        Map<String, SolrDocument> result = findAllByField(solrServer, solrField, ids);
        return result;
    }

    /**
     * Find solr document by specified field in argument
     * @param solrServer
     * @param solrField field where we looking for ids
     * @param ids ids that must be in solrField
     * @return Map key:TreeNode id, value; solr document associate with id
     * @since 3.1
     */
   static public Map<String, SolrDocument> findAllByField(
           SolrServer solrServer, String solrField, Collection<String> ids) {
       Map<String, SolrDocument> result = new HashMap<String, SolrDocument>();
       if (ids != null && ids.size() > 0) {
           try {
               String request = solrField + ":(";
               String or = "";
               for (String id : ids) {
                   id = quoteForSolr(id);
                   request += or + id;
                   or = " OR ";
               }
               request += ")";

               SolrQuery query = new SolrQuery(request);
               QueryResponse response = solrServer.query(query);
               SolrDocumentList results = response.getResults();
               for (SolrDocument doc : results) {
                   String id = (String) doc.getFieldValue(SOLR_ID);
                   result.put(id, doc);
               }
           } catch (SolrServerException eee) {
               throw new WikittyException("Error during find", eee);
           }
       }
       return result;
    }

    /**
     * if you change this method, change
     * {@link TypeFieldModifer#convertToField(org.nuiton.wikitty.services.WikittyTransaction, java.lang.String)}
     * too
     *
     * @param fqfieldName
     * @param type
     * @return
     */
    static public String getSolrFieldName(String fqfieldName, TYPE type) {
        switch (type) {
            case BINARY:
                return fqfieldName + SUFFIX_BINARY;
            case BOOLEAN:
                return fqfieldName + SUFFIX_BOOLEAN;
            case DATE:
                return fqfieldName + SUFFIX_DATE;
            case STRING:
                return fqfieldName + SUFFIX_STRING;
            case NUMERIC:
                return fqfieldName + SUFFIX_NUMERIC;
            case WIKITTY:
                return fqfieldName + SUFFIX_WIKITTY;
            default:
                return fqfieldName;
        }
    }

    /**
     *
     * @param source
     * @param dest
     * @param fields only copy this field, if null or empty, copy all field
     */
    static public void copySolrDocument(
            SolrDocument source, SolrInputDocument dest,
            String[] fieldToInclude, String[] fieldToExclude) {
        Collection<String> fieldNames = source.getFieldNames();

        Set<String> fieldToCopy = new HashSet<String>();
        if (fieldToInclude == null || fieldToInclude.length == 0) {
            fieldToCopy.addAll(fieldNames);
        } else {
            for (String fieldName : fieldNames) {
                for (String fieldRegexp : fieldToInclude) {
                    if (fieldName.matches(fieldRegexp)) {
                        fieldToCopy.add(fieldName);
                        break;
                    }
                }
            }
        }
        
        if (fieldToExclude != null && fieldToExclude.length > 0) {
            for (String fieldName : fieldNames) {
                for (String fieldRegexp : fieldToExclude) {
                    if (fieldName.matches(fieldRegexp)) {
                        fieldToCopy.remove(fieldName);
                        break;
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "Copiable field are %s but only field %s are copied",
                    fieldNames, fieldToCopy));
        }

        for (String fieldName : fieldToCopy) {
            dest.removeField(fieldName); // to prevent add in already exist dest field
            Collection<Object> fieldValues = source.getFieldValues(fieldName);
            for (Object fieldValue : fieldValues) {
                dest.addField(fieldName, fieldValue);
            }
        }
    }

    /**
     *
     * @param source
     * @param dest
     * @param fieldToInclude only copy this field, if null or empty, copy all field
     * @since 3.1
     */
    static public void copySolrDocument(
            SolrDocument source, SolrInputDocument dest, String... fieldToInclude) {
        copySolrDocument(source, dest, fieldToInclude, null);
    }

    /**
     *
     * @param source
     * @param dest
     * @param fields only copy this field, if null or empty, copy all field
     * @since 3.1
     */
    static public void copySolrDocumentExcludeSomeField(
            SolrDocument source, SolrInputDocument dest, String... fieldToExclude) {
        copySolrDocument(source, dest, null, fieldToExclude);
    }

    /**
     * Quote s for solr. Currently only ':' is escaped
     * @param s to quote
     * @return new string solr compliant
     */
    static public String quoteForSolr(String s) {
        String result = s.replaceAll(":", "\\\\:");
        return result;
    }
}
