/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaDAO.java 2142 2010-11-18 16:17:42Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.5/topia-persistence/src/main/java/org/nuiton/topia/persistence/TopiaDAO.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * TopiaDAO.java
 *
 * Created: 30 déc. 2005 03:00:57
 *
 * @author poussin <poussin@codelutin.com>
 * @version $Revision: 2142 $
 *
 * Last update: $Date: 2010-11-18 17:17:42 +0100 (Thu, 18 Nov 2010) $
 * by : $Author: tchemit $
 */
package org.nuiton.topia.persistence;

import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaQuery;
import org.nuiton.topia.generator.DAOAbstractTransformer;
import org.nuiton.topia.generator.DAOImplTransformer;
import org.nuiton.topia.generator.DAOTransformer;

import java.security.Permission;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * TopiaDAO is used to manipulate entities corresponding to {@code E} type :
 * create, delete, update or find entities using properties of {@link
 * TopiaQuery}.
 * <p/>
 * This interface is implemented by {@link TopiaDAOImpl} overriden by generation
 * from {@link DAOTransformer} for specific entity interface, {@link
 * DAOAbstractTransformer} for abstract implementation and {@link
 * DAOImplTransformer} for final implementation class.
 * <p/>
 * TODO-fdesbois-20100508 : Need translation of javadoc.
 *
 * @author poussin <poussin@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: TopiaDAO.java 2142 2010-11-18 16:17:42Z tchemit $
 * @param <E> the entity type linked with the dao
 */
public interface TopiaDAO<E extends TopiaEntity> {

    /**
     * When TopiaContextImpl create the TopiaDAOHibernate, it must call this
     * method just after.
     *
     * @param context     context
     * @param entityClass entity class
     * @throws TopiaException if any pb while init
     */
    void init(TopiaContextImplementor context, Class<E> entityClass)
            throws TopiaException;

    /**
     * Create a new instance of managed entity.
     *
     * @return new entity instance
     * @throws TopiaException if any pb while creating the entitie
     * @since 2.3.1
     */
    E newInstance() throws TopiaException;

    /**
     * Return class of entity managed by this DAO.
     *
     * @return entity managed by this DAO
     */
    Class<E> getEntityClass();

    /**
     * Return context used by this DAO.
     *
     * @return Returns the context.
     */
    TopiaContextImplementor getContext();

    /**
     * Find usages of the given {@code entity} in the entities of the given
     * {@code type}.
     *
     * @param type   the type of entity to search
     * @param entity the entity
     * @param <U>    tthe type of entity to search
     * @return the list of entities of the given type which uses the given
     *         entity
     * @throws TopiaException if any problem while getting data
     * @since 2.3.0
     */
    <U extends TopiaEntity> List<U> findUsages(Class<U> type, E entity)
            throws TopiaException;

    /**
     * Find all usages of the given {@code entity}.
     *
     * @param entity the entity
     * @return the dictionnary of usages of the given entities (keys are entity
     *         usage container, values are the list of this type of entity to
     *         use the given entity).
     * @throws TopiaException if any pb while getting data
     * @since 2.3.0
     */

    Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> findAllUsages(E entity)
            throws TopiaException;

    /**
     * Retourne les permissions a verifier pour l'acces a l'entite pour le
     * service Taas.
     *
     * @param topiaId topiaId d'une entite
     * @param actions encoded actions
     * @return la liste des permissions
     * @throws TopiaException if any pb while getting datas
     */
    List<Permission> getRequestPermission(String topiaId, int actions)
            throws TopiaException;

    /* Adders */

    void addTopiaEntityListener(TopiaEntityListener listener);

    void addTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    /* Removers */

    void removeTopiaEntityListener(TopiaEntityListener listener);

    void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    /**
     * Construit une nouvelle instance de l'objet géré par ce DAO
     *
     * @param properties la liste des propriétés que doit avoir l'objet créé les
     *                   arguments vont par paire (propertyName, value)
     * @return un nouvel objet
     * @throws TopiaException           si un problème est rencontré durant
     *                                  l'instanciation
     * @throws IllegalArgumentException Si le nombre on le type des arguments
     *                                  n'est pas bon ou que le type ou le nom
     *                                  d'une propriété est fausse
     */
    E create(Object... properties) throws TopiaException;

    /**
     * Construit une nouvelle instance de l'objet géré par ce DAO
     *
     * @param properties la liste des propriétés que doit avoir l'objet créé
     * @return un nouvel objet
     * @throws TopiaException           si un problème est rencontré durant
     *                                  l'instanciation
     * @throws IllegalArgumentException Si le nombre on le type des arguments
     *                                  n'est pas bon ou que le type ou le nom
     *                                  d'une propriété est fausse
     */
    E create(Map<String, Object> properties) throws TopiaException;

    /**
     * Permet de sauver un object instancié sans le DAO.
     *
     * @param e l'entité instanciée à sauver
     * @return l'entité avec son topiaID valorisé
     * @throws TopiaException if any pb while creating datas
     * @since 2.3.1
     */
    E create(E e) throws TopiaException;

    /**
     * Permet d'ajouter ou de mettre a jour un objet. Cela permet d'ajouter par
     * exemple un objet provenant d'un autre context mais du meme type de DAO.
     *
     * @param e l'entite a ajouter ou mettre a jour
     * @return l'entity passé en paramètre.
     * @throws TopiaException if any pb while updating datas
     */
    E update(E e) throws TopiaException;

    /**
     * Permet de supprimer une entite.
     *
     * @param e l'entite a supprimer
     * @throws TopiaException if any pb while deleting datas
     */
    void delete(E e) throws TopiaException;

    /**
     * Crée une requete basé sur l'entité lié au DAO. Résultat attendu : "FROM
     * E"
     *
     * @return une nouvelle TopiaQuery vide. (uniquement avec le From sur le
     *         type d'entité)
     * @since 2.3
     */
    TopiaQuery createQuery();

    /**
     * Crée une requête basé sur l'entité lié au DAO et lui assigne un alias
     * valable dans la requête..
     * <p/>
     * Résultat attendu : "FROM E AS entityAlias"
     *
     * @param entityAlias alias permettant de manipuler l'entité dans la
     *                    requête
     * @return une nouvelle TopiaQuery
     * @since 2.3
     */
    TopiaQuery createQuery(String entityAlias);

    /**
     * Find an entity corresponding to the {@code id}. If the {@code id} is
     * null, nothing will be search.
     *
     * @param id    topiaId of the entity to found
     * @return the entity found or null if not
     * @throws TopiaException for Topia errors on query
     */
    E findByTopiaId(String id) throws TopiaException;

    E findByProperty(String propertyName, Object value)
            throws TopiaException;

    /**
     * @param propertyName le nom de la propriété
     * @param value        la valeur à tester
     * @param others       les autres proprietes doivent aller par 2
     *                     propertyName, value
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    E findByProperties(String propertyName, Object value,
                       Object... others) throws TopiaException;

    E findByProperties(Map<String, Object> properties) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * entité correspondant à la requête.
     *
     * @param query la requête
     * @return l'entité correspondant à la recherche ou null si aucune entité
     *         n'a été trouvée
     * @throws TopiaException if any pb while getting datas
     * @see TopiaQuery#executeToEntity(TopiaContext, Class)
     * @since 2.3
     */
    E findByQuery(TopiaQuery query) throws TopiaException;

    List<E> findAll() throws TopiaException;

    /**
     * Recuperation de tous les ids en base pour le type d'entite du dao.
     *
     * @return la liste de tous les ids
     * @throws TopiaException si pb en base
     */
    List<String> findAllIds() throws TopiaException;

    List<E> findAllByProperty(String propertyName, Object value)
            throws TopiaException;

    /**
     * @param propertyName le nom de la propriété
     * @param value        la valeur à tester
     * @param others       les autres proprietes doivent aller par 2
     *                     propertyName, value
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAllByProperties(String propertyName, Object value,
                                Object... others) throws TopiaException;

    List<E> findAllByProperties(Map<String, Object> properties)
            throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * liste d'entités correspondant à la requête.
     *
     * @param query la requête
     * @return la liste d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see TopiaQuery#executeToEntityList(TopiaContext, Class)
     * @since 2.3
     */
    List<E> findAllByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une map
     * d'entités correspondant à la requête. La clé de la map étant le topiaId
     * de l'entité.
     *
     * @param query la requête
     * @return la map d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see TopiaQuery#executeToEntityMap(TopiaContext, Class)
     * @since 2.3
     */
    Map<String, E> findAllMappedByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une map
     * d'entités correspondant à la requête. Le type et le nom de la propriété
     * utilisé comme clé de la map doit être passé en argument.
     *
     * @param <K>      type de la clé de la map
     * @param query    la requête
     * @param keyName  nom de la propriété de l'entité utilisée comme clé
     * @param keyClass type de la propriété de l'entité utilisée comme clé
     * @return la map d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see TopiaQuery#executeToEntityMap(TopiaContext, Class)
     * @since 2.3
     */
    <K> Map<K, E> findAllMappedByQuery(TopiaQuery query,
                                       String keyName, Class<K> keyClass) throws TopiaException;

    List<E> findAllWithOrder(String... propertyNames)
            throws TopiaException;

    /**
     * Check the existence of an entity with technical {@code id}.
     *
     * @param id unique id of the entity to test existence.
     * @return true if entity exists, false otherwise
     * @throws TopiaException for Topia errors
     * @since 2.3.4
     */
    boolean existByTopiaId(String id) throws TopiaException;

    /**
     * Check the existence of an entity with {@code propertyName} with {@code
     * propertyValue}. {@code others} properties can be added to test
     * existence.
     *
     * @param propertyName  first property name to test existence
     * @param propertyValue first property value to test existence
     * @param others        altern propertyName and propertyValue
     * @return true if entity exists, false otherwise
     * @throws TopiaException for Topia errors
     * @since 2.3.4
     */
    boolean existByProperties(String propertyName, Object propertyValue,
                              Object... others) throws TopiaException;


    /**
     * Check the existence of an entity using a {@code query}.
     *
     * @param query query used to test existence
     * @return true if entity exists, false otherwise
     * @throws TopiaException
     * @since 2.3.4
     */
    boolean existByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Compte le nombre d'entites en base
     *
     * @return le nombre total d'entites existantes
     * @throws TopiaException if any pb while getting datas
     * @deprecated since 2.3.4 ambiguous name, use {@link #count()} instead.
     */
    @Deprecated
    long size() throws TopiaException;

    /**
     * Count the number of existing entities.
     *
     * @return number of total entities
     * @throws TopiaException if any pb while getting datas
     * @since 2.3.4
     */
    int count() throws TopiaException;

    /**
     * Count the number of entities based on {@code query}.
     *
     * @return number of entities filtered by the query
     * @throws TopiaException if any pb while getting datas
     * @since 2.3.4
     */
    int countByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Recherche la classe en utilisant la cle naturelle, chaque champs de la
     * cle naturelle est une entre de la map passe en argument.
     *
     * @param keys la liste des champs de la cle naturelle avec leur valeur
     * @return l'entite trouvé
     * @throws TopiaException if any pb while getting datas
     */
    E findByPrimaryKey(Map<String, Object> keys) throws TopiaException;

    /**
     * Recherche la classe en utilisant la cle naturelle, si la cle naturelle
     * est composé de plusieurs champs alors les arguments passés doivent être
     * dans l'ordre de declaration dans le fichier de mapping
     *
     * @param k l'objet cle naturelle de la classe
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    E findByPrimaryKey(Object... k) throws TopiaException;

} //TopiaDAO

