/*
 * #%L
 * Nuiton Utils :: Nuiton Utils
 * 
 * $Id: Import.java 2244 2011-11-30 17:12:49Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.4/nuiton-csv/src/main/java/org/nuiton/util/csv/Import.java $
 * %%
 * Copyright (C) 2011 CodeLutin, Tony Chemit, Brendan Le Ny
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util.csv;

import com.csvreader.CsvReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Import engine for a given import model.
 *
 * It acts as an {@link Iterable}, you can use directly inside a foreach.
 *
 * The method {@link #prepareAndValidate()} will be invoked before all and
 * only once. It mainly obtain header from the csv input, pass it to the model
 * and then validate the model.
 *
 * @author bleny <leny@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <desbois@codelutin.com>
 * @since 2.4
 */
public class Import<E> implements Iterable<E>, Closeable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(Import.class);

    /** Csv import model. */
    protected ImportModel<E> model;

    /** Csv reader (this is the input). */
    protected CsvReader reader;

    /**
     * A flag to know if model was already validated.
     * <p/>
     * Save once done to prevent multiple headers read leading to consider
     * first lines as headers.
     */
    protected boolean validate;

    public static <E> Import<E> newImport(ImportModel<E> model,
                                          InputStream inputStream) {
        return new Import<E>(model, inputStream);
    }

    public static <E> Import<E> newImport(ImportModel<E> model,
                                          Reader reader) {
        return new Import<E>(model, reader);
    }

    @Override
    public Iterator<E> iterator() {

        prepareAndValidate();

        readFirstLine();

        return new Iterator<E>() {

            boolean hasNext = true;

            int lineNumber;

            E lastElement;

            @Override
            public boolean hasNext() {
                return hasNext;
            }

            @Override
            public E next()
                    throws NoSuchElementException, ImportRuntimeException {

                if (!hasNext) {
                    throw new NoSuchElementException();
                }

                lineNumber += 1;

                E element = model.newEmptyInstance();

                for (ImportableColumn<E, Object> field : getNonIgnoredHeaders()) {

                    // read value from csv cell
                    String value;
                    try {
                        value = reader.get(field.getHeaderName());
                    } catch (Exception e) {
                        reader.close();
                        throw new ImportRuntimeException(
                                _("csv.import.error.unableToReadField",
                                  field.getHeaderName(), lineNumber), e);
                    }

                    // contravariance ftw
                    Object parsedValue;
                    try {
                        parsedValue = field.parseValue(value);
                    } catch (Exception e) {
                        String message = _("csv.import.error.unableToParseValue",
                                           value, field.getHeaderName(), lineNumber)
                                         + "\n" + e.getMessage();
                        throw new ImportRuntimeException(message, e);
                    }

                    // set value to element
                    try {
                        field.setValue(element, parsedValue);
                    } catch (Exception e) {
                        String message = _("csv.import.error.unableToSetValue",
                                           parsedValue,
                                           element.toString(),
                                           lineNumber, field.getHeaderName());
                        if (log.isErrorEnabled()) {
                            log.error(message);
                        }
                        throw new ImportRuntimeException(message, e);
                    }

                    lastElement = element;
                }

                try {
                    hasNext = reader.readRecord();
                } catch (IOException e) {
                    reader.close();
                    throw new ImportRuntimeException(
                            _("csv.import.error.unableToReadLine",
                              lineNumber + 1), e);
                }

                return lastElement;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    @Override
    public void close() {
        reader.close();
    }

    public void prepareAndValidate() {

        if (validate) {

            // was already validated
            return;
        }

        // mark as validated
        validate = true;

        // obtains headers
        String[] headers = getHeaders();

        if (log.isTraceEnabled()) {
            log.trace("headers of the CSV file are : " +
                      Arrays.toString(headers));
        }

        // hook to do some stuff from the model
        model.pushCsvHeaderNames(Arrays.asList(headers));

        // check model columns name are unique
        checkUniqueModelColumnNames();

        // check that given headers from csv file are all known

        checkHeaderNamesAreAllKnown(headers);

        // check all mandatories column are on csv header
        checkAllManadatoryHeadersArePresent(headers);
    }

    protected void checkHeaderNamesAreAllKnown(String[] headers) {
        List<String> csvHeaders = new ArrayList<String>();
        Collections.addAll(csvHeaders, headers);

        for (ImportableColumn<E, ?> field : model.getColumnsForImport()) {
            csvHeaders.remove(field.getHeaderName());
        }
        if (!csvHeaders.isEmpty()) {
            List<String> validHeaderNames = new LinkedList<String>();
            for (ImportableColumn<E, ?> importableColumn :
                    model.getColumnsForImport()) {
                validHeaderNames.add(importableColumn.getHeaderName());
            }
            String validationMessage = _("csv.import.error.unrecognizedHeaders",
                                         StringUtil.join(csvHeaders, ", ", true),
                                         StringUtil.join(validHeaderNames, ", ", true));
            throw new ImportRuntimeException(validationMessage);
        }
    }

    protected void checkUniqueModelColumnNames() {
        Set<String> headerNames = new HashSet<String>();
        Set<String> doubleHeaderNames = new HashSet<String>();
        for (ImportableColumn<E, ?> importableColumn :
                model.getColumnsForImport()) {
            String headerName = importableColumn.getHeaderName();
            boolean alreadyUsed = !headerNames.add(headerName);
            if (alreadyUsed) {
                doubleHeaderNames.add(headerName);
            }
        }
        if (!doubleHeaderNames.isEmpty()) {
            String message = _("csv.import.error.duplicatedHeaders",
                               StringUtil.join(doubleHeaderNames, ", ", true));

            throw new ImportRuntimeException(
                    message);
        }
    }

    protected void checkAllManadatoryHeadersArePresent(String[] headers) {

        List<String> csvHeaders = new ArrayList<String>();
        Collections.addAll(csvHeaders, headers);

        List<String> mandatoryHeadersNames = new ArrayList<String>();
        for (ImportableColumn<E, ?> field : getAllMandatoryHeaders()) {
            mandatoryHeadersNames.add(field.getHeaderName());
        }
        mandatoryHeadersNames.removeAll(csvHeaders);

        if (!mandatoryHeadersNames.isEmpty()) {
            String validationMessage = _("csv.import.error.missingMandatoryHeaders",
                                         StringUtil.join(mandatoryHeadersNames, ", ", true));
            throw new ImportRuntimeException(validationMessage);
        }

        Set<String> headerNames = new HashSet<String>();
        for (ImportableColumn<E, ?> importableColumn :
                model.getColumnsForImport()) {
            String headerName = importableColumn.getHeaderName();
            boolean alreadyUsed = !headerNames.add(headerName);
            if (alreadyUsed) {
                throw new ImportRuntimeException(
                        "model contains multiple columnsForImport named '" +
                        headerName + "'");
            }
        }
    }

    protected String[] getHeaders() throws ImportRuntimeException {
        try {
            boolean canReadHeaders = reader.readHeaders();
            if (!canReadHeaders) {
                throw new ImportRuntimeException("can't read headers");
            }
        } catch (IOException e) {
            throw new ImportRuntimeException("can't read headers");
        }

        try {
            String[] result = reader.getHeaders();
            return result;
        } catch (IOException eee) {
            throw new ImportRuntimeException("can't get headers", eee);
        }
    }

    protected List<ImportableColumn<E, Object>> getNonIgnoredHeaders() {
        List<ImportableColumn<E, Object>> nonIgnoredHeaders = new ArrayList<ImportableColumn<E, Object>>();
        for (ImportableColumn<E, Object> field : model.getColumnsForImport()) {
            if (!field.isIgnored()) {
                nonIgnoredHeaders.add(field);
            }
        }
        return nonIgnoredHeaders;
    }

    protected List<ImportableColumn<E, ?>> getAllMandatoryHeaders() {
        List<ImportableColumn<E, ?>> allMandatoryHeaders = new ArrayList<ImportableColumn<E, ?>>();
        for (ImportableColumn<E, ?> field : model.getColumnsForImport()) {
            if (field.isMandatory()) {
                allMandatoryHeaders.add(field);
            }
        }
        return allMandatoryHeaders;
    }

    protected Import(ImportModel<E> model, InputStream inputStream) {
        if (inputStream == null) {
            throw new NullPointerException("inputStream is null");
        }
        this.model = model;
        reader = new CsvReader(inputStream, model.getSeparator(), Charset.forName("UTF-8"));
        reader.setTrimWhitespace(true);
    }

    protected Import(ImportModel<E> model, Reader reader) {
        if (reader == null) {
            throw new NullPointerException("reader is null");
        }
        this.model = model;
        this.reader = new CsvReader(reader, model.getSeparator());
        this.reader.setTrimWhitespace(true);
    }

    protected void readFirstLine() throws ImportRuntimeException {
        try {
            boolean emptyFile = !reader.readRecord();

            if (emptyFile) {
                throw new ImportRuntimeException("CSV file has no line");
            }
        } catch (IOException e) {
            reader.close();
            throw new ImportRuntimeException(_("csv.import.error.unableToReadLine", 1), e);
        }
    }

}
