/*
 * #%L
 * Nuiton Utils :: Nuiton Utils
 * 
 * $Id: Export.java 2255 2011-12-11 18:07:30Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-parent-2.4.1/nuiton-csv/src/main/java/org/nuiton/util/csv/Export.java $
 * %%
 * Copyright (C) 2011 CodeLutin, Tony Chemit, Brendan Le Ny
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util.csv;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

/**
 * Object to realize a export from a {@link ExportModel} and some datas.
 * <h2>Usefull static methods</h2>
 * There is some usefull methods here to do operation in one line :
 * <h3>To export to a file</h3>
 * <pre>
 *     Export.exportToFile(model, data, file);
 * </pre>
 * <h3>To export to a writer</h3>
 * <pre>
 *     Export.exportToWriter(model, data, writer);
 * </pre>
 * <h3>To export as a string</h3>
 * <pre>
 *     String exportcontent = Export.exportToString(model, data);
 * </pre>
 * <h3>To obtain a new instance of an exporter</h3>
 * <pre>
 *     Export&lt;E&gt; exporter = Export.newExport(model, data);
 * </pre>
 *
 * @author bleny <leny@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4
 */
public class Export<E> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(Export.class);

    protected ExportModel<E> model;

    protected Iterable<E> data;

    public static <E> Export<E> newExport(ExportModel<E> model,
                                          Iterable<E> data) {
        return new Export<E>(model, data);
    }

    public static <E> void exportToFile(ExportModel<E> model,
                                        Iterable<E> data,
                                        File file) throws Exception {
        Export<E> exporter = newExport(model, data);
        exporter.exportToFile(file);
    }

    public static <E> String exportToString(ExportModel<E> model,
                                            Iterable<E> data) throws Exception {
        Export<E> exporter = newExport(model, data);
        return exporter.startExportAsString();
    }

    public static <E> void exportToWriter(ExportModel<E> model,
                                          Iterable<E> data,
                                          Writer writer) throws Exception {
        Export<E> exporter = newExport(model, data);
        exporter.startExport(writer);
    }

    protected Export(ExportModel<E> model, Iterable<E> data) {
        this.model = model;
        this.data = data;
    }

    public void exportToFile(File file) throws Exception {
        Writer writer = new BufferedWriter(new FileWriter(file));
        try {
            startExport(writer);
        } finally {
            writer.close();
        }
    }

    public void startExport(Writer writer) throws Exception {
        String separator = String.valueOf(model.getSeparator());

        // add headers
        List<String> headerNames = new LinkedList<String>();
        for (ExportableColumn<E, ?> column : model.getColumnsForExport()) {
            headerNames.add(column.getHeaderName());
        }
        String headersLine = StringUtil.join(headerNames, separator, true);
        writer.write(headersLine);
        writer.write('\n');
        if (log.isDebugEnabled()) {
            log.debug("headers for export are '" + headersLine + "'");
            if (data instanceof Collection) {
                log.debug("will export " + ((Collection<E>) data).size() + " lines");
            }
        }

        Iterable<ExportableColumn<E, Object>> columnsForExport =
                model.getColumnsForExport();

        for (E object : data) {
            for (ExportableColumn<E, Object> column :
                    columnsForExport) {
                Object value = column.getValue(object);
                String formattedValue = column.formatValue(value);
                if (formattedValue == null) {
                    throw new NullPointerException(
                            "column for header " + column.getHeaderName() +
                                    " returned a null value." + column.toString());
                }
                formattedValue =
                        StringUtil.escapeCsvValue(formattedValue, separator);
                writer.write(formattedValue);
                writer.write(separator);
            }
            writer.write('\n');
        }
    }

    public String startExportAsString() throws Exception {
        StringWriter writer = new StringWriter();
        try {
            startExport(writer);
            String result = writer.toString();
            return result;
        } finally {
            writer.close();
        }
    }

    public InputStream startExport() throws Exception {
        String content = startExportAsString();
        return IOUtils.toInputStream(content);
    }
}
