/*
 * #%L
 * 
 * 
 * $Id: MatrixViewerPanel.java 322 2011-01-04 17:39:14Z echatellier $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-matrix/tags/matrix-2.1/nuiton-matrix-gui/src/main/java/org/nuiton/math/matrix/viewer/MatrixViewerPanel.java $
 * %%
 * Copyright (C) 2010 Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.math.matrix.viewer;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSplitPane;
import javax.swing.JToggleButton;
import javax.swing.JToggleButton.ToggleButtonModel;

import org.nuiton.math.matrix.MatrixND;
import org.nuiton.util.Resource;

/**
 * Panel that can display matrix list details (dimension) and rendering solutions.
 * 
 * @author chatellier
 * @version $Revision: 322 $
 * 
 * Last update : $Date: 2011-01-04 18:39:14 +0100 (mar., 04 janv. 2011) $
 * By : $Author: echatellier $
 */
public class MatrixViewerPanel extends JPanel {

    /** serialVersionUID. */
    private static final long serialVersionUID = -5447856858278176837L;

    public static final String PROPERTY_MATRIX_COMBO_VISIBLE = "matrixComboVisible";

    public static final String PROPERTY_MATRIX_RENDERER_SOLUTION = "matrixRendererSolution";

    public static final String PROPERTY_MATRIX_RENDERERS = "matrixRenderers";

    public static final String PROPERTY_MATRICES = "matrices";

    /** Matrix list combo visibility (defaut to false). */
    protected boolean matrixComboVisible;

    /** Matrix renderer list solution. (default to {@link MatrixRendererSolution#RADIO_BUTTON} */
    protected MatrixRendererSolution matrixRendererSolution = MatrixRendererSolution.RADIO_BUTTON;

    /** Matrix renderer plugins. */
    protected List<MatrixRenderer> matrixRenderers;

    /** Matrix filters (used after matrix reduction). */
    protected List<MatrixFilter> matrixFilters;

    /** Matrices to render (depend on {@link #matrixComboVisible} to {@code true}). */
    protected List<MatrixND> matrices;

    /** Matrix list combo box. */
    protected MatrixComboBox matrixComboBox;

    protected MatrixDimensionPanel dimensionPanel;

    protected RadioButtonRenderingPanel radioPanel;

    protected IconButtonRenderingPanel iconPanel;

    protected JPanel renderingComponentContainer;

    /**
     * Map entre les renderers et les composants (valorisé par bouton de
     * rendu) et utilisé par le choix du renderer.
     */
    protected Map<MatrixRenderer, Component> componentForRenderers;

    public MatrixViewerPanel() {
        matrixRenderers = new ArrayList<MatrixRenderer>();
        matrixFilters = new ArrayList<MatrixFilter>();
        matrices = new ArrayList<MatrixND>();
        componentForRenderers = new HashMap<MatrixRenderer, Component>();

        buildPanel();
    }

    protected boolean isMatrixComboVisible() {
        return matrixComboVisible;
    }

    public void setMatrixComboVisible(boolean matrixComboVisible) {
        boolean oldValue = this.matrixComboVisible;
        this.matrixComboVisible = matrixComboVisible;
        firePropertyChange(PROPERTY_MATRIX_COMBO_VISIBLE, oldValue, matrixComboVisible);
    }

    public MatrixRendererSolution getMatrixRendererSolution() {
        return matrixRendererSolution;
    }

    public void setMatrixRendererSolution(MatrixRendererSolution matrixRendererSolution) {
        MatrixRendererSolution oldValue = this.matrixRendererSolution;
        this.matrixRendererSolution = matrixRendererSolution;
        firePropertyChange(PROPERTY_MATRIX_RENDERER_SOLUTION, oldValue, matrixRendererSolution);
    }

    public boolean addMatrixRenderer(MatrixRenderer matrixRenderer) {
        boolean result = matrixRenderers.add(matrixRenderer);
        firePropertyChange(PROPERTY_MATRIX_RENDERERS, null, matrixRenderers);
        return result;
    }

    public boolean removeMatrixRenderer(Object matrixRenderer) {
        boolean result = matrixRenderers.remove(matrixRenderer);
        firePropertyChange(PROPERTY_MATRIX_RENDERERS, null, matrixRenderers);
        return result;
    }

    /**
     * Add new matrix filter.
     * Used after matrix reduction and before matrix rendering.
     * 
     * @param matrixFilter matrix filter
     * @return {@code true} (as specified by {@link Collection#add(Object)}
     */
    public boolean addMatrixFilter(MatrixFilter matrixFilter) {
        boolean result = matrixFilters.add(matrixFilter);
        return result;
    }

    public boolean removeMatrixFilter(Object matrixFilter) {
        boolean result = matrixFilters.remove(matrixFilter);
        return result;
    }

    public void addMatrix(MatrixND... matrices) {
        for (MatrixND matrix : matrices) {
            this.matrices.add(matrix);
        }
        firePropertyChange(PROPERTY_MATRICES, null, matrices);
    }

    public void removeMatrix(Object... matrices) {
        for (Object matrix : matrices) {
            this.matrices.remove(matrix);
        }
        firePropertyChange(PROPERTY_MATRICES, null, matrices);
    }

    public void clearMatrix() {
        matrices.clear();
        firePropertyChange(PROPERTY_MATRICES, null, matrices);
    }

    /** Matrix list combo box. */
    protected class MatrixComboBox extends JComboBox implements PropertyChangeListener, ItemListener {

        /** serialVersionUID. */
        private static final long serialVersionUID = -3166642639854195273L;

        public MatrixComboBox() {
            addItemListener(this);
        }

        /*
         * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
         */
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            setVisible(matrixComboVisible);
        }

        /*
         * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
         */
        @Override
        public void itemStateChanged(ItemEvent e) {
            dimensionPanel.setMatrix((MatrixND)getSelectedItem());
        }
    }

    /** Matrix list combo box model. */
    protected class MatrixComboBoxModel extends DefaultComboBoxModel implements PropertyChangeListener {

        /** serialVersionUID. */
        private static final long serialVersionUID = 4294576545040155208L;

        @Override
        public int getSize() {
            int result = matrices.size();
            return result;
        }

        @Override
        public Object getElementAt(int index) {
            Object matrix = matrices.get(index);
            return matrix;
        }

        /*
         * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
         */
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            fireContentsChanged(this, 0, matrices.size() - 1);
            
            if (getSelectedItem() == null && getSize() > 0) {
                setSelectedItem(getElementAt(0));
            }
        }
    }

    /** Matrix list combo renderer. */
    protected static class MatrixComboRenderer extends DefaultListCellRenderer {

        /** serialVersionUID. */
        private static final long serialVersionUID = 6151127818315270895L;

        @Override
        public Component getListCellRendererComponent(JList list, Object value,
                int index, boolean isSelected, boolean cellHasFocus) {

            MatrixND matrix = (MatrixND)value;
            String matrixName = null;
            if (matrix != null) {
                matrixName = _(matrix.getName());
            }
            return super.getListCellRendererComponent(list, matrixName, index, isSelected, cellHasFocus);
        }
    }

    /** Button model from button containing rendered instance. */
    protected static class RendererButtonModel extends ToggleButtonModel {

        /** serialVersionUID. */
        private static final long serialVersionUID = -5737246124430280412L;

        protected MatrixRenderer renderer;
        
        public RendererButtonModel(MatrixRenderer renderer) {
            this.renderer = renderer;
        }
        
        public MatrixRenderer getRenderer() {
            return renderer;
        }
    }

    /**
     * Icon button rendering panel.
     * Also contains main render action button (arrow).
     */
    protected class IconButtonRenderingPanel extends JPanel implements PropertyChangeListener, ActionListener {

        /** serialVersionUID. */
        private static final long serialVersionUID = 2591696695747738619L;

        protected ButtonGroup buttonGroup;
        
        public IconButtonRenderingPanel() {
            super(new GridBagLayout());
        }
        
        /*
         * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
         */
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            rebuildPanel();
            validate();
            repaint();
        }
        
        /**
         * Rebuild radio button lists.
         */
        protected void rebuildPanel() {
            removeAll();
            
            JButton renderButton = new JButton(Resource.getIcon("/icons/1rightarrow.png"));
            renderButton.setActionCommand("render");
            renderButton.addActionListener(this);
            add(renderButton, new GridBagConstraints(0, 0, 1, 1, 1, 1,
                    GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));

            if (matrixRendererSolution == MatrixRendererSolution.ICON) {

                int index = 1;
                buttonGroup = new ButtonGroup();
                for (MatrixRenderer renderer : matrixRenderers) {
                    JToggleButton radioButton = new JToggleButton(renderer.getIcon());
                    radioButton.addActionListener(this);
                    radioButton.setModel(new RendererButtonModel(renderer));
                    
                    // auto select first matrix renderer
                    if (index == 1) {
                        radioButton.setSelected(true);
                    }

                    buttonGroup.add(radioButton);
                    add(radioButton, new GridBagConstraints(0, index, 1, 1, 1, 0,
                            GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
                    index++;
                }
            }
        }

        public MatrixRenderer getSelectedRender() {
            MatrixRenderer renderer = null;
            RendererButtonModel model = (RendererButtonModel)buttonGroup.getSelection();
            if (model != null) {
                renderer = model.getRenderer();
            }
            return renderer;
        }

        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {

            String actionCommand = e.getActionCommand();

            if ("render".equals(actionCommand)) {
                // get matrix to display
                MatrixND matrix = dimensionPanel.getModifiedMatrix();
                // filter matrix
                matrix = getFilteredMatrix(matrix);

                // matrice superieur a 2 dimensions non geree!!
                if (matrix.getDimCount() > 2) {
                    JOptionPane.showMessageDialog(this, _("nuitonmatrix.viewer.matrix.more.2d"),
                            _("nuitonmatrix.error"), JOptionPane.ERROR_MESSAGE);
                }
                else {
                    // get all display component for each renderer
                    componentForRenderers.clear();
                    for (MatrixRenderer matrixRenderer : matrixRenderers) {
                        Component component = matrixRenderer.getComponent(matrix);
                        componentForRenderers.put(matrixRenderer, component);
                    }
    
                    updateSelectedRenderingComponent();
                }
            }
            else {
                // clic on render button
                updateSelectedRenderingComponent();
            }
        }

        /**
         * Filter matrix if any filter is defined.
         * (after reduction and before rendering)
         * 
         * @param matrix matrix to filter
         * @return filtered matrix
         */
        protected MatrixND getFilteredMatrix(MatrixND matrix) {
            MatrixND filteredMatrix = matrix;
            for (MatrixFilter matrixFilter : matrixFilters) {
                filteredMatrix = matrixFilter.filter(filteredMatrix);
            }
            return filteredMatrix;
        }
    }

    /** Radio button rendering panel. */
    protected class RadioButtonRenderingPanel extends JPanel implements PropertyChangeListener, ActionListener {

        /** serialVersionUID. */
        private static final long serialVersionUID = -6312518069621077533L;

        protected ButtonGroup buttonGroup;

        public RadioButtonRenderingPanel() {
            super(new GridBagLayout());
        }

        /*
         * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
         */
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            rebuildPanel();
            validate();
            repaint();
        }

        /**
         * Rebuild radio button lists.
         */
        protected void rebuildPanel() {
            removeAll();
            if (matrixRendererSolution == MatrixRendererSolution.RADIO_BUTTON) {

                int index = 0;
                buttonGroup = new ButtonGroup();
                for (MatrixRenderer renderer : matrixRenderers) {
                    JRadioButton radioButton = new JRadioButton(renderer.getName());
                    radioButton.addActionListener(this);
                    radioButton.setModel(new RendererButtonModel(renderer));
                    buttonGroup.add(radioButton);
                    
                    // auto select first matrix renderer
                    if (index == 0) {
                        radioButton.setSelected(true);
                    }

                    add(radioButton, new GridBagConstraints(index, 0, 1, 1, 1, 1,
                            GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
                    index++;
                }

                setVisible(true);
            }
            else {
                setVisible(false);
            }
        }

        public MatrixRenderer getSelectedRender() {
            MatrixRenderer renderer = null;
            RendererButtonModel model = (RendererButtonModel)buttonGroup.getSelection();
            if (model != null) {
                renderer = model.getRenderer();
            }
            return renderer;
        }

        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            updateSelectedRenderingComponent();
        }
    }

    /**
     * Build main panel.
     */
    protected void buildPanel() {

        setLayout(new BorderLayout());

        // split main ui left/rigth
        JPanel editionSidePanel = new JPanel(new BorderLayout());
        JPanel renderSidePanel = new JPanel(new BorderLayout());
        JSplitPane mainSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, editionSidePanel, renderSidePanel);
        mainSplitPane.setDividerLocation(0.3);
        add(mainSplitPane, BorderLayout.CENTER);

        // matrix list combo
        matrixComboBox = new MatrixComboBox();
        MatrixComboBoxModel matrixListComboBoxModel = new MatrixComboBoxModel();
        matrixComboBox.setModel(matrixListComboBoxModel);
        matrixComboBox.setRenderer(new MatrixComboRenderer());
        matrixComboBox.setVisible(matrixComboVisible);
        addPropertyChangeListener(PROPERTY_MATRIX_COMBO_VISIBLE, matrixComboBox);
        addPropertyChangeListener(PROPERTY_MATRICES, matrixListComboBoxModel);
        editionSidePanel.add(matrixComboBox, BorderLayout.NORTH);

        // panel d'affichage des dimensions
        dimensionPanel = new MatrixDimensionPanel();
        editionSidePanel.add(dimensionPanel, BorderLayout.CENTER);

        // fleche de d'action de rendu
        // render type : icon
        iconPanel = new IconButtonRenderingPanel();
        editionSidePanel.add(iconPanel, BorderLayout.EAST);
        addPropertyChangeListener(PROPERTY_MATRIX_RENDERER_SOLUTION, iconPanel);
        addPropertyChangeListener(PROPERTY_MATRIX_RENDERERS, iconPanel);

        // render type : combo box

        // current rendering pane
        renderingComponentContainer = new JPanel(new BorderLayout());
        renderSidePanel.add(renderingComponentContainer, BorderLayout.CENTER);

        // render type : radio button
        radioPanel = new RadioButtonRenderingPanel();
        renderSidePanel.add(radioPanel, BorderLayout.SOUTH);
        addPropertyChangeListener(PROPERTY_MATRIX_RENDERER_SOLUTION, radioPanel);
        addPropertyChangeListener(PROPERTY_MATRIX_RENDERERS, radioPanel);
    }

    /**
     * Set rendering component in rendering container.
     */
    protected void updateSelectedRenderingComponent() {
        renderingComponentContainer.removeAll();
        
        MatrixRenderer matrixRenderer = null;
        switch (matrixRendererSolution) {
        case ICON:
            matrixRenderer = iconPanel.getSelectedRender();
            break;
        case RADIO_BUTTON:
            matrixRenderer = radioPanel.getSelectedRender();
            break;
        }

        if (matrixRenderer != null) {
            Component component = componentForRenderers.get(matrixRenderer);
            if (component != null) {
                renderingComponentContainer.add(component, BorderLayout.CENTER);
            }
        }

        renderingComponentContainer.validate();
        renderingComponentContainer.repaint();
    }
}
