/* *##% 
 * JAXX Maven plugin
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.jaxx.plugin;

import com.sun.java.help.search.Indexer;
import java.io.BufferedReader;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.i18n.I18n;
import org.nuiton.plugin.VelocityTemplateGenerator;
import org.nuiton.util.FileUtil;
import org.nuiton.util.SortedProperties;

/**
 * Mojo to generate javax help stuff for your project.
 *
 * HelpIds should have been discovered by the JaxxMojo.
 *
 * @author chemit
 * @goal generate-help
 * @phase process-sources
 * 
 * @requiresProject
 * @requiresDependencyResolution compile
 * @since 1.3
 */
public class GenerateHelpMojo extends AbstractJaxxMojo {

    private static final String AUTOREMOVE_LINE = "REMOVE THS LINE TO DISABLE AUTO-REGENERATE THE FILE";
    /**
     * The directory where to generate help files.
     * 
     * @parameter expression="${jaxx.outHelp}" default-value="${project.basedir}/src/main/help"
     * @required
     * 
     * @since 1.3
     */
    protected File outHelp;
    /**
     * The locales to generate for help, seprated by comma.
     *
     * The first locale given is the default locale.
     * 
     * @parameter expression="${jaxx.locales}"
     * @required
     *
     * @since 2.0.0
     */
    protected String locales;
    /**
     * The name of the helpset to generate.
     *
     * @parameter expression="${jaxx.helpsetName}" default-value="${project.artifactId}"
     * @required
     *
     * @since 1.3
     */
    protected String helpsetName;
    /**
     * The template used to generate helpset file.
     *
     * Must be an existing file or a ressource in class-path
     * 
     * @parameter expression="${jaxx.helpsetTemplate}" default-value="/defaultHelpSet.hs.vm"
     * @required
     *
     * @since 1.3
     */
    protected File helpsetTemplate;
    /**
     * The template used to generate helpset map file.
     * 
     * Must be an existing file or a ressource in class-path
     * 
     * @parameter expression="${jaxx.mapTemplate}" default-value="/defaultMap.jhm.vm"
     * @required
     *
     * @since 1.3
     */
    protected File mapTemplate;
    /**
     * The template used to generate helpset index file.
     * 
     * Must be an existing file or a ressource in class-path
     * 
     * @parameter expression="${jaxx.indexTemplate}" default-value="/defaultIndex.xml.vm"
     * @required
     *
     * @since 1.3
     */
    protected File indexTemplate;
    /**
     * The template used to generate helpset toc file.
     * 
     * Must be an existing file or a ressource in class-path
     * 
     * @parameter expression="${jaxx.tocTemplate}" default-value="/defaultToc.xml.vm"
     * @required
     *
     * @since 1.3
     */
    protected File tocTemplate;
    /**
     * The template used to generate helpset content file.
     *
     * Must be an existing file or a ressource in class-path
     *
     * @parameter expression="${jaxx.contentTemplate}" default-value="/defaultContent.html.vm"
     * @required
     * 
     * @since 1.3
     */
    protected File contentTemplate;
    /**
     * Flag to generate the search index.
     *
     * @parameter expression="${jaxx.generateSearch}" default-value="true"
     * @required
     *
     * @since 1.3
     */
    protected boolean generateSearch;
    /**
     * The help ids discovered in {@link #helpIdsStore} files.
     */
    private Properties helpIds;
    /**
     * Default locale (the first locale in {@link #localesToTreate}.
     */
    private Locale defaultLocale;
    /**
     * Locales to treate
     */
    private Locale[] localesToTreate;

    @Override
    public boolean init() throws Exception {

        File idsStore = getHelpIdsStore();

        if (!idsStore.exists()) {
            getLog().info("no helpIdStore to react at " + idsStore);
            return false;
        }

        if (locales == null || locales.trim().isEmpty()) {
            throw new MojoFailureException("You must set the 'locales' property properly (was " + locales + ").");
        }

        // check there is a outHelp
        if (outHelp == null) {
            throw new MojoFailureException("You must set the 'target' property.");
        }

        List<Locale> tmp = new ArrayList<Locale>();
        for (String loc : locales.split(",")) {
            Locale l = I18n.newLocale(loc);
            tmp.add(l);
        }

        if (tmp.isEmpty()) {
            throw new MojoFailureException("you must set the 'locales' property.");
        }

        localesToTreate = tmp.toArray(new Locale[tmp.size()]);
        defaultLocale = localesToTreate[0];

        // check ressources
        checkResource(helpsetTemplate);
        checkResource(mapTemplate);
        checkResource(indexTemplate);
        checkResource(tocTemplate);
        checkResource(contentTemplate);

        if (!getTargetDirectory().exists()) {
            getLog().info("mkdir " + getTargetDirectory());
            getTargetDirectory().mkdirs();
        }

        // load ids from idsStore file

        helpIds = new SortedProperties();

        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(idsStore), getEncoding()));
        String id = null;
        while ((id = reader.readLine()) != null) {
            id = id.trim();
            String path = id.replaceAll("\\.", File.separator) + ".html";
            helpIds.put(id, path);
        }

        if (helpIds.isEmpty()) {

            // no ids detected
            getLog().warn("No helpIds detected, will skip.");
            return false;
        }
        return true;
    }

    @Override
    public void doAction() throws Exception {

        int touchedFiles = 0;

        String mapFileName = helpsetName + "Map.jhm";
        String indexFileName = helpsetName + "Index.xml";
        String tocFileName = helpsetName + "TOC.xml";

        for (Locale locale : localesToTreate) {

            boolean isDefaultLocale = locale == defaultLocale;

            String language = locale.getLanguage();

            String localePath = (isDefaultLocale ? "default" : language);

            File localizedTarget = new File(outHelp, localePath);

            if (!localizedTarget.exists()) {
                localizedTarget.mkdirs();
            }
            getLog().info("Generate help for language " + language);
            if (isVerbose()) {
                getLog().info("  Localized target : " + localizedTarget);
            }

            Properties env = new Properties();

            env.put("helpSetName", helpsetName);
            env.put("locale", language);
            env.put("localePath", localePath);
            env.put("separator", "    ");
            env.put("autoremoveLine", AUTOREMOVE_LINE);

            String localeSuffix = isDefaultLocale ? "" : "_" + language;
            String helpsetFileName = helpsetName + localeSuffix + ".hs";

            env.put("helpSetFileName", helpsetFileName);

            env.put("mapFileName", mapFileName);
            env.put("indexFileName", indexFileName);
            env.put("tocFileName", tocFileName);
            env.put("generateSearch", generateSearch);

            // ---------------------------------------------------------------
            // --- main helpset file -----------------------------------------
            // ---------------------------------------------------------------

            File file = new File(outHelp, helpsetFileName);

            boolean doCreate = generateHelpsetFile(file, env);

            if (doCreate) {
                touchedFiles++;
            }

            // ---------------------------------------------------------------
            // --- helpset map file ------------------------------------------
            // ---------------------------------------------------------------

            file = new File(localizedTarget, mapFileName);

            generateMapFile(file, env);
            touchedFiles++;

            // ---------------------------------------------------------------
            // --- helpset index file ----------------------------------------
            // ---------------------------------------------------------------

            file = new File(localizedTarget, indexFileName);

            generateIndexFile(file, env);
            touchedFiles++;

            // ---------------------------------------------------------------
            // --- helpset toc file ------------------------------------------
            // ---------------------------------------------------------------

            file = new File(localizedTarget, tocFileName);

            generateTocFile(file, env);
            touchedFiles++;

            // ---------------------------------------------------------------
            // --- helpset content files -------------------------------------
            // ---------------------------------------------------------------

            touchedFiles += generateContentFiles(localizedTarget, env, localePath);

            // ---------------------------------------------------------------
            // --- generate search index -------------------------------------
            // ---------------------------------------------------------------
            
            if (generateSearch) {

                generateSearchIndex(localizedTarget, locale);
            }


        }

        getLog().info(touchedFiles + " file(s) treated.");
    }

    @Override
    public File getTargetDirectory() {
        return outHelp;
    }

    @Override
    public void setTargetDirectory(File targetDirectory) {
        this.outHelp = targetDirectory;
    }

    protected int generateContentFiles(File localizedTarget, Properties env, String localePath) throws Exception {

        int touchedFiles = 0;
        VelocityTemplateGenerator gen = prepareGenerator(contentTemplate);
        Enumeration<?> keys = helpIds.keys();
        while (keys.hasMoreElements()) {
            String key = (String) keys.nextElement();
            String url = (String) helpIds.get(key);
            url = helpsetName + File.separator + url;
            File f = new File(localizedTarget, url);
            boolean exist = f.exists();
            if (exist) {
                // check if there is a autoremoveLine in content
                String content = FileUtil.readAsString(f);
                if (!content.contains(AUTOREMOVE_LINE)) {
                    // no regenerate marker detected, so skip this file
                    if (verbose) {
                        getLog().debug("skip existing file " + f);
                    }
                    continue;
                }
            }
            f.getParentFile().mkdirs();
            if (verbose) {
                if (exist) {
                    getLog().info("regenerate content file " + f);
                } else {
                    getLog().info("generate   content file " + f);
                }
            }
            env.put("helpId", key);
            env.put("helpIdUrl", localePath + "/" + url);
            gen.generate(env, f);
            touchedFiles++;
        }
        return touchedFiles;
    }

    protected boolean generateHelpsetFile(File file, Properties env) throws Exception {

        if (file.exists()) {
            // check the autoremove line presence
            String content = FileUtil.readAsString(file);
            if (!content.contains(AUTOREMOVE_LINE)) {
                // no regenerate marker detected, so skip this file
                if (verbose) {
                    getLog().info("skip existing helpset main file " + file);
                }
                return false;
            }
        }

        if (verbose) {
            if (file.exists()) {
                getLog().info("regenerate helpset main file " + file);
            } else {
                getLog().info("generate helpset main file " + file);
            }
        }
        doGen(helpsetTemplate, file, env);
        return true;
    }

    protected Properties generateMapFile(File file, Properties env) throws Exception {

        boolean create;

        Properties mergedHelpIds = null;
        if (file.exists()) {

            // get back the exisiting data and merge it with incoming ones

            if (verbose) {
                getLog().info("loading existing helpset map file " + file);
            }

            mergedHelpIds = XmlHelper.getExistingHelpIds(file, verbose, getLog());
            create = false;

        } else {

            mergedHelpIds = new SortedProperties();
            create = true;
        }

        // inject new helpIds

        for (Object k : helpIds.keySet()) {
            mergedHelpIds.put(k, helpsetName + "/" + helpIds.get(k));
        }

        if (!mergedHelpIds.contains("top")) {

            // on ajoute une entree vers le root du helpset

            String topUrl = helpsetName + ".html";
            helpIds.put("top", topUrl);
            mergedHelpIds.put("top", helpsetName + "/" + topUrl);
            if (verbose) {
                getLog().debug("add top entry with url " + topUrl);
            }
        }

        if (verbose) {
            if (create) {
                getLog().info("generate helpset map file " + file);
            } else {
                getLog().info("udpate helpset map file " + file);
            }
        }

        env.put("helpIds", mergedHelpIds);
        doGen(mapTemplate, file, env);
        env.remove("helpIds");
        return mergedHelpIds;
    }

    protected NodeItem generateIndexFile(File file, Properties env) throws Exception {
        NodeItem rootItem = null;

        boolean create;

        if (file.exists()) {

            create = false;

            rootItem = XmlHelper.getExistingItems("indexitem", file);
        } else {
            create = true;
        }

        if (rootItem == null) {
            rootItem = new NodeItem("top", helpsetName);
        }

        // inject new index entries

        for (Object k : helpIds.keySet()) {
            NodeItem toc = rootItem.findChild(k + "");
            if (verbose) {
                getLog().debug("index " + k + " : " + toc);
            }
        }

        if (verbose) {
            if (create) {
                getLog().info("generate helpset index file " + file);
            } else {
                getLog().info("udpate helpset index file " + file);
            }
        }

        env.put("rootItem", rootItem);
        doGen(indexTemplate, file, env);
        env.remove("rootItem");
        return rootItem;
    }

    protected void generateSearchIndex(File localizedTarget, Locale locale) throws IllegalArgumentException, IOException, InvocationTargetException, SecurityException, IllegalAccessException, NoSuchMethodException {

        Method m = Indexer.class.getDeclaredMethod("main", String[].class);
        File indexDir = new File(localizedTarget, "JavaHelpSearch");
        // remove old index
        FileUtils.deleteDirectory(indexDir);
        //copy resources to a tmp dir (without any VCS infos)
        File tmpDir = new File(project.getBasedir(), "target" + File.separator + "jaxx-tmp" + File.separator + "indexer-" + locale + "-" + System.nanoTime());
        getLog().info("copy files to " + tmpDir + " for indexing them");
        FileUtils.copyDirectory(localizedTarget, tmpDir, "**/*", StringUtils.join(DirectoryScanner.DEFAULTEXCLUDES, ","));
        List<String> params = new ArrayList<String>();
        params.add("-verbose");
        params.add("-db");
        params.add(indexDir.getAbsolutePath());
        params.add("-logfile");
        File logFile = new File(project.getBasedir(), "target" + File.separator + "generated-sources" + File.separator + "jaxx" + File.separator + "indexer-" + locale + ".log");
        params.add(logFile.getAbsolutePath());
        params.add(tmpDir.getAbsolutePath());
        PrintStream out = System.out;
        PrintStream err = System.err;
        try {
            m.invoke(null, (Object) params.toArray(new String[params.size()]));
        } finally {
            System.setOut(out);
            System.setErr(err);
        }
        getLog().info("Search Index generated");
    }

    protected NodeItem generateTocFile(File file, Properties env) throws Exception {
        NodeItem rootItem = null;

        boolean create;

        if (file.exists()) {

            create = false;

            rootItem = XmlHelper.getExistingItems("tocitem", file);
        } else {
            create = true;
        }

        if (rootItem == null) {
            rootItem = new NodeItem("top", helpsetName);
        }
        // inject new toc entries

        for (Object k : helpIds.keySet()) {
            NodeItem toc = rootItem.findChild(k + "");
            if (verbose) {
                getLog().debug("toc " + k + " : " + toc);
            }
        }

        if (verbose) {
            if (create) {
                getLog().info("generate helpset toc file " + file);
            } else {
                getLog().info("udpate helpset toc file " + file);
            }
        }

        env.put("rootItem", rootItem);
        doGen(tocTemplate, file, env);
        env.remove("rootItem");
        return rootItem;
    }

    protected void doGen(File template, File f, Properties env) throws Exception {
        VelocityTemplateGenerator gen = prepareGenerator(template);
        gen.generate(env, f);
    }

    protected VelocityTemplateGenerator prepareGenerator(File template) throws Exception {
        URL templateURL = getTemplate(template);

        if (verbose) {
            getLog().info("using template " + templateURL);
        }
        VelocityTemplateGenerator gen = new VelocityTemplateGenerator(project, templateURL);
        return gen;
    }
}
