/*
 * #%L
 * JAXX :: Widgets
 * 
 * $Id: TimeEditorHandler.java 2532 2012-12-12 16:33:34Z kmorin $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.21/jaxx-widgets/src/main/java/jaxx/runtime/swing/editor/TimeEditorHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.runtime.swing.editor;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JLabel;
import javax.swing.JSlider;
import javax.swing.plaf.basic.BasicSliderUI;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Map;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;

/** @author tchemit <chemit@codelutin.com> */
public class TimeEditorHandler {

    public static final Log log = LogFactory.getLog(TimeEditorHandler.class);

    public static final String BEAN_PROPERTY = "bean";

    public static final String PROPERTY_PROPERTY = "property";

    public static final String DATE_PROPERTY = "date";

    public static final String TIME_MODEL_PROPERTY = "timeModel";

    /** editor ui */
    protected TimeEditor editor;

    /** the mutator method on the property of boxed bean in the editor */
    protected Method mutator;

    protected Calendar calendar;

    protected Calendar calendarMinute;

    public TimeEditorHandler(TimeEditor ui) {
        editor = ui;
        calendar = Calendar.getInstance();
        calendarMinute = Calendar.getInstance();
    }

    public void init() {

        if (editor.getBean() == null) {
            throw new NullPointerException("can not have a null bean in ui " + editor);
        }

        // create slider labels
        Map<Integer, JLabel> labelTable = new Hashtable<Integer, JLabel>();
        for (int i = 0; i < 25; i += 2) {
            labelTable.put(i * 60, new JLabel(i + ""));
        }
        JSlider slider = editor.getSlider();
        slider.setLabelTable((Dictionary<?, ?>) labelTable);

        MouseAdapter m = new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                // set the value
                int value = getSliderValue(e);
                JSlider slider = (JSlider) e.getComponent();
                slider.setValueIsAdjusting(true);
                slider.setValue(value);
                slider.setValueIsAdjusting(false);
                showToolTip(e);
                e.consume();
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseEntered(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseMoved(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseWheelMoved(MouseWheelEvent e) {
                JSlider slider = (JSlider) e.getComponent();

                // compute new value
                int nb = e.getWheelRotation();
                int value = slider.getValue() - nb;

                // set the value
                slider.setValueIsAdjusting(true);
                slider.setValue(value);
                slider.setValueIsAdjusting(false);
                e.consume();
            }

            int getSliderValue(MouseEvent e) {
                JSlider slider = (JSlider) e.getSource();
                int value = -1;
                if (slider.getUI() instanceof BasicSliderUI) {
                    BasicSliderUI ui = (BasicSliderUI) slider.getUI();
                    value = slider.getOrientation() == JSlider.HORIZONTAL
                            ? ui.valueForXPosition(e.getX())
                            : ui.valueForYPosition(e.getY());
                }
                return value;
            }

            void showToolTip(MouseEvent e) {

                int value = getSliderValue(e);
                if (value == -1) {
                    return;
                }
                int h = value / 60;
                int m = value % 60;

                String text = "";
                if (h < 10) {
                    text = "0";
                }
                text += h + " : ";
                if (m < 10) {
                    text += "0";
                }
                text += m;

                JSlider slider = (JSlider) e.getSource();
                slider.setToolTipText(text);

            }
        };
        slider.addMouseListener(m);
        slider.addMouseMotionListener(m);
        slider.addMouseWheelListener(m);

        // listen when date changes (should come from outside)
        editor.addPropertyChangeListener(DATE_PROPERTY, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Date date = (Date) evt.getNewValue();

                if (date == null) {
                    return;
                }
                calendar.setTime(date);
                int hours = calendar.get(Calendar.HOUR_OF_DAY);
                int minutes = calendar.get(Calendar.MINUTE);
                if (log.isDebugEnabled()) {
                    log.debug("date changed : new value " + hours + ":" + minutes);
                }
                getEditor().setTimeModel(hours * 60 + minutes);
            }
        });

        // listen when time model changes (should come from editor)
        editor.addPropertyChangeListener(TIME_MODEL_PROPERTY, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Integer time = (Integer) evt.getNewValue();
                int hours = time / 60;
                int minutes = time % 60;
                calendar.set(Calendar.HOUR_OF_DAY, hours);
                calendar.set(Calendar.MINUTE, minutes);
                setDate(null, calendar.getTime());
            }
        });
    }

    public TimeEditor getEditor() {
        return editor;
    }

    protected Date setMinuteModel(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
        }
        calendarMinute.setTime(incomingDate);
        calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
        incomingDate = calendarMinute.getTime();
        return incomingDate;
    }

    public int getMinute() {
        return getEditor().getTimeModel() % 60;
    }

    public int getHour() {
        return getEditor().getTimeModel() / 60;
    }

    public void updateTimeModelFromMinuteModel(Date minuteDate) {

        calendarMinute.setTime(minuteDate);
        int newHour = calendarMinute.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarMinute.get(Calendar.MINUTE);

        int oldHour = getHour();
        int oldMinute = getMinute();

        if (oldHour == newHour && oldMinute == newMinute) {

            // do nothing, same data
            if (log.isDebugEnabled()) {
                log.debug("Do not update time model , stay on same time = " + oldHour + ":" + oldMinute);
            }
            return;
        }

        // by default stay on same hour
        int hour = oldHour;

        // by default, use the new minute data
        int minute = newMinute;

        if (log.isDebugEnabled()) {
            log.debug("hh:mm (old from dateModel)   = " + oldHour + ":" + oldMinute);
            log.debug("hh:mm (new from minuteModel) = " + newHour + ":" + newMinute);
        }

        if (newMinute == 0) {

            // minute pass to zero (check if a new hour is required)
            if (newHour == 1) {

                if (oldHour == 23) {

                    // can't pass from 23:59 to 0:00, stay on 23:59
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    getEditor().getMinuteModel().setValue(getEditor().getMinuteModel().getPreviousValue());
                    return;
                }
                hour = (oldHour + 1) % 24;
            }
        } else if (newMinute == 59) {

            // minute pass to 59 (check if a new hour is required)

            if (newHour == 23) {

                if (oldHour == 0) {

                    // can't pass from 0:00 to 23:59, stay on 0:00
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    getEditor().getMinuteModel().setValue(getEditor().getMinuteModel().getNextValue());
                    return;
                }

                // decrease hour
                hour = (oldHour - 1) % 24;
            }
        }

        // date has changed
        if (log.isDebugEnabled()) {
            log.debug("Update time model to hh:mm = " + hour + ":" + minute);
        }
        getEditor().setTimeModel(hour * 60 + minute);
    }

    protected void setDate(Date oldValue, Date newValue) {
        if (editor.getBean() == null) {
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug(editor.getProperty() + " on " + editor.getBean().getClass() + " :: " + oldValue + " to " + newValue);
        }

        try {
            getMutator().invoke(editor.getBean(), newValue);

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected Method getMutator() {
        if (mutator == null) {
            mutator = BeanUIUtil.getMutator(editor.getBean(), editor.getProperty());
        }
        return mutator;
    }
}
