/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: AbstractI18nMojo.java 1800 2010-11-11 13:02:21Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.0/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/AbstractI18nMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin;

import org.apache.maven.project.MavenProject;
import org.nuiton.i18n.I18nUtil;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginHelper;
import org.nuiton.plugin.PluginWithEncoding;

import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Lower level i18n mojo abstraction.
 * <p/>
 * We defines here all commons parameters and shared behaviour.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author jruchaud <ruchaud@codelutin.com>
 */
public abstract class AbstractI18nMojo extends AbstractPlugin implements PluginWithEncoding {

    /** Le nombre de getters détectés pendant le cycle de vie du build. */
    private static int NB_GETTER_FILES;

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.2
     */
    protected MavenProject project;

    /**
     * Name to use as prefix of generated files.
     * <p/>
     * <b>Note :</b> By default, use the artifact id.
     *
     * @parameter expression="${i18n.artifactId}" default-value="${project.artifactId}"
     * @readonly
     */
    protected String artifactId;

    /**
     * Locales to treate, separated by comma.
     * <p/>
     * Example :
     * <pre>fr_FR,en_GB</pre>
     *
     * @parameter expression="${i18n.bundles}" default-value="fr_FR,en_GB"
     * @required
     */
    protected String bundles;

    /**
     * Directory where to find project i18n files.
     *
     * @parameter expression="${i18n.src}" default-value="${basedir}/src/main/resources/i18n"
     * @required
     */
    protected File src;

    /**
     * Directory where to generate i18n files.
     *
     * @parameter expression="${i18n.out}" default-value="${basedir}/target/generated-sources/i18n"
     * @required
     */
    protected File out;

    /**
     * Encoding used to load and store files.
     * <p/>
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     *
     * @parameter expression="${i18n.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     */
    protected String encoding;

    /**
     * To update generated files to user i18n files.
     * <p/>
     * <b>Note :</b> By default, this is active, in order to have a project uptodate
     * with last i18n bundles detected.
     *
     * @parameter expression="${i18n.genSrc}" default-value="true"
     */
    protected boolean genSrc;

    /**
     * Verbose flag.
     * <p/>
     * <b>Note :</b> if not setted, we used the {@code maven.verbose} property.
     *
     * @parameter expression="${i18n.verbose}" default-value="${maven.verbose}"
     */
    protected boolean verbose;

    /**
     * Silent flag to see only errors in console.
     *
     * @parameter expression="${i18n.silent}" default-value="false"
     * @since 1.0.0-rc-5
     */
    protected boolean silent;

    /**
     * Strict mode to only keep in user i18n detected i18n keys and remove obsolete keys.
     * <p/>
     * <b>Note :</b> By default not active. Use this with care since it can
     * delete keys. Moreover if this flag is activated, then all files will be parsed.
     *
     * @parameter expression="${i18n.strictMode}" default-value="false"
     */
    protected boolean strictMode;

    /** locales to process */
    protected Locale[] locales;


    @Override
    protected boolean checkPackaging() {

        // nothing to do on a pom module
        return !acceptPackaging(Packaging.pom);
    }

    @Override
    public void init() throws Exception {

        if (verbose) {
            // in verbose mode, no silent
            silent = false;
            getLog().info("config - verbose mode is on");
        }
        locales = I18nUtil.parseLocales(bundles);
        if (locales == null || locales.length == 0) {
            throw new IllegalStateException(
                    "Il faut au moins une locale declaree (utiliser " +
                    "la propriete 'bundles')");
        }
    }

    public String getArtifactId() {
        return artifactId;
    }

    /**
     * @return {@code true} si des getters ont etes enregistres pendant le
     *         cycle de vie, {@code false} sinon.
     */
    protected boolean needGeneration() {
        boolean needGeneration = NB_GETTER_FILES > 0;
        return needGeneration;
    }

    /**
     * Prend en compte qu'un getter a été détecté.
     * <p/>
     * Cela veut dire qu'un goal de parser a détecté des clefs. Il faudra donc
     * activer les goal get et gen.
     */
    protected void addGetter() {
        NB_GETTER_FILES++;
    }

    /**
     * @param root       le repertoire ou sont stockes les fichiers i18n
     * @param artifactId le nom de l'artifact
     * @param locale     le nom du bundle
     * @param create     {@code true} pour creer le fichier si non present
     * @return le fichier i18n
     * @throws IOException si probleme lors de la creation du fichier
     */
    public File getI18nFile(File root, String artifactId,
                            Locale locale, boolean create) throws IOException {
        File file = new File(
                root.getAbsolutePath() + File.separatorChar + artifactId +
                "_" + locale.toString() + ".properties");
        if (create && !file.exists()) {
            createNewFile(file);
        }
        return file;
    }

    /**
     * @param root   le repertoire ou sont stockes les fichiers getter
     * @param getter le nom du getter
     * @param create {@code true} pour creer le fichier si non present
     * @return le fichier i18n
     * @throws IOException si probleme lors de la creation du fichier
     */
    public File getGetterFile(File root, String getter, boolean create)
            throws IOException {
        File file = new File(
                root.getAbsolutePath() + File.separatorChar + getter);
        if (create && !file.exists()) {
            createNewFile(file);
        }
        return file;
    }

    protected void checkBundle(Locale locale,
                               Properties propertiesOut,
                               boolean showEmpty,
                               Map<Locale, SortedSet<String>> unsafeHolder) {

        // on verifie qu'il n'y a pas de traduction vide
        SortedSet<String> emptyEntries = PluginHelper.getEmptyKeys(propertiesOut);
        if (!emptyEntries.isEmpty()) {
            if (unsafeHolder != null) {

                // push empties i18n keys in the holder
                SortedSet<String> empties = unsafeHolder.get(locale);
                if (empties == null) {
                    empties = new TreeSet<String>();
                    unsafeHolder.put(locale, empties);
                }
                empties.addAll(emptyEntries);
            }
            StringBuilder buffer = new StringBuilder();
            int size = emptyEntries.size();
            buffer.append("bundle ");
            buffer.append(locale);
            buffer.append(" contains ");
            buffer.append(size);
            buffer.append("/");
            buffer.append(propertiesOut.size());
            buffer.append(" empty entries!");
            if (showEmpty) {
                int index = 0;
                for (String key : emptyEntries) {
                    buffer.append("\n  - ");
                    buffer.append(index++);
                    buffer.append("/");
                    buffer.append(size);
                    buffer.append(" : ");
                    buffer.append(key);
                }
            } else {
                buffer.append(" (use -Di18n.showEmpty to see these" +
                              " entries)");
            }
            getLog().warn(buffer.toString());
        } else {
            if (!silent && verbose) {
                getLog().info("bundle " + locale + " is valid (no empty" +
                              " entries).");
            }
        }
    }


    @Override
    public File getBackupFile(File file) {
        return new File(file.getAbsolutePath() + "~");
    }

    @Override
    protected void backupFile(File f) throws IOException {
        File dst = getBackupFile(f);
        copyFile(f, dst);
    }

    @Override
    public String getEncoding() {
        return encoding;
    }

    @Override
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public boolean isSilent() {
        return silent;
    }

    public boolean isStrictMode() {
        return strictMode;
    }
}
