/*
 * *##% 
 * I18n :: Maven Plugin
 * Copyright (C) 2007 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.i18n.plugin.bundle;

import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.bundle.I18nBundleUtil;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.io.SortedProperties;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.*;
import java.util.Map.Entry;

/**
 * Créer un bundle pour une application finale.
 * <p/>
 * Cela génère un merge de tous les fichiers i18n utilisés en un seul.
 * <p/>
 * On utilise la dépendance sur les artifacts pour connaitre l'ordre le
 * chargement des bundles.
 * <p/>
 * Si dans un bundle childs, la valeur de la clef est vide, on conserve alors
 * celui du parent,
 * <p/>
 * Ainsi on obtient un bundle dont toutes les clefs sont traduites.
 * <p/>
 * Le but aussi d'utiliser un unique bundle est de gagner du temps au runtime
 * car la recherche des bundles devient trop couteuse en temps lorsque l'on a de
 * nombreuses dépendances (au dessus de 100 deps cela peut prendre plusieurs
 * secondes, ce qui 'est pas acceptable).
 * <p/>
 * On a ajoute un second mode d'initialisation dans la clesse I18n pour
 * n'utiliser qu'un seul bundle et courcircuiter le chargement couteux...
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal bundle
 * @phase generate-resources
 * @execute goal=collect-i18n-artifacts
 * @requiresProject true
 * @requiresDependencyResolution runtime
 * @since 0.12
 */
public class BundleMojo extends AbstractI18nBundleMojo {

    /**
     * Repertoire ou generer les bundles.
     *
     * @parameter expression="${i18n.bundleOutputDir}" default-value="${basedir}/target/generated-sources/resources/META-INF"
     * @required
     * @since 1.0.0
     */
    protected File bundleOutputDir;

    /**
     * Nom du bundle a generer.
     *
     * @parameter expression="${i18n.bundleOutputName}" default-value="${project.artifactId}-i18n"
     * @required
     * @since 1.0.2
     */
    protected String bundleOutputName;

    /**
     * Un drapeau pour vérifier que les bundles ne contiennent pas d'entrées
     * vides.
     *
     * @parameter expression="${i18n.checkBundle}" default-value="true"
     * @required
     * @since 1.0.0
     */
    protected boolean checkBundle;

    /**
     * Un drapeau pour afficher les entrées vides. (nécessite {@link
     * #checkBundle} activé).
     *
     * @parameter expression="${i18n.showEmpty}" default-value="false"
     * @required
     * @since 1.0.0
     */
    protected boolean showEmpty;

    @Override
    public void init() throws Exception {
        super.init();

        createDirectoryIfNecessary(bundleOutputDir);

        // ajout de repertoire de generation (le parent en fait)
        // dans les resources du projet

        addResourceDir(bundleOutputDir.getParentFile(), "**/*.properties");
    }

    @Override
    protected void doAction() throws Exception {
        long t00 = System.nanoTime();

        String version = getProject().getVersion();
        version = PluginHelper.removeSnapshotSuffix(version);

        if (!silent) {
            getLog().info("config - bundle name : " + bundleOutputName);
            getLog().info("config - basedir     : " + bundleOutputDir);
            getLog().info("config - locales     : " + Arrays.toString(locales));
            getLog().info("config - version     : " + version);
        }

        Map<Locale, String> bundleDico =
                new LinkedHashMap<Locale, String>(locales.length);

        for (Locale locale : locales) {

            long t0 = System.nanoTime();

            File bundleOut = getI18nFile(bundleOutputDir, bundleOutputName,
                                         locale, false);

            if (!silent) {
                getLog().info("generate bundle for locale " + locale);
            }

            SortedProperties propertiesOut =
                    new SortedProperties(encoding, false);
            StringBuilder buffer = new StringBuilder();

            URL[] urls = getCollectI18nResources(locale);
            if (urls.length == 0) {
                getLog().warn("no bundle for locale " + locale);
                continue;
            }

            for (URL url : urls) {
                long t000 = System.nanoTime();
                I18nBundleEntry bundleEntry =
                        new I18nBundleEntry(url, locale, null);
                bundleEntry.load(propertiesOut);
                String strPath = bundleEntry.getPath().toString();
                int index = strPath.indexOf("i18n/");

                buffer.append(',').append(strPath.substring(index));
                if (verbose) {
                    getLog().info(
                            "loaded " + bundleEntry.getPath() + " in " +
                            PluginHelper.convertTime(t000, System.nanoTime()));
                }
            }

            if (buffer.length() > 0) {
                bundleDico.put(locale, buffer.substring(1));
                if (!silent) {
                    getLog().info(
                            "bundles for locale : " + bundleDico.get(locale));
                }
            }
            propertiesOut.store(bundleOut);
            if (!silent && verbose) {
                getLog().info(
                        "bundle created in " +
                        PluginHelper.convertTime(t0, System.nanoTime()) +
                        " (detected sentences : " + propertiesOut.size() + ")");
            }
            if (checkBundle) {
                checkBundle(locale, propertiesOut, showEmpty);
            }
        }

        // ecriture du ficher des definitions i18n (permet de faire une
        // recherche extact sur un fichier puis d'en deduire les bundles a
        // charger
        String f = String.format(DefaultI18nInitializer.UNIQUE_BUNDLE_DEF,
                                 bundleOutputName);
        File defOut = new File(bundleOutputDir, f);
        if (!silent) {
            getLog().info("prepare i18n definition " +
                          defOut.getAbsolutePath());
        }
        SortedProperties p = new SortedProperties(encoding, false);
        p.setProperty(DefaultI18nInitializer.BUNDLE_DEF_LOCALES, bundles);
        p.setProperty(DefaultI18nInitializer.BUNDLE_DEF_VERSION, version);
        for (Entry<Locale, String> e : bundleDico.entrySet()) {
            p.setProperty(DefaultI18nInitializer.BUNDLES_FOR_LOCALE +
                          e.getKey().toString(), e.getValue());
        }
        FileOutputStream out = new FileOutputStream(defOut);
        try {
            p.store(out, null);
        } finally {
            out.close();
        }

        if (!silent && verbose) {
            getLog().info("done in " +
                          PluginHelper.convertTime(t00, System.nanoTime()));
        }
    }

    @Override
    protected URL[] getCollectI18nResources(Locale locale) throws IOException {
        File file = getCollectOutputFile(locale, false);
        if (!file.exists()) {
            return I18nBundleUtil.EMPTY_URL_ARRAY;
        }
        URL[] urls = getLinesAsURL(file);
        return urls;
    }

    /**
     * TODO Move this in PluginHelper.
     *
     * @param src the source file to read
     * @return the url instanciated from lines of the source file.
     * @throws IOException if any pb while reading file
     */
    public static URL[] getLinesAsURL(File src) throws IOException {
        List<URL> result = new ArrayList<URL>();
        for (String line : PluginHelper.getLines(src)) {
            if (!line.isEmpty()) {
                result.add(new URL(line));
            }
        }
        return result.toArray(new URL[result.size()]);
    }
}
