/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2011 - 2012 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * http://glassfish.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.glassfish.websocket.platform;

import org.glassfish.websocket.api.ConversionException;
import org.glassfish.websocket.api.Peer;
import org.glassfish.websocket.api.EndpointContext;
import org.glassfish.websocket.api.Conversation;
import org.glassfish.grizzly.websockets.*;
import java.lang.reflect.*;
import java.io.*;
import java.util.*;
import java.util.logging.*;
import org.glassfish.websocket.api.*;
/**
 *
 * @author dannycoward
 */
public class WebSocketWrapper implements Peer {
    private WebSocket grizzlySocket;
    private WebSocketConversationImpl webSocketSession;
    private EndpointContextImpl webSocketContext;
    private String clientAddress;
    Date activationTime = new Date();
    private static Set<WebSocketWrapper> wsws = new HashSet<WebSocketWrapper>();

    final static Logger logger = Logger.getLogger("wsplatform");

    public static WebSocketWrapper getWebSocketWrapper(WebSocket socket, WebSocketEndpoint application) {
        String remoteImpl = application.getRemoteImplClassname();
        //logger.info("getting wrapper for socket and type " + remoteImpl + " for " + application);

        WebSocketWrapper wsw = null;
        if ( WebSocketWrapper.findWebSocketWrapper(socket) != null) {
            // there is one with the same socket
            // but is it of the right type ?
            wsw = WebSocketWrapper.findWebSocketWrapper(socket);
            if (wsw.getClass().getCanonicalName().equals(remoteImpl)) {
                //logger.info("Reusing old wsw");
            } else {
                throw new RuntimeException("Should not get here. You are asking for a remote impl :"+remoteImpl+": of a different type from one I already have :"+wsw.getClass().getCanonicalName()+": !");
            }

        } else { // create a new one
            if (remoteImpl.equals(WebSocketWrapper.class.getCanonicalName())) {
                wsw = new WebSocketWrapper();
            } else {
                try {
                    Class c = application.getClass().getClassLoader().loadClass(remoteImpl);
                    wsw = (WebSocketWrapper) c.newInstance();
                } catch (Exception e) {
                    throw new RuntimeException("Cannot instantiate a " + remoteImpl);
                }
            }
            wsw.grizzlySocket = socket;
            wsw.webSocketSession = new WebSocketConversationImpl(wsw);
            wsw.webSocketContext = application.getEndpointContext();
            wsw.webSocketContext.addWebSocketSession(wsw.webSocketSession);

            wsws.add(wsw);
        }
        //logger.info("got the wsw - done..." + wsw.getClass());
        return wsw;

    }

    private static Set<WebSocketWrapper> getWebSocketWrappers() {
        weedExpiredWebSocketWrappers();
        return wsws;
    }

    private static void weedExpiredWebSocketWrappers() {
        Set<WebSocketWrapper> expired = new HashSet<WebSocketWrapper>();
        for (WebSocketWrapper wsw : wsws) {
            if (!wsw.isConnected()) {
                expired.add(wsw);
            }
        }
        for (WebSocketWrapper toRemove : expired) {
            wsws.remove(toRemove);
        }
    }

    static WebSocketWrapper findWebSocketWrapper(WebSocket socket) {
        for (WebSocketWrapper nextWsw : getWebSocketWrappers()) {
            if (nextWsw.grizzlySocket == socket) {
                return nextWsw;
            }
        }
        return null;
    }

    // may be called by subclasses...
    protected WebSocketWrapper() {
        this.activationTime = new Date();
    }

    public String getAddress() {
        return this.clientAddress;
    }

    void setAddress(String clientAddress) {
        this.clientAddress = clientAddress;
    }


    public EndpointContext getContext() {
        return this.webSocketContext;
    }

    public Conversation getConversation() {
        return this.webSocketSession;
    }

    public boolean isConnected() {
        return this.grizzlySocket.isConnected();
    }

    private boolean isPrimitiveData(Object data) {
        Class dataClass = data.getClass();
        boolean isPrimitive = (dataClass.equals(Integer.class) ||
                dataClass.equals(Byte.class) ||
                dataClass.equals(Short.class) ||
                dataClass.equals(Long.class) ||
                dataClass.equals(Float.class) ||
                dataClass.equals(Double.class) ||
                dataClass.equals(Boolean.class) ||
                dataClass.equals(Character.class));
        return isPrimitive;
    }

    public void sendPrimitiveMessage(Object data) throws IOException, ConversionException {

        if (isPrimitiveData(data)) {
            this.sendMessage(data.toString());
        } else {
            throw new ConversionException("object " + data + " is not a primitive type.");
        }
    }

    public void sendPolymorphic(Object o, String[] encoderClassnames) throws IOException, ConversionException {
        // deal with finding all the custom remotes and their encoders.
        if (o instanceof String) {
            this.sendMessage((String) o);
            return;
        }
        List encoders = Arrays.asList(encoderClassnames);
        //System.out.println("Try to send object " + o + " which is of type " + o.getClass() + " with encoders " + encoders);
        for (String encoder : encoderClassnames) {
            try {
                Class c = webSocketContext.getBeanClassloader().loadClass(encoder);
                //System.out.println("Loaded encoder class " + encoder);
                List interfaces = Arrays.asList(c.getInterfaces());
                if (interfaces.contains(org.glassfish.websocket.api.TextEncoder.class)) {
                    //System.out.println("It implements TextEncoder");
                    Method m = null;
                    try {
                        m = c.getMethod("encode", o.getClass());
                    } catch (Exception e) {

                    }
                    //System.out.println("found the encode method");
                    if (m != null) {

                        //System.out.println("!!got it : " + c);
                        TextEncoder te = (TextEncoder) c.newInstance();

                        String toSendString = te.encode(o);
                        this.sendMessage(toSendString);
                        return;
                    }
                }
                //encoderClasses.add(c);
            } catch (ConversionException e) {
                throw e;
            } catch (Exception e) {
                e.printStackTrace();
                throw new IOException(e.getMessage() + " :Could not send message object " + o + " which is of type " + o.getClass() + " with encoders " + encoderClassnames);

            }

        }
        if (isPrimitiveData(o)) {
            this.sendPrimitiveMessage(o);
            return;
        }
        throw new RuntimeException("Could not send message object " + o + " which is of type " + o.getClass() + " with encoders " + encoders);

    }

    WebSocket getSocket() {
        return this.grizzlySocket;
    }

    public void sendMessage(String data) throws IOException {
        this.grizzlySocket.send(data);
    }

    public void sendMessage(byte[] data) throws IOException {
        this.grizzlySocket.send(data);
    }


    public void doNothing() throws ConversionException {
        if (false) throw new ConversionException("");
    }

    public String toString() {
        return "WSW: " + getClass().getSimpleName();
    }

    public void doClose(int code, String reason) throws IOException {
        this.grizzlySocket.close(code, reason);
    }

}
