/*
 * #%L
 * Pollen :: Vote Counting
 * $Id: VoteCountingService.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/services/VoteCountingService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.services;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.votecounting.business.Choice;
import org.chorem.pollen.votecounting.business.CondorcetMethod;
import org.chorem.pollen.votecounting.business.Context;
import org.chorem.pollen.votecounting.business.Group;
import org.chorem.pollen.votecounting.business.Method;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.business.PercentageMethod;
import org.chorem.pollen.votecounting.business.StandardMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;
import org.chorem.pollen.votecounting.utils.Utils;

import java.util.List;

/**
 * Service qui permet le dépouillement d'un sondage.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class VoteCountingService {

    /** Logger. */
    private static final Log log = LogFactory.getLog(VoteCountingService.class);

    public VoteCountingResultDTO execute(PollDTO poll) {
        boolean byGroup = poll.getPollType() == PollType.GROUP;
        VoteCountingResultDTO result = execute(poll, byGroup);
        return result;
    }

    public VoteCountingResultDTO executeVoteCounting(PollDTO poll) {
        return execute(poll, false);
    }

    public VoteCountingResultDTO executeGroupCounting(PollDTO poll) {
        return execute(poll, true);
    }

    /**
     * Execution du dépouillement
     *
     * @param poll : sondage
     * @return resultat
     */
    private VoteCountingResultDTO execute(PollDTO poll, boolean isByGroup) {
        if (log.isInfoEnabled()) {
            log.info("Dépouillement (byGroup=" + isByGroup + ") du sondage "
                     + poll.getPollId());
        }


        // Création et remplissage du contexte
        Context context = createContext(poll, isByGroup);

        // Execution
        if (!context.executeCounting()) {
            //FIXME tchemit ? Why returning null, prefer use an exception
            return null;
        }
        List<ChoiceDTO> resChoices = context.executeStats();

        VoteCountingResultDTO result = new VoteCountingResultDTO();
        result.setNbVotes(Utils.calculateNbVotes(poll.getVotingGroups(),
                                                 isByGroup));
        result.setTypeVoteCounting(poll.getVoteCounting());
        result.setByGroup(isByGroup);
        result.setIdPoll(poll.getPollId());
        result.setChoices(resChoices);
        return result;
    }

    /**
     * Création du contexte en fonction du type de dépouillement
     *
     * @param poll
     * @param isByGroup
     * @return
     */
    private Context createContext(PollDTO poll, boolean isByGroup) {

        Method method;
        switch (poll.getVoteCounting()) {
            case NORMAL:
                method = new StandardMethod();
                break;
            case PERCENTAGE:
                method = new PercentageMethod();
                break;
            case CONDORCET:
                method = new CondorcetMethod();
                break;
            case NUMBER:
                method = new NumberMethod();
                break;
            default:
                method = new StandardMethod();
        }
        Context result = new Context(method, isByGroup);

        for (PollChoiceDTO choice : poll.getChoices()) {
            if (choice.isHidden()) {
                result.addHiddenChoice(choice);
            } else {
                result.addChoice(choice);
            }
        }
        for (VotingGroupDTO votingGroup : poll.getVotingGroups()) {
            result.addGroup(votingGroup.getIdGroup(), votingGroup.getWeight());
            for (VotingPersonDTO person : votingGroup.getVotingPersons()) {
                String currentIdGroup = votingGroup.getIdGroup();
                for (VoteToChoiceDTO vote : person.getChoices()) {

                    Choice choice = result.getChoice(vote.getIdChoice());
                    Group group = choice.getGroup(currentIdGroup);

                    group.addVote(vote.getValue(),
                                  person.getWeight(),
                                  person.getVotingId());
                }
            }
        }
        return result;
    }

}
