/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.EnumSet;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSTestWrapper;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileContextTestWrapper;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystemTestWrapper;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public abstract class SymlinkBaseTest {
    static final long seed = 3735928559L;
    static final int blockSize = 8192;
    static final int fileSize = 16384;
    static final int numBlocks = 2;
    protected static FSTestWrapper wrapper;

    protected abstract String getScheme();

    protected abstract String testBaseDir1() throws IOException;

    protected abstract String testBaseDir2() throws IOException;

    protected abstract URI testURI();

    protected boolean emulatingSymlinksOnWindows() {
        return false;
    }

    protected IOException unwrapException(IOException e) {
        return e;
    }

    protected static void createAndWriteFile(Path p) throws IOException {
        SymlinkBaseTest.createAndWriteFile(wrapper, p);
    }

    protected static void createAndWriteFile(FSTestWrapper wrapper, Path p) throws IOException {
        wrapper.createFile(p, 2, new Options.CreateOpts[]{Options.CreateOpts.createParent(), Options.CreateOpts.repFac((short)1), Options.CreateOpts.blockSize((long)8192L)});
    }

    protected static void readFile(Path p) throws IOException {
        wrapper.readFile(p, 16384);
    }

    protected static void appendToFile(Path p) throws IOException {
        wrapper.appendToFile(p, 2, new Options.CreateOpts[]{Options.CreateOpts.blockSize((long)8192L)});
    }

    @Before
    public void setUp() throws Exception {
        wrapper.mkdir(new Path(this.testBaseDir1()), FileContext.DEFAULT_PERM, true);
        wrapper.mkdir(new Path(this.testBaseDir2()), FileContext.DEFAULT_PERM, true);
    }

    @After
    public void tearDown() throws Exception {
        wrapper.delete(new Path(this.testBaseDir1()), true);
        wrapper.delete(new Path(this.testBaseDir2()), true);
    }

    @Test(timeout=10000L)
    public void testStatRoot() throws IOException {
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(new Path("/")).isSymlink());
    }

    @Test(timeout=10000L)
    public void testSetWDNotResolvesLinks() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir1() + "/link");
        wrapper.createSymlink(dir, linkToDir, false);
        wrapper.setWorkingDirectory(linkToDir);
        Assert.assertEquals((Object)linkToDir.getName(), (Object)wrapper.getWorkingDirectory().getName());
    }

    @Test(timeout=10000L)
    public void testCreateDanglingLink() throws IOException {
        Path file = new Path("/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        wrapper.createSymlink(file, link, false);
        try {
            wrapper.getFileStatus(link);
            Assert.fail((String)"Got file status of non-existant file");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        wrapper.delete(link, false);
    }

    @Test(timeout=10000L)
    public void testCreateLinkToNullEmpty() throws IOException {
        Path link = new Path(this.testBaseDir1() + "/link");
        try {
            wrapper.createSymlink(null, link, false);
            Assert.fail((String)"Can't create symlink to null");
        }
        catch (NullPointerException e) {
            // empty catch block
        }
        try {
            wrapper.createSymlink(new Path(""), link, false);
            Assert.fail((String)"Can't create symlink to empty string");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkCanCreateParent() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir2() + "/linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.delete(new Path(this.testBaseDir2()), true);
        try {
            wrapper.createSymlink(file, link, false);
            Assert.fail((String)"Created link without first creating parent dir");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertFalse((boolean)wrapper.exists(new Path(this.testBaseDir2())));
        wrapper.createSymlink(file, link, true);
        SymlinkBaseTest.readFile(link);
    }

    @Test(timeout=10000L)
    public void testMkdirExistingLink() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/targetFile");
        SymlinkBaseTest.createAndWriteFile(file);
        Path dir = new Path(this.testBaseDir1() + "/link");
        wrapper.createSymlink(file, dir, false);
        try {
            wrapper.mkdir(dir, FileContext.DEFAULT_PERM, false);
            Assert.fail((String)"Created a dir where a symlink exists");
        }
        catch (FileAlreadyExistsException e) {
        }
        catch (IOException e) {
            Assert.assertEquals((Object)"file", (Object)this.getScheme());
        }
    }

    @Test(timeout=10000L)
    public void testCreateFileViaDanglingLinkParent() throws IOException {
        Path dir = new Path(this.testBaseDir1() + "/dangling");
        Path file = new Path(this.testBaseDir1() + "/dangling/file");
        wrapper.createSymlink(new Path("/doesNotExist"), dir, false);
        try {
            FSDataOutputStream out = wrapper.create(file, EnumSet.of(CreateFlag.CREATE), new Options.CreateOpts[]{Options.CreateOpts.repFac((short)1), Options.CreateOpts.blockSize((long)8192L)});
            out.close();
            Assert.fail((String)"Created a link with dangling link parent");
        }
        catch (FileNotFoundException e) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testDeleteLink() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir1() + "/linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        SymlinkBaseTest.readFile(link);
        wrapper.delete(link, false);
        try {
            SymlinkBaseTest.readFile(link);
            Assert.fail((String)"Symlink should have been deleted");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        wrapper.createSymlink(file, link, false);
    }

    @Test(timeout=10000L)
    public void testOpenResolvesLinks() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        wrapper.createSymlink(file, link, false);
        try {
            wrapper.open(link);
            Assert.fail((String)"link target does not exist");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        wrapper.delete(link, false);
    }

    @Test(timeout=10000L)
    public void testStatLinkToFile() throws IOException {
        Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        Path file = new Path(this.testBaseDir1() + "/file");
        Path linkToFile = new Path(this.testBaseDir1() + "/linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, linkToFile, false);
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(linkToFile).isDirectory());
        Assert.assertTrue((boolean)wrapper.isSymlink(linkToFile));
        Assert.assertTrue((boolean)wrapper.isFile(linkToFile));
        Assert.assertFalse((boolean)wrapper.isDir(linkToFile));
        Assert.assertEquals((Object)file, (Object)wrapper.getLinkTarget(linkToFile));
        if (!"file".equals(this.getScheme())) {
            Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileStatus(linkToFile));
            Assert.assertEquals((Object)wrapper.makeQualified(file), (Object)wrapper.getFileStatus(linkToFile).getPath());
            Assert.assertEquals((Object)wrapper.makeQualified(linkToFile), (Object)wrapper.getFileLinkStatus(linkToFile).getPath());
        }
    }

    @Test(timeout=10000L)
    public void testStatRelLinkToFile() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToFile = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(new Path("file"), linkToFile, false);
        Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileStatus(linkToFile));
        Assert.assertEquals((Object)wrapper.makeQualified(file), (Object)wrapper.getFileStatus(linkToFile).getPath());
        Assert.assertEquals((Object)wrapper.makeQualified(linkToFile), (Object)wrapper.getFileLinkStatus(linkToFile).getPath());
    }

    @Test(timeout=10000L)
    public void testStatLinkToDir() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir1() + "/linkToDir");
        wrapper.createSymlink(dir, linkToDir, false);
        Assert.assertFalse((boolean)wrapper.getFileStatus(linkToDir).isSymlink());
        Assert.assertTrue((boolean)wrapper.isDir(linkToDir));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(linkToDir).isDirectory());
        Assert.assertTrue((boolean)wrapper.getFileLinkStatus(linkToDir).isSymlink());
        Assert.assertFalse((boolean)wrapper.isFile(linkToDir));
        Assert.assertTrue((boolean)wrapper.isDir(linkToDir));
        Assert.assertEquals((Object)dir, (Object)wrapper.getLinkTarget(linkToDir));
    }

    @Test(timeout=10000L)
    public void testStatDanglingLink() throws IOException {
        Path file = new Path("/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        wrapper.createSymlink(file, link, false);
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(link).isDirectory());
        Assert.assertTrue((boolean)wrapper.getFileLinkStatus(link).isSymlink());
    }

    @Test(timeout=10000L)
    public void testStatNonExistentFiles() throws IOException {
        Path fileAbs = new Path("/doesNotExist");
        try {
            wrapper.getFileLinkStatus(fileAbs);
            Assert.fail((String)"Got FileStatus for non-existant file");
        }
        catch (FileNotFoundException f) {
            // empty catch block
        }
        try {
            wrapper.getLinkTarget(fileAbs);
            Assert.fail((String)"Got link target for non-existant file");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testStatNonLinks() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1() + "/file");
        SymlinkBaseTest.createAndWriteFile(file);
        try {
            wrapper.getLinkTarget(dir);
            Assert.fail((String)"Lstat'd a non-symlink");
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            wrapper.getLinkTarget(file);
            Assert.fail((String)"Lstat'd a non-symlink");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testRecursiveLinks() throws IOException {
        Path link1 = new Path(this.testBaseDir1() + "/link1");
        Path link2 = new Path(this.testBaseDir1() + "/link2");
        wrapper.createSymlink(link1, link2, false);
        wrapper.createSymlink(link2, link1, false);
        try {
            SymlinkBaseTest.readFile(link1);
            Assert.fail((String)"Read recursive link");
        }
        catch (FileNotFoundException f) {
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Possible cyclic loop while following symbolic link " + link1.toString()), (Object)x.getMessage());
        }
    }

    private void checkLink(Path linkAbs, Path expectedTarget, Path targetQual) throws IOException {
        block7: {
            Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
            Path dir = new Path(this.testBaseDir1());
            Assert.assertTrue((boolean)wrapper.isFile(linkAbs));
            Assert.assertFalse((boolean)wrapper.isDir(linkAbs));
            Assert.assertFalse((boolean)wrapper.getFileStatus(linkAbs).isSymlink());
            Assert.assertFalse((boolean)wrapper.getFileStatus(linkAbs).isDirectory());
            Assert.assertEquals((long)16384L, (long)wrapper.getFileStatus(linkAbs).getLen());
            Assert.assertTrue((boolean)wrapper.isSymlink(linkAbs));
            Assert.assertFalse((boolean)wrapper.getFileLinkStatus(linkAbs).isDirectory());
            Assert.assertEquals((Object)targetQual.toString(), (Object)wrapper.getFileLinkStatus(linkAbs).getSymlink().toString());
            Assert.assertEquals((Object)targetQual, (Object)wrapper.getFileLinkStatus(linkAbs).getSymlink());
            if (!"file".equals(this.getScheme())) {
                FileContext localFc = FileContext.getLocalFSFileContext();
                Path linkQual = new Path(this.testURI().toString(), linkAbs);
                Assert.assertEquals((Object)targetQual, (Object)localFc.getFileLinkStatus(linkQual).getSymlink());
            }
            Assert.assertEquals((Object)expectedTarget, (Object)wrapper.getLinkTarget(linkAbs));
            wrapper.setWorkingDirectory(dir);
            SymlinkBaseTest.readFile(new Path("linkToFile"));
            SymlinkBaseTest.readFile(linkAbs);
            SymlinkBaseTest.readFile(new Path(this.testURI().toString(), linkAbs));
            boolean failureExpected = true;
            if ("file".equals(this.getScheme())) {
                failureExpected = false;
            } else if (wrapper instanceof FileSystemTestWrapper) {
                failureExpected = false;
            }
            try {
                SymlinkBaseTest.readFile(new Path(this.getScheme() + ":///" + this.testBaseDir1() + "/linkToFile"));
                Assert.assertFalse((boolean)failureExpected);
            }
            catch (Exception e) {
                if (failureExpected) break block7;
                throw new IOException(e);
            }
        }
        if (wrapper instanceof FileContextTestWrapper && !"file".equals(this.getScheme())) {
            FSTestWrapper localWrapper = wrapper.getLocalFSWrapper();
            localWrapper.readFile(new Path(this.testURI().toString(), linkAbs), 16384);
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkUsingRelPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1(), "file");
        Path linkAbs = new Path(this.testBaseDir1(), "linkToFile");
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileQual = new Path(schemeAuth, this.testBaseDir1() + "/file");
        SymlinkBaseTest.createAndWriteFile(fileAbs);
        wrapper.setWorkingDirectory(new Path(this.testBaseDir1()));
        wrapper.createSymlink(new Path("file"), new Path("linkToFile"), false);
        this.checkLink(linkAbs, new Path("file"), fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        Path fileViaDir2 = new Path(schemeAuth, this.testBaseDir2() + "/file");
        wrapper.rename(dir1, dir2, Options.Rename.OVERWRITE);
        FileStatus[] stats = wrapper.listStatus(dir2);
        Assert.assertEquals((Object)fileViaDir2, (Object)wrapper.getFileLinkStatus(linkViaDir2).getSymlink());
        SymlinkBaseTest.readFile(linkViaDir2);
    }

    @Test(timeout=10000L)
    public void testCreateLinkUsingAbsPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1() + "/file");
        Path linkAbs = new Path(this.testBaseDir1() + "/linkToFile");
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileQual = new Path(schemeAuth, this.testBaseDir1() + "/file");
        SymlinkBaseTest.createAndWriteFile(fileAbs);
        wrapper.createSymlink(fileAbs, linkAbs, false);
        this.checkLink(linkAbs, fileAbs, fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        wrapper.rename(dir1, dir2, Options.Rename.OVERWRITE);
        Assert.assertEquals((Object)fileQual, (Object)wrapper.getFileLinkStatus(linkViaDir2).getSymlink());
        try {
            SymlinkBaseTest.readFile(linkViaDir2);
            Assert.fail((String)"The target should not exist");
        }
        catch (FileNotFoundException x) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkUsingFullyQualPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1(), "file");
        Path linkAbs = new Path(this.testBaseDir1(), "linkToFile");
        Path fileQual = new Path(this.testURI().toString(), fileAbs);
        Path linkQual = new Path(this.testURI().toString(), linkAbs);
        SymlinkBaseTest.createAndWriteFile(fileAbs);
        wrapper.createSymlink(fileQual, linkQual, false);
        this.checkLink(linkAbs, "file".equals(this.getScheme()) ? fileAbs : fileQual, fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        wrapper.rename(dir1, dir2, Options.Rename.OVERWRITE);
        Assert.assertEquals((Object)fileQual, (Object)wrapper.getFileLinkStatus(linkViaDir2).getSymlink());
        try {
            SymlinkBaseTest.readFile(linkViaDir2);
            Assert.fail((String)"The target should not exist");
        }
        catch (FileNotFoundException x) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkUsingPartQualPath1() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileWoHost = new Path(this.getScheme() + "://" + this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir1() + "/linkToFile");
        Path linkQual = new Path(schemeAuth, this.testBaseDir1() + "/linkToFile");
        FSTestWrapper localWrapper = wrapper.getLocalFSWrapper();
        wrapper.createSymlink(fileWoHost, link, false);
        Assert.assertEquals((Object)fileWoHost, (Object)wrapper.getLinkTarget(linkQual));
        Assert.assertEquals((Object)fileWoHost.toString(), (Object)wrapper.getFileLinkStatus(link).getSymlink().toString());
        Assert.assertEquals((Object)fileWoHost.toString(), (Object)wrapper.getFileLinkStatus(linkQual).getSymlink().toString());
        if (wrapper instanceof FileContextTestWrapper) {
            Assert.assertEquals((Object)fileWoHost.toString(), (Object)localWrapper.getFileLinkStatus(linkQual).getSymlink().toString());
        }
        try {
            SymlinkBaseTest.readFile(link);
            Assert.fail((String)"DFS requires URIs with schemes have an authority");
        }
        catch (RuntimeException e) {
            Assert.assertTrue((boolean)(wrapper instanceof FileContextTestWrapper));
        }
        catch (FileNotFoundException e) {
            Assert.assertTrue((boolean)(wrapper instanceof FileSystemTestWrapper));
            GenericTestUtils.assertExceptionContains("File does not exist: /test1/file", e);
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkUsingPartQualPath2() throws IOException {
        block4: {
            Path link = new Path(this.testBaseDir1(), "linkToFile");
            Path fileWoScheme = new Path("//" + this.testURI().getAuthority() + this.testBaseDir1() + "/file");
            if ("file".equals(this.getScheme())) {
                return;
            }
            wrapper.createSymlink(fileWoScheme, link, false);
            Assert.assertEquals((Object)fileWoScheme, (Object)wrapper.getLinkTarget(link));
            Assert.assertEquals((Object)fileWoScheme.toString(), (Object)wrapper.getFileLinkStatus(link).getSymlink().toString());
            try {
                SymlinkBaseTest.readFile(link);
                Assert.fail((String)"Accessed a file with w/o scheme");
            }
            catch (IOException e) {
                if (wrapper instanceof FileContextTestWrapper) {
                    Assert.assertEquals((Object)"No AbstractFileSystem for scheme: null", (Object)e.getMessage());
                }
                if (!(wrapper instanceof FileSystemTestWrapper)) break block4;
                Assert.assertEquals((Object)"No FileSystem for scheme: null", (Object)e.getMessage());
            }
        }
    }

    @Test(timeout=10000L)
    public void testLinkStatusAndTargetWithNonLink() throws IOException {
        Path schemeAuth = new Path(this.testURI().toString());
        Path dir = new Path(this.testBaseDir1());
        Path dirQual = new Path(schemeAuth, dir.toString());
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileQual = new Path(schemeAuth, file.toString());
        SymlinkBaseTest.createAndWriteFile(file);
        Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileLinkStatus(file));
        Assert.assertEquals((Object)wrapper.getFileStatus(dir), (Object)wrapper.getFileLinkStatus(dir));
        try {
            wrapper.getLinkTarget(file);
            Assert.fail((String)"Get link target on non-link should throw an IOException");
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Path " + fileQual + " is not a symbolic link"), (Object)x.getMessage());
        }
        try {
            wrapper.getLinkTarget(dir);
            Assert.fail((String)"Get link target on non-link should throw an IOException");
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Path " + dirQual + " is not a symbolic link"), (Object)x.getMessage());
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkToDirectory() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dir1, linkToDir, false);
        Assert.assertFalse((boolean)wrapper.isFile(linkToDir));
        Assert.assertTrue((boolean)wrapper.isDir(linkToDir));
        Assert.assertTrue((boolean)wrapper.getFileStatus(linkToDir).isDirectory());
        Assert.assertTrue((boolean)wrapper.getFileLinkStatus(linkToDir).isSymlink());
    }

    @Test(timeout=10000L)
    public void testCreateFileViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        wrapper.createSymlink(dir, linkToDir, false);
        SymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertTrue((boolean)wrapper.isFile(fileViaLink));
        Assert.assertFalse((boolean)wrapper.isDir(fileViaLink));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)wrapper.getFileStatus(fileViaLink).isDirectory());
        SymlinkBaseTest.readFile(fileViaLink);
        wrapper.delete(fileViaLink, true);
        Assert.assertFalse((boolean)wrapper.exists(fileViaLink));
    }

    @Test(timeout=10000L)
    public void testCreateDirViaSymlink() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path subDir = new Path(this.testBaseDir1(), "subDir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path subDirViaLink = new Path(linkToDir, "subDir");
        wrapper.createSymlink(dir1, linkToDir, false);
        wrapper.mkdir(subDirViaLink, FileContext.DEFAULT_PERM, true);
        Assert.assertTrue((boolean)wrapper.isDir(subDirViaLink));
        wrapper.delete(subDirViaLink, false);
        Assert.assertFalse((boolean)wrapper.exists(subDirViaLink));
        Assert.assertFalse((boolean)wrapper.exists(subDir));
    }

    @Test(timeout=10000L)
    public void testCreateLinkViaLink() throws IOException {
        Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path linkToFile = new Path(linkToDir, "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dir1, linkToDir, false);
        wrapper.createSymlink(fileViaLink, linkToFile, false);
        Assert.assertTrue((boolean)wrapper.isFile(linkToFile));
        Assert.assertTrue((boolean)wrapper.getFileLinkStatus(linkToFile).isSymlink());
        SymlinkBaseTest.readFile(linkToFile);
        Assert.assertEquals((long)16384L, (long)wrapper.getFileStatus(linkToFile).getLen());
        Assert.assertEquals((Object)fileViaLink, (Object)wrapper.getLinkTarget(linkToFile));
    }

    @Test(timeout=10000L)
    public void testListStatusUsingLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "link");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(new Path(this.testBaseDir1()), link, false);
        FileStatus[] stats = wrapper.listStatus(link);
        Assert.assertTrue((stats.length == 2 || stats.length == 3 ? 1 : 0) != 0);
        RemoteIterator<FileStatus> statsItor = wrapper.listStatusIterator(link);
        int dirLen = 0;
        while (statsItor.hasNext()) {
            statsItor.next();
            ++dirLen;
        }
        Assert.assertTrue((dirLen == 2 || dirLen == 3 ? 1 : 0) != 0);
    }

    @Test(timeout=10000L)
    public void testCreateLinkTwice() throws IOException {
        Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        try {
            wrapper.createSymlink(file, link, false);
            Assert.fail((String)"link already exists");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkToLink() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path linkToLink = new Path(this.testBaseDir2(), "linkToLink");
        Path fileViaLink = new Path(this.testBaseDir2(), "linkToLink/file");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dir1, linkToDir, false);
        wrapper.createSymlink(linkToDir, linkToLink, false);
        Assert.assertTrue((boolean)wrapper.isFile(fileViaLink));
        Assert.assertFalse((boolean)wrapper.isDir(fileViaLink));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)wrapper.getFileStatus(fileViaLink).isDirectory());
        SymlinkBaseTest.readFile(fileViaLink);
    }

    @Test(timeout=10000L)
    public void testCreateFileDirExistingLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        try {
            SymlinkBaseTest.createAndWriteFile(link);
            Assert.fail((String)"link already exists");
        }
        catch (IOException x) {
            // empty catch block
        }
        try {
            wrapper.mkdir(link, FsPermission.getDefault(), false);
            Assert.fail((String)"link already exists");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testUseLinkAferDeleteLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        wrapper.delete(link, false);
        try {
            SymlinkBaseTest.readFile(link);
            Assert.fail((String)"link was deleted");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        SymlinkBaseTest.readFile(file);
        wrapper.createSymlink(file, link, false);
        SymlinkBaseTest.readFile(link);
    }

    @Test(timeout=10000L)
    public void testCreateLinkToDot() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToDot");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.setWorkingDirectory(dir);
        try {
            wrapper.createSymlink(new Path("."), link, false);
            Assert.fail((String)"Created symlink to dot");
        }
        catch (IOException x) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testCreateLinkToDotDot() throws IOException {
        Path file = new Path(this.testBaseDir1(), "test/file");
        Path dotDot = new Path(this.testBaseDir1(), "test/..");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "test/file");
        Assert.assertEquals((Object)new Path(this.testBaseDir1()), (Object)dotDot);
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dotDot, linkToDir, false);
        SymlinkBaseTest.readFile(fileViaLink);
        Assert.assertEquals((long)16384L, (long)wrapper.getFileStatus(fileViaLink).getLen());
    }

    @Test(timeout=10000L)
    public void testCreateLinkToDotDotPrefix() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path dir = new Path(this.testBaseDir1(), "test");
        Path link = new Path(this.testBaseDir1(), "test/link");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.mkdir(dir, FsPermission.getDefault(), false);
        wrapper.setWorkingDirectory(dir);
        wrapper.createSymlink(new Path("../file"), link, false);
        SymlinkBaseTest.readFile(link);
        Assert.assertEquals((Object)new Path("../file"), (Object)wrapper.getLinkTarget(link));
    }

    @Test(timeout=10000L)
    public void testRenameFileViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path fileNewViaLink = new Path(linkToDir, "fileNew");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dir, linkToDir, false);
        wrapper.rename(fileViaLink, fileNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)wrapper.exists(fileViaLink));
        Assert.assertFalse((boolean)wrapper.exists(file));
        Assert.assertTrue((boolean)wrapper.exists(fileNewViaLink));
    }

    @Test(timeout=10000L)
    public void testRenameFileToDestViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path subDir = new Path(linkToDir, "subDir");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(dir, linkToDir, false);
        wrapper.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        try {
            wrapper.rename(file, subDir, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)wrapper.exists(file));
    }

    @Test(timeout=10000L)
    public void testRenameDirViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path dir = new Path(baseDir, "dir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path dirViaLink = new Path(linkToDir, "dir");
        Path dirNewViaLink = new Path(linkToDir, "dirNew");
        wrapper.mkdir(dir, FileContext.DEFAULT_PERM, false);
        wrapper.createSymlink(baseDir, linkToDir, false);
        Assert.assertTrue((boolean)wrapper.exists(dirViaLink));
        wrapper.rename(dirViaLink, dirNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)wrapper.exists(dirViaLink));
        Assert.assertFalse((boolean)wrapper.exists(dir));
        Assert.assertTrue((boolean)wrapper.exists(dirNewViaLink));
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "link");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path linkViaLink = new Path(linkToDir, "link");
        Path linkNewViaLink = new Path(linkToDir, "linkNew");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        wrapper.createSymlink(baseDir, linkToDir, false);
        wrapper.rename(linkViaLink, linkNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)wrapper.exists(linkViaLink));
        Assert.assertTrue((boolean)wrapper.exists(file));
        Assert.assertTrue((wrapper.getFileLinkStatus(linkNewViaLink).isSymlink() || this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        SymlinkBaseTest.readFile(linkNewViaLink);
    }

    @Test(timeout=10000L)
    public void testRenameDirToSymlinkToDir() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path subDir = new Path(this.testBaseDir2(), "subDir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        wrapper.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        wrapper.createSymlink(subDir, linkToDir, false);
        try {
            wrapper.rename(dir1, linkToDir, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)wrapper.exists(dir1));
        Assert.assertTrue((boolean)wrapper.exists(linkToDir));
    }

    @Test(timeout=10000L)
    public void testRenameDirToSymlinkToFile() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir2(), "file");
        Path linkToFile = new Path(this.testBaseDir2(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, linkToFile, false);
        try {
            wrapper.rename(dir1, linkToFile, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)wrapper.exists(dir1));
        Assert.assertTrue((boolean)wrapper.exists(linkToFile));
    }

    @Test(timeout=10000L)
    public void testRenameDirToDanglingSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path link = new Path(this.testBaseDir2(), "linkToFile");
        wrapper.createSymlink(new Path("/doesNotExist"), link, false);
        try {
            wrapper.rename(dir, link, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)wrapper.exists(dir));
        Assert.assertTrue((wrapper.getFileLinkStatus(link) != null ? 1 : 0) != 0);
    }

    @Test(timeout=10000L)
    public void testRenameFileToSymlinkToDir() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path subDir = new Path(this.testBaseDir1(), "subDir");
        Path link = new Path(this.testBaseDir1(), "link");
        wrapper.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        wrapper.createSymlink(subDir, link, false);
        SymlinkBaseTest.createAndWriteFile(file);
        try {
            wrapper.rename(file, link, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to symlink w/o overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        wrapper.rename(file, link, Options.Rename.OVERWRITE);
        Assert.assertFalse((boolean)wrapper.exists(file));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isFile(link));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(link).isSymlink());
    }

    @Test(timeout=10000L)
    public void testRenameFileToSymlinkToFile() throws IOException {
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path file2 = new Path(this.testBaseDir1(), "file2");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file1);
        SymlinkBaseTest.createAndWriteFile(file2);
        wrapper.createSymlink(file2, link, false);
        try {
            wrapper.rename(file1, link, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to symlink w/o overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        wrapper.rename(file1, link, Options.Rename.OVERWRITE);
        Assert.assertFalse((boolean)wrapper.exists(file1));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isFile(link));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(link).isSymlink());
    }

    @Test(timeout=10000L)
    public void testRenameFileToDanglingSymlink() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file1);
        wrapper.createSymlink(new Path("/doesNotExist"), link, false);
        try {
            wrapper.rename(file1, link, new Options.Rename[0]);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        wrapper.rename(file1, link, Options.Rename.OVERWRITE);
        Assert.assertFalse((boolean)wrapper.exists(file1));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isFile(link));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(link).isSymlink());
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkNonExistantDest() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link1 = new Path(this.testBaseDir1(), "linkToFile1");
        Path link2 = new Path(this.testBaseDir1(), "linkToFile2");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link1, false);
        wrapper.rename(link1, link2, new Options.Rename[0]);
        Assert.assertTrue((wrapper.getFileLinkStatus(link2).isSymlink() || this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        SymlinkBaseTest.readFile(link2);
        SymlinkBaseTest.readFile(file);
        Assert.assertFalse((boolean)wrapper.exists(link1));
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkToExistingFile() throws IOException {
        Path file1 = new Path(this.testBaseDir1(), "file");
        Path file2 = new Path(this.testBaseDir1(), "someFile");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file1);
        SymlinkBaseTest.createAndWriteFile(file2);
        wrapper.createSymlink(file2, link, false);
        try {
            wrapper.rename(link, file1, new Options.Rename[0]);
            Assert.fail((String)"Renamed w/o passing overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        wrapper.rename(link, file1, Options.Rename.OVERWRITE);
        Assert.assertFalse((boolean)wrapper.exists(link));
        if (!this.emulatingSymlinksOnWindows()) {
            Assert.assertTrue((boolean)wrapper.getFileLinkStatus(file1).isSymlink());
            Assert.assertEquals((Object)file2, (Object)wrapper.getLinkTarget(file1));
        }
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkToExistingDir() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path subDir = new Path(this.testBaseDir2(), "subDir");
        Path link = new Path(this.testBaseDir1(), "linkToDir");
        wrapper.createSymlink(dir1, link, false);
        try {
            wrapper.rename(link, dir2, new Options.Rename[0]);
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        try {
            wrapper.rename(link, dir2, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        wrapper.mkdir(subDir, FsPermission.getDefault(), false);
        try {
            wrapper.rename(link, dir2, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkToItself() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        SymlinkBaseTest.createAndWriteFile(file);
        Path link = new Path(this.testBaseDir1(), "linkToFile1");
        wrapper.createSymlink(file, link, false);
        try {
            wrapper.rename(link, link, new Options.Rename[0]);
            Assert.fail((String)"Failed to get expected IOException");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        try {
            wrapper.rename(link, link, Options.Rename.OVERWRITE);
            Assert.fail((String)"Failed to get expected IOException");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
    }

    @Test(timeout=10000L)
    public void testRenameSymlink() throws IOException {
        Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        Path file = new Path(this.testBaseDir1(), "file");
        Path link1 = new Path(this.testBaseDir1(), "linkToFile1");
        Path link2 = new Path(this.testBaseDir1(), "linkToFile2");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link1, false);
        wrapper.rename(link1, link2, new Options.Rename[0]);
        Assert.assertTrue((boolean)wrapper.getFileLinkStatus(link2).isSymlink());
        Assert.assertFalse((boolean)wrapper.getFileStatus(link2).isDirectory());
        SymlinkBaseTest.readFile(link2);
        SymlinkBaseTest.readFile(file);
        try {
            SymlinkBaseTest.createAndWriteFile(link2);
            Assert.fail((String)"link was not renamed");
        }
        catch (IOException x) {
            // empty catch block
        }
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkToFileItLinksTo() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        try {
            wrapper.rename(link, file, new Options.Rename[0]);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)wrapper.isFile(file));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isSymlink(link));
        Assert.assertEquals((Object)file, (Object)wrapper.getLinkTarget(link));
        try {
            wrapper.rename(link, file, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)wrapper.isFile(file));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isSymlink(link));
        Assert.assertEquals((Object)file, (Object)wrapper.getLinkTarget(link));
    }

    @Test(timeout=10000L)
    public void testRenameSymlinkToDirItLinksTo() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path link = new Path(this.testBaseDir1(), "linkToDir");
        wrapper.mkdir(dir, FileContext.DEFAULT_PERM, false);
        wrapper.createSymlink(dir, link, false);
        try {
            wrapper.rename(link, dir, new Options.Rename[0]);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)wrapper.isDir(dir));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isSymlink(link));
        Assert.assertEquals((Object)dir, (Object)wrapper.getLinkTarget(link));
        try {
            wrapper.rename(link, dir, Options.Rename.OVERWRITE);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)wrapper.isDir(dir));
        Assert.assertTrue((boolean)wrapper.exists(link));
        Assert.assertTrue((boolean)wrapper.isSymlink(link));
        Assert.assertEquals((Object)dir, (Object)wrapper.getLinkTarget(link));
    }

    @Test(timeout=10000L)
    public void testRenameLinkTarget() throws IOException {
        Assume.assumeTrue((!this.emulatingSymlinksOnWindows() ? 1 : 0) != 0);
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileNew = new Path(this.testBaseDir1(), "fileNew");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        wrapper.rename(file, fileNew, Options.Rename.OVERWRITE);
        try {
            SymlinkBaseTest.readFile(link);
            Assert.fail((String)"Link should be dangling");
        }
        catch (IOException x) {
            // empty catch block
        }
        wrapper.rename(fileNew, file, Options.Rename.OVERWRITE);
        SymlinkBaseTest.readFile(link);
    }

    @Test(timeout=10000L)
    public void testRenameFileWithDestParentSymlink() throws IOException {
        Path link = new Path(this.testBaseDir1(), "link");
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path file2 = new Path(this.testBaseDir1(), "file2");
        Path file3 = new Path(link, "file3");
        Path dir2 = new Path(this.testBaseDir2());
        wrapper.createSymlink(dir2, link, false);
        SymlinkBaseTest.createAndWriteFile(file1);
        wrapper.rename(file1, file3, new Options.Rename[0]);
        Assert.assertFalse((boolean)wrapper.exists(file1));
        Assert.assertTrue((boolean)wrapper.exists(file3));
        wrapper.rename(file3, file1, new Options.Rename[0]);
        wrapper.delete(link, false);
        wrapper.createSymlink(file2, link, false);
        try {
            wrapper.rename(file1, file3, new Options.Rename[0]);
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileNotFoundException));
        }
        SymlinkBaseTest.createAndWriteFile(file2);
        try {
            wrapper.rename(file1, file3, new Options.Rename[0]);
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof ParentNotDirectoryException));
        }
    }

    @Test(timeout=10000L)
    public void testAccessFileViaInterSymlinkAbsTarget() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileNew = new Path(baseDir, "fileNew");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path fileNewViaLink = new Path(linkToDir, "fileNew");
        wrapper.createSymlink(baseDir, linkToDir, false);
        SymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertTrue((boolean)wrapper.exists(fileViaLink));
        Assert.assertTrue((boolean)wrapper.isFile(fileViaLink));
        Assert.assertFalse((boolean)wrapper.isDir(fileViaLink));
        Assert.assertFalse((boolean)wrapper.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)wrapper.isDir(fileViaLink));
        Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileLinkStatus(file));
        Assert.assertEquals((Object)wrapper.getFileStatus(fileViaLink), (Object)wrapper.getFileLinkStatus(fileViaLink));
        SymlinkBaseTest.readFile(fileViaLink);
        SymlinkBaseTest.appendToFile(fileViaLink);
        wrapper.rename(fileViaLink, fileNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)wrapper.exists(fileViaLink));
        Assert.assertTrue((boolean)wrapper.exists(fileNewViaLink));
        SymlinkBaseTest.readFile(fileNewViaLink);
        Assert.assertEquals((long)wrapper.getFileBlockLocations(fileNew, 0L, 1L).length, (long)wrapper.getFileBlockLocations(fileNewViaLink, 0L, 1L).length);
        Assert.assertEquals((Object)wrapper.getFileChecksum(fileNew), (Object)wrapper.getFileChecksum(fileNewViaLink));
        wrapper.delete(fileNewViaLink, true);
        Assert.assertFalse((boolean)wrapper.exists(fileNewViaLink));
    }

    @Test(timeout=10000L)
    public void testAccessFileViaInterSymlinkQualTarget() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        wrapper.createSymlink(wrapper.makeQualified(baseDir), linkToDir, false);
        SymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileLinkStatus(file));
        Assert.assertEquals((Object)wrapper.getFileStatus(fileViaLink), (Object)wrapper.getFileLinkStatus(fileViaLink));
        SymlinkBaseTest.readFile(fileViaLink);
    }

    @Test(timeout=10000L)
    public void testAccessFileViaInterSymlinkRelTarget() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path file = new Path(dir, "file");
        Path linkToDir = new Path(this.testBaseDir1(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        wrapper.mkdir(dir, FileContext.DEFAULT_PERM, false);
        wrapper.createSymlink(new Path("dir"), linkToDir, false);
        SymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertEquals((Object)wrapper.makeQualified(file), (Object)wrapper.getFileStatus(file).getPath());
        Assert.assertEquals((Object)wrapper.getFileStatus(file), (Object)wrapper.getFileLinkStatus(file));
        Assert.assertEquals((Object)wrapper.getFileStatus(fileViaLink), (Object)wrapper.getFileLinkStatus(fileViaLink));
        Assert.assertEquals((Object)wrapper.getFileStatus(fileViaLink), (Object)wrapper.getFileLinkStatus(file));
    }

    @Test(timeout=10000L)
    public void testAccessDirViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path dirViaLink = new Path(linkToDir, "dir");
        wrapper.createSymlink(baseDir, linkToDir, false);
        wrapper.mkdir(dirViaLink, FileContext.DEFAULT_PERM, true);
        Assert.assertTrue((boolean)wrapper.getFileStatus(dirViaLink).isDirectory());
        FileStatus[] stats = wrapper.listStatus(dirViaLink);
        Assert.assertEquals((long)0L, (long)stats.length);
        RemoteIterator<FileStatus> statsItor = wrapper.listStatusIterator(dirViaLink);
        Assert.assertFalse((boolean)statsItor.hasNext());
        wrapper.delete(dirViaLink, false);
        Assert.assertFalse((boolean)wrapper.exists(dirViaLink));
        Assert.assertFalse((boolean)wrapper.exists(dir));
    }

    @Test(timeout=10000L)
    public void testSetTimes() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        SymlinkBaseTest.createAndWriteFile(file);
        wrapper.createSymlink(file, link, false);
        long at = wrapper.getFileLinkStatus(link).getAccessTime();
        wrapper.setTimes(link, 2L, 3L);
        if (!"file".equals(this.getScheme())) {
            Assert.assertEquals((long)at, (long)wrapper.getFileLinkStatus(link).getAccessTime());
            Assert.assertEquals((long)3L, (long)wrapper.getFileStatus(file).getAccessTime());
            Assert.assertEquals((long)2L, (long)wrapper.getFileStatus(file).getModificationTime());
        }
    }
}

