/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;

import fr.ifremer.suiviobsmer.SuiviObsmerBusinessException;
import fr.ifremer.suiviobsmer.SuiviObsmerContext;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.entity.Boat;
import fr.ifremer.suiviobsmer.entity.Company;
import fr.ifremer.suiviobsmer.entity.FishingZone;
import fr.ifremer.suiviobsmer.entity.Profession;
import fr.ifremer.suiviobsmer.entity.ProfessionImpl;
import fr.ifremer.suiviobsmer.entity.SampleMonth;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.entity.SampleRowLog;
import fr.ifremer.suiviobsmer.entity.SampleRowLogImpl;
import fr.ifremer.suiviobsmer.entity.WaoUser;
import fr.ifremer.suiviobsmer.services.ServiceBoat;
import fr.ifremer.suiviobsmer.services.ServiceReferential;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.services.ServiceUser;
import fr.ifremer.suiviobsmer.ui.base.GenericSelectModel;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.data.SuiviObsmerPropertyChangeListener;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.Field;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.BeanModelSource;
import org.nuiton.util.PeriodDates;
import org.slf4j.Logger;

/**
 * SampleRowForm
 *
 * Created: 26 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 330 $
 *
 * Mise a jour: $Date: 2010-02-05 15:04:29 +0100 (ven. 05 févr. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/sampling.css")
public class SampleRowForm implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return true;
    }

    @Inject
    private Logger log;
    
    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceBoat serviceBoat;

    @SessionState
    @Property
    private WaoUser user;
    
    @Inject
    private PropertyAccess propertyAccess;

    private boolean edited;

    void onActivate(EventContext ec) {
        if (ec.getCount() > 0) {
            sampleRowId = ec.get(String.class, 0);
        }
    }

    String onPassivate() {
        return sampleRowId;
    }

    void setupRender() throws SuiviObsmerException {
        // Reset data
        sampleRow = null;
        getSampleRow();
        sampleMonths = null;
        getSampleMonths();
        sampleRowLog = null;
        getSampleRowLog();

        // Reset persist select
        professionSelectModel = null;
        getProfessionSelectModel();
        programSelectModel = null;
        getProgramSelectModel();
        fishingZoneSelectModel = null;
        getFishingZoneSelectModel();
        companySelectModel = null;
        getCompanySelectModel();
        
        professionId = null;
        sampleRowForm.clearErrors();

        if (!isCreateMode()) {
            programId = getSampleRow().getProgramName();
        } else {
            programId = null;
        }

        // initialize companyId if exist for companySelect
        Company company = getSampleRow().getCompany();
        if (company != null) {
            companyId = company.getTopiaId();
        }
        
        immatriculations = getSampleRow().getMainElligibleBoatsAsString();
    }

    /*************************** PROFESSION ***********************************/

    /** Select model which contains all professions **/
    @Persist
    private GenericSelectModel<Profession> professionSelectModel;

    /** The current selected professionId **/
    @Property
    private String professionId;

    /** Bean model source to create the BeanModel for profession **/
    @Inject
    private BeanModelSource beanModelSource;

    /** Component resources needed to create the BeanModel for profession **/
    @Inject
    private ComponentResources resources;

    /**
     * Get SelectModel for profession input Select. The select contains all existing professions.
     *
     * @return a GenericSelectModel<Profession>
     * @throws SuiviObsmerException
     */
    public GenericSelectModel<Profession> getProfessionSelectModel() throws SuiviObsmerException {
        if (professionSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getProfessions]");
            }
            List<Profession> professions = serviceReferential.getProfessions();
            professionSelectModel = new GenericSelectModel<Profession>(professions, Profession.class,
                    "code", "topiaId", propertyAccess);
        }
        return professionSelectModel;
    }

    /**
     * Get the Profession model for BeanDisplay component.
     *
     * @return a BeanModel<Profession>
     */
    public BeanModel<Profession> getProfessionModel() {
        return beanModelSource.createDisplayModel(Profession.class, resources.getMessages());
    }

    /**
     * Action on submit : addNewProfession. Used to set a new profession from the selected one.
     *
     * @throws SuiviObsmerException
     */
    void onSelectedFromAddNewProfession() throws SuiviObsmerException {
        Profession profession = new ProfessionImpl();
         if (!StringUtils.isEmpty(professionId)) {
            Profession professionSelected = getProfessionSelectModel().findObject(professionId);
            if (log.isDebugEnabled()) {
                log.debug("Add new profession from : " + professionSelected);
            }
            profession = serviceSampling.getNewProfession(professionSelected);
        }
        getSampleRow().setProfession(profession);
        edited = true;
    }

    /*************************** FISHING ZONE *********************************/

    /** Select model for the FishingZone list **/
    @Persist
    private GenericSelectModel<FishingZone> fishingZoneSelectModel;

    /** Current fishingZone id selected **/
    @Property
    private String fishingZoneId;

    /** Current fishingZone for loop **/
    @Property
    private FishingZone fishingZone;

    /** Current index in the loop **/
    @Property
    private int indexFishingZone;

    /** Field fishingZone for validation : must have at least one fishingZone **/
    @InjectComponent
    private Field fishingZoneSelect;

    /**
     * Get the select model for FishingZone. The select contains all existing fishingZones.
     *
     * @return a GenericSelectModel<FishingZone>
     * @throws SuiviObsmerException
     */
    public GenericSelectModel<FishingZone> getFishingZoneSelectModel() throws SuiviObsmerException {
        if (fishingZoneSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getFishingZones]");
            }
            List<FishingZone> zones = serviceReferential.getFishingZones();
            fishingZoneSelectModel = new GenericSelectModel<FishingZone>(zones, FishingZone.class,
                    "code", "topiaId", propertyAccess);
        }
        return fishingZoneSelectModel;
    }

    /**
     * Get the list of FishingZones from the SampleRow.
     *
     * @return a List of FishingZone linked with the SampleRow
     * @throws SuiviObsmerException
     */
    public List<FishingZone> getFishingZones() throws SuiviObsmerException {
        return getSampleRow().getFishingZone();
    }

    /**
     * Action on submit : addFishingZone. Used to add the selected fishingZone to the SampleRow.
     *
     * @throws SuiviObsmerException
     */
    void onSelectedFromAddFishingZone() throws SuiviObsmerException {
        fishingZone = getFishingZoneSelectModel().findObject(fishingZoneId);
        if (!getFishingZones().contains(fishingZone)) {
            if (log.isDebugEnabled()) {
                log.debug("Add new fishingZone");
            }
            getFishingZones().add(fishingZone);
        }
        edited = true;
    }

    /**
     * Action on submit : removeFishingZone. Used to remove the fishingZone from the SampleRow.
     *
     * @param index of the FishingZone in the list (from loop component)
     * @throws SuiviObsmerException
     */
    void onSelectedFromRemoveFishingZone(int index) throws SuiviObsmerException {
        if (log.isDebugEnabled()) {
            log.debug("Remove fishingZone at position : " + index);
        }
        getFishingZones().remove(index);
        edited = true;
    }

    /*************************** PROGRAM & MONTHS *****************************/

    @Persist
    private SelectModel programSelectModel;

    @Property
    private String programName;

    @Property
    private String programId;

//    @Property
//    private Date month;

    private PeriodDates period;

//    @Persist
//    private List<Date> months;

    @Persist
    private List<SampleMonth> sampleMonths;

    @Property
    private int monthIndex;

    @Property
    private SampleMonth sampleMonth;

    @InjectComponent
    private Field programPeriodBegin;

    @InjectComponent
    private Field programPeriodEnd;

    @InjectComponent
    private Field program;

    private boolean periodChanged;

    public SelectModel getProgramSelectModel() throws SuiviObsmerException {
        if (programSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getPrograms]");
            }
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String name : serviceSampling.getPrograms(null)) {
                options.add(new OptionModelImpl(name, name));
            }
            programSelectModel = new SelectModelImpl(null, options);
        }
        return programSelectModel;
    }

//    public List<Date> getMonths() throws SuiviObsmerException {
//        if (months == null) {
//            if (log.isDebugEnabled()) {
//                log.debug("Reset months list");
//            }
//            months = getPeriod().getMonths();
//            sampleMonths = null;
//            getSampleMonths();
//        }
//        return months;
//    }

//    public Program getProgram() throws SuiviObsmerException {
//        return getSampleRow().getProgram();
//    }

    public PeriodDates getPeriod() throws SuiviObsmerException {
        if (period == null) {
            Date begin = getSampleRow().getPeriodBegin();
            Date end = getSampleRow().getPeriodEnd();
            period = new PeriodDates(begin, end);
        }
        return period;
    }

    public List<SampleMonth> getSampleMonths() throws SuiviObsmerException {
        if (sampleMonths == null) {
            sampleMonths = new ArrayList<SampleMonth>();
            for (Date currMonth : getPeriod().getMonths()) {
                SampleMonth curr = getSampleRow().getSampleMonth(currMonth);
                if (curr != null) {
                    sampleMonths.add(curr);
                // If an other period is set
                } else if (periodChanged) {
                    if (curr == null) {
                        curr = serviceSampling.getNewSampleMonth(currMonth, getSampleRow());
                    }
                    sampleMonths.add(curr);
                }
            }
        }
        return sampleMonths;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    public String getMonthStyle() {
        String style = " ";
        
        Date current = SuiviObsmerContext.getCurrentDate();
        if (sampleMonth.isCurrentMonth()) {
            style += "selected";
        } else if (current.after(sampleMonth.getPeriodDate())) {
            style += "even";
        }
        return style;
    }

    /**
     * Search a program from the list.
     * The program will be set from the one selected. If no selection, a new one will be set.
     * @throws SuiviObsmerException
     */
//    @Log
//    void onSelectedFromSearchProgram() throws SuiviObsmerException {
//        if (!StringUtils.isEmpty(programId)) {
//            programName = null;
////            period = null;
////            months = null;
//            getSampleRow().setProgramName(programId);
//
//            // No period in this case ?!?
//        }
//        edited = true;
//    }

    /**
     * Refresh months from program period.
     * If the programName is not empty, a new program will be set.
     * The period dates will be saved in program (validation will be done in onSuccess).
     * @throws SuiviObsmerException
     */
    @Log
    void onSelectedFromRefreshMonths() throws SuiviObsmerException {
        getSampleRow().setPeriod(period);
        periodChanged = true;
        // Reset sampleMonths for new period
        sampleMonths = null;
        getSampleMonths();
        saveProgram();
    }

    protected void saveProgram() throws SuiviObsmerException {
        // Save programName
        if (StringUtils.isNotEmpty(programName)) {
            getSampleRow().setProgramName(programName);
            programId = null;
        } else if (StringUtils.isNotEmpty(programId)) {
            getSampleRow().setProgramName(programId);
        }
        if (log.isInfoEnabled()) {
            log.info("Program : " + getSampleRow().getProgramName());
        }
    }

    @Log
    void onSelectedFromDeleteMonth(int index) throws SuiviObsmerException {
        getSampleMonths().remove(index);
        edited = true;
    }

    /*************************** SAMPLE ROW ***********************************/

    private String sampleRowId;

    @Persist
    private SampleRow sampleRow;

    private BeanModel<SampleRow> sampleRowModel;

    @InjectComponent
    private Form sampleRowForm;

    @InjectComponent
    private Field updateComment;

    @Persist
    private SampleRowLog sampleRowLog;

    @Property
    private String immatriculations;

    @Property
    private String companyId;

    private List<Boat> boats;

    private GenericSelectModel<Company> companySelectModel;

    private boolean cancel;
    
    @Persist
    private SuiviObsmerPropertyChangeListener propertyChange;

    public SampleRow getSampleRow() throws SuiviObsmerException {
        if (sampleRow == null) {
            if (!StringUtils.isEmpty(sampleRowId)) {
                if (log.isInfoEnabled()) {
                    log.info("BUSINESS REQUEST [getSampleRow]");
                }
                sampleRow = serviceSampling.getSampleRow(sampleRowId);
                propertyChange = new SuiviObsmerPropertyChangeListener();
                for (SampleMonth month : sampleRow.getSampleMonth()) {
                    month.addPropertyChangeListener(SampleMonth.EXPECTED_TIDES_VALUE, propertyChange);
                }        
            } else {
                sampleRow = serviceSampling.getNewSampleRow();
            }
        }
        return sampleRow;
    }

    public SampleRowLog getSampleRowLog() throws SuiviObsmerException {
        if (sampleRowLog == null) {
             sampleRowLog = new SampleRowLogImpl();
             sampleRowLog.setSampleRow(getSampleRow());
             sampleRowLog.setAdmin(user);
        }
        return sampleRowLog;
    }

    public BeanModel<SampleRow> getSampleRowModel() {
        if (sampleRowModel == null) {
            sampleRowModel = beanModelSource.createEditModel(SampleRow.class, resources.getMessages());
            sampleRowModel.add("company", null);
            sampleRowModel.add("boats", null);
            sampleRowModel.include("code", "company", "nbObservants", "averageTideTime", "boats", "comment");
        }
        return sampleRowModel;
    }

    public GenericSelectModel<Company> getCompanySelectModel() throws SuiviObsmerException {
        if (companySelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getCompanies]");
            }
            List<Company> companies = serviceUser.getCompanies(true);
            companySelectModel = new GenericSelectModel<Company>(companies, Company.class,
                    "name", "topiaId", propertyAccess);
        }
        return companySelectModel;
    }

    public boolean isCreateMode() throws SuiviObsmerException {
        return StringUtils.isEmpty(getSampleRow().getTopiaId());
    }

    void onSelectedFromCancelSave() {
        cancel = true;
    }

    void onSelectedFromSaveData() throws SuiviObsmerException {
        saveProgram();
    }

    @Log
    void onValidateFormFromSampleRowForm() throws SuiviObsmerException {
        sampleRowForm.clearErrors();
        try {
            // Only for save or periodChanged (refreshMonth)
            if (!cancel && !edited) {
                if (StringUtils.isEmpty(getSampleRow().getProgramName())) {
                    sampleRowForm.recordError(program,
                            "Vous devez choisir un programme existant ou en créer un nouveau");
                }

                Date periodBegin = getSampleRow().getPeriodBegin();
                Date periodEnd = getSampleRow().getPeriodEnd();

                // Validation for period dates
                if (periodBegin == null) {
                    sampleRowForm.recordError(programPeriodBegin,
                            "Vous ne pouvez pas rafraîchir les mois avec une date de début de programme vide");
                }
                if (periodEnd == null) {
                    sampleRowForm.recordError(programPeriodEnd,
                            "Vous ne pouvez pas rafraîchir les mois avec une date de fin de programme vide");
                }

                if (periodBegin != null && periodEnd != null) {

                    int size = getSampleMonths().size();

                    SampleMonth firstMonth = getSampleMonths().get(0);
                    //SampleMonth lastMonth = getSampleMonths().get(size-1);

                    if (firstMonth != null) {
                        Date firstDate = firstMonth.getPeriodDate();

                        if (periodBegin.after(firstDate) && firstMonth.getRealTidesValue() != 0) {
                            sampleRowForm.recordError(programPeriodBegin,
                                "Vous ne pouvez pas réduire la période du programme. " +
                                "Il existe des enregistrements de marées réels.");
                        }
                    }

                    for (SampleMonth month : getSampleMonths()) {
                        if (log.isTraceEnabled()) {
                            log.trace("month : " + month.formatMonth() + " _ " +
                                    propertyChange.hasChanged(month.getTopiaId()));
                        }

                        if (propertyChange.hasChanged(month.getTopiaId()) &&
                                month.getExpectedTidesValue() < month.getRealTidesValue()) {
                            sampleRowForm.recordError(
                                "Il n'est pas possible de mettre moins de marées (" + month.getExpectedTidesValue() + ") " +
                                "pour le mois " + getDateFormat().format(month.getPeriodDate()) +
                                " qu'il y en a de réels (" + month.getRealTidesValue() + ")");
                        }
                    }
                }

                // Validate profession, fishingZone and boats (from immatriculations) if no programChanged
                if (!periodChanged) {

                    if (!isCreateMode() && StringUtils.isEmpty(sampleRowLog.getComment())) {
                        sampleRowForm.recordError(updateComment,
                                "Un commentaire est obligatoire lors d'une modification de la ligne");
                    }

                    Profession profession = getSampleRow().getProfession();
                    if (StringUtils.isEmpty(profession.getCodeDCF5())) {
                        sampleRowForm.recordError(
                                "Le métier doit obligatoirement avoir un Code DCF niveau 5");
                    }
                    int nbZones = getSampleRow().sizeFishingZone();
                    if (nbZones == 0) {
                        sampleRowForm.recordError(fishingZoneSelect,
                                "Vous devez ajouter au moins une zone de pêche à la ligne");
                    }
                    
                    boats = serviceBoat.getBoatsByImmatriculations(immatriculations);
                // Update sampleRowCode from program only if changed
                } else if (isCreateMode()) {
                    String rowCode = serviceSampling.getNewSampleRowCode(periodBegin);
                    getSampleRow().setCode(rowCode);
                }
            }
        } catch (SuiviObsmerBusinessException eee) {
            sampleRowForm.recordError(eee.getMessage());
            log.error("BUSINESS ERROR : " + eee.getType() + " depuis " + eee.getServiceName() +
                    " : " + eee.getMessage());
        }

        if (log.isDebugEnabled()) {
            log.debug("SampleMonths : " + getSampleMonths());
        }
    }

    @Log
    Object onSuccessFromSampleRowForm() throws SuiviObsmerException {
        if (cancel) {
            return this;
        }
        if (!edited && !periodChanged) {               

                // No company change possible when sampleRow has real tide time
                if (!getSampleRow().hasSampleMonthRealTideTime()) {
                    // Save company
                    Company company = getCompanySelectModel().findObject(companyId);
                    getSampleRow().setCompany(company);
                }

                // Save sampleMonths

                if (log.isDebugEnabled()) {
                    log.debug("SampleMonths : " + getSampleMonths());
                }

                getSampleRow().setSampleMonth(getSampleMonths());

                serviceSampling.createUpdateSampleRow(sampleRow, boats, getSampleRowLog());

                return SamplingPlan.class;
        }  else {
            // Reinitialize professionId for select to avoid selection from existing Profession
            professionId = null;
        }
        return sampleRowForm;
    }

    @Log
    Object onFailureFromSampleRowForm() {
        return sampleRowForm;
    }

        
}
