/*
 * #%L
 * SGQ :: Business
 * $Id: ReferentialService.java 48 2012-09-11 12:46:57Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.2/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ReferentialService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import java.io.File;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;
import org.xBaseJ.DBF;
import org.xBaseJ.fields.CharField;
import org.xBaseJ.fields.NumField;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.entities.AnalyzeCategory;
import com.herbocailleau.sgq.entities.AnalyzeCategoryDAO;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.AnalyzeTypeDAO;
import com.herbocailleau.sgq.entities.Client;
import com.herbocailleau.sgq.entities.ClientDAO;
import com.herbocailleau.sgq.entities.Place;
import com.herbocailleau.sgq.entities.PlaceDAO;
import com.herbocailleau.sgq.entities.Product;
import com.herbocailleau.sgq.entities.ProductDAO;
import com.herbocailleau.sgq.entities.Supplier;
import com.herbocailleau.sgq.entities.SupplierDAO;

public class ReferentialService extends SgqService {

    private static final Log log = LogFactory.getLog(ReferentialService.class);

    /**
     * Import client dbf file.
     * 
     * @param file dbf file
     */
    public void importClients(File file) {
        
        if (log.isInfoEnabled()) {
            log.info("Importing client from file " + file.getAbsolutePath());
        }

        DBF fileDB = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();

            // Open dbf file
            fileDB = new DBF(file.getAbsolutePath(), DBF.READ_ONLY, "UTF-8");

            // code,nom
            CharField codeField  = (CharField) fileDB.getField("code");
            CharField nameField  = (CharField) fileDB.getField("nom");
            for (int i = 0; i < fileDB.getRecordCount(); i++) {
                fileDB.read();

                Client client = clientDAO.createByNaturalId(codeField.get());
                client.setName(nameField.get());

                clientDAO.update(client);
            }
            
            daoHelper.commit();
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't read dbf file", ex);
        } finally {
            SgqUtils.closeFile(fileDB);
        }
    }

    /**
     * Import products dbf file.
     * 
     * @param file dbf file
     */
    public void importProducts(File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing products from file " + file.getAbsolutePath());
        }

        DBF fileDB = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();

            // Open dbf file
            fileDB = new DBF(file.getAbsolutePath(), DBF.READ_ONLY, "UTF-8");

            // "code";"name";"latinname";"family"
            CharField codeField  = (CharField) fileDB.getField("code");
            CharField nameField  = (CharField) fileDB.getField("name");
            CharField latinNameField  = (CharField) fileDB.getField("latinname");
            CharField familyField  = (CharField) fileDB.getField("family");

            for (int i = 0; i < fileDB.getRecordCount(); i++) {
                fileDB.read();

                Product product = productDAO.createByNaturalId(codeField.get());
                product.setName(nameField.get());
                product.setFamily(familyField.get());
                product.setLatinName(latinNameField.get());

                productDAO.update(product);
            }
            
            daoHelper.commit();
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't read dbf file", ex);
        } finally {
            SgqUtils.closeFile(fileDB);
        }
    }

    /**
     * Import suppliers dbf file.
     * 
     * @param file dbf file
     */
    public void importPlaces(File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing places from file " + file.getAbsolutePath());
        }

        DBF fileDB = null;
        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();

            // Open dbf file
            fileDB = new DBF(file.getAbsolutePath(), DBF.READ_ONLY, "UTF-8");

            // "code"
            CharField codeField  = (CharField) fileDB.getField("code");

            for (int i = 0; i < fileDB.getRecordCount(); i++) {
                fileDB.read();

                Place place = placeDAO.createByNaturalId(codeField.get());

                placeDAO.update(place);
            }
            
            daoHelper.commit();
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't read dbf file", ex);
        } finally {
            SgqUtils.closeFile(fileDB);
        }
    }

    /**
     * Import suppliers dbf file.
     * 
     * @param file dbf file
     */
    public void importSuppliers(File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing suppliers from file " + file.getAbsolutePath());
        }

        DBF fileDB = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();

            // Open dbf file
            fileDB = new DBF(file.getAbsolutePath(), DBF.READ_ONLY, "UTF-8");

            // "code";"name"
            CharField codeField  = (CharField) fileDB.getField("code");
            CharField nameField  = (CharField) fileDB.getField("name");
            NumField labField  = (NumField) fileDB.getField("lab");
            
            for (int i = 0; i < fileDB.getRecordCount(); i++) {
                fileDB.read();

                Supplier supplier = supplierDAO.createByNaturalId(codeField.get());
                supplier.setName(nameField.get());
                supplier.setLaboratory("1.00".equals(labField.get()));
                supplierDAO.update(supplier);
            }
            
            daoHelper.commit();
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't read dbf file", ex);
        } finally {
            SgqUtils.closeFile(fileDB);
        }
    }

    public long getClientsCount() {
        long result = 0;

        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            result = clientDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get clients ordered by code.
     * 
     * @param offset offset
     * @param count count
     * @return count clients
     */
    public List<Client> getClients(int offset, int count) {
        List<Client> results = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            results = clientDAO.findAllByQueryWithBound("from " + Client.class.getName() +
                    " order by " + Client.PROPERTY_CODE, offset, offset + count - 1);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }

    public Client findClientById(String id) {
        Client result = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            result = clientDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get client", ex);
        }
        return result;
    }

    public void saveClient(Client client) {
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            if (StringUtils.isEmpty(client.getTopiaId())) {
                clientDAO.create(client);
            } else {
                clientDAO.update(client);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save client", ex);
        }
        
    }

    public long getPlacesCount() {
        long result = 0;

        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            result = placeDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get places ordered by code.
     * 
     * @return all places
     */
    public List<Place> getPlaces() {
        List<Place> results = null;
        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            results = placeDAO.findAll();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }

    public long getSuppliersCount() {
        long result = 0;

        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get suppliers ordered by code.
     * 
     * @param offset offset
     * @param count count
     * @return count suppliers
     */
    public List<Supplier> getSuppliers(int offset, int count) {
        List<Supplier> results = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            results = supplierDAO.findAllByQueryWithBound("from " + Supplier.class.getName() +
                    " order by " + Supplier.PROPERTY_CODE, offset, offset + count - 1);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }
    
    /**
     * Get supplier by topia id.
     * 
     * @param offset offset
     * @param count count
     * @return count suppliers
     */
    public Supplier getSupplierById(String supplierId) {
        Supplier result = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.findByTopiaId(supplierId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get supplier", ex);
        }
        return result;
    }

    public long getProductsCount() {
        long result = 0;

        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = productDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }
    
    /**
     * Get product ordered by code.
     * 
     * @param offset offset
     * @param count count
     * @return count products
     */
    public List<Product> getProducts(int offset, int count) {
        List<Product> results = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            results = productDAO.findAllByQueryWithBound("from " + Product.class.getName() +
                    " order by " + Product.PROPERTY_CODE, offset, offset + count - 1);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }

    /**
     * Create new client.
     * 
     * @param code client code
     * @param name client name
     */
    public void createClient(String code, String name) {
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            Client client = clientDAO.createByNaturalId(code);
            client.setName(name);
            clientDAO.update(client);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
    }

    /**
     * Find all analyze types.
     * 
     * @return all analyze types
     */
    public List<AnalyzeType> findAllAnalyzeTypes() {
        List<AnalyzeType> result = null;
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            result = analyzeTypeDAO.findAll();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze types", ex);
        }
        return result;
    }
    
    /**
     * Find single analyze types by id.
     * 
     * @return analyze type
     */
    public AnalyzeType findAnalyzeTypeById(String id) {
        AnalyzeType result = null;
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            result = analyzeTypeDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze type", ex);
        }
        return result;
    }

    /**
     * Save (create or update) analyze type.
     * 
     * @return analyze type
     */
    public void saveAnalyzeType(AnalyzeType analyzeType) {
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            if (StringUtils.isEmpty(analyzeType.getTopiaId())) {
                analyzeTypeDAO.create(analyzeType);
            } else {
                analyzeTypeDAO.update(analyzeType);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save analyze type", ex);
        }
    }
 
    /**
     * Find all analyze categories.
     * 
     * @return
     */
    public List<AnalyzeCategory> findAllAnalyzeCategories() {
        List<AnalyzeCategory> result = null;
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            result = analyzeCategoryDAO.findAll();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze categories", ex);
        }
        return result;
    }
    
    public AnalyzeCategory findAnalyzeCategoryById(String id) {
        AnalyzeCategory result = null;
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            result = analyzeCategoryDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze category", ex);
        }
        return result;
    }

    /**
     * Return all suppliers (laboratory first).
     * 
     * @return all supplier
     */
    public List<Supplier> getAllSupplierLabFirst() {
        List<Supplier> result;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.findAllWithOrder(Supplier.PROPERTY_LABORATORY +" DESC",
                    Supplier.PROPERTY_NAME + " ASC");
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze category", ex);
        }
        return result;
    }
}
