/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author poussin
 * @version $Revision: 8 $
 *
 * Last update: $Date: 2010-04-20 17:41:03 +0200 (mar., 20 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyExtension implements Serializable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyExtension.class);

    private static final long serialVersionUID = 1L;

    /**
     * Name of this extension.
     */
    protected String name;

    /**
     * Name of other extension needed to put this extension to object.
     * 
     * Warning : Multiples extensions are not supported yet.
     * 
     * FIXME EC-20100420 multiples extensions need to work
     */
    protected String requires;

    /**
     * use to know version objet, when you change field number, type or other
     * you must change version number.
     */
    protected String version = WikittyUtil.DEFAULT_VERSION;

    /** used to store tag/value used by client side ex: updatedDate=101212 */
    Map<String, String> tagValues = new HashMap<String, String>();

    /**
     * fields use ordered map, to keep order insertion of field
     * key: field name
     * value: field type
     */
    protected LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

    protected WikittyExtension() {
    }

    public WikittyExtension(String name) {
        if (name == null) {
            throw new IllegalArgumentException("Name must not be null");
        }
        this.name = name;
    }
    
    public WikittyExtension(String name, String version,
            String requires, LinkedHashMap<String, FieldType> fields) {
        if (name == null) {
            throw new IllegalArgumentException("Name must not be null");
        }
        if (version == null) {
            throw new IllegalArgumentException("Version must not be null");
        }
        this.name = name;
        this.version = WikittyUtil.normalizeVersion(version);
        this.requires = requires;
        if (fields != null) {
            this.fields = fields;
        }
    }

    public WikittyExtension cloneForUpgrade() {
        String nextRevision = WikittyUtil.incrementMajorRevision(getVersion());
        
        LinkedHashMap<String, FieldType> nextFields = null;
        if (fields != null) {
            nextFields = new LinkedHashMap<String, FieldType>();
            for (Map.Entry<String, FieldType> entry : fields.entrySet()) {
                FieldType type = entry.getValue();
                FieldType nextType = new FieldType(type.getType(), type.getLowerBound(), type.getUpperBound(), type.isUnique(), type.isNotNull());
                Set<String> tagNames = type.getTagNames();
                if (tagNames != null) {
                    for (String tagName : tagNames) {
                        String tagValue = type.getTagValue(tagName);
                        nextType.addTagValue(tagName, tagValue);
                    }
                }
                nextFields.put(entry.getKey(), nextType);
            }
        }

        WikittyExtension result = new WikittyExtension(name, nextRevision, requires, nextFields);
        return result;
    }

    /**
     * compute id for extension name and version in argument
     * @param name
     * @param version
     * @return
     */
    static public String computeId(String name, String version) {
        String result = name + "[" + version + "]";
        return result;
    }

    static public String computeName(String id) {
        int i = id.lastIndexOf("[");
        String result = id;
        if (i != -1) {
            result = id.substring(0, i);
        }
        return result;
    }
    
    static public String computeVersion(String id) {
        int b = id.lastIndexOf("[");
        int e = id.lastIndexOf("]");
        String result = null;
        if (b != -1 && e != -1) {
            result = id.substring(b+1, e);
        }
        result = WikittyUtil.normalizeVersion(result);
        return result;
    }

    public String getId() {
        String result = computeId(getName(), getVersion());
        return result;
    }

    public String getName() {
        return name;
    }

    public String getVersion() {
        return version;
    }

    public String getRequires() {
        return requires;
    }
    
    public FieldType getFieldType(String fieldName) {
        return fields.get(fieldName);
    }

    public Collection<String> getFieldNames() {
        Collection<String> result = fields.keySet();
        return result;
    }

    public void addField(String fieldName, FieldType type) {
        fields.put(fieldName, type);
    }

    public void removeField(String fieldName) {
        fields.remove(fieldName);
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    public void addTagValue(String tag, String value) {
        tagValues.put(tag, value);
    }

    public String getTagValue(String tag) {
        String result = tagValues.get(tag);
        return result;
    }

    public Set<String> getTagNames() {
        return tagValues.keySet();
    }

    public Map<String, String> getTagValues() {
        return tagValues;
    }

    public void setTagValues(Map<String, String> tagValues) {
        this.tagValues = tagValues;
    }

    @Override
    public boolean equals(Object obj) {
        boolean result = false;
        if (obj instanceof WikittyExtension) {
            WikittyExtension other = (WikittyExtension)obj;
            result = this.getId().equals(other.getId());
        }
        return result;
    }

    @Override
    public String toString() {
        return getId();
    }
    
    public String toDefinition() {
        String result = "Extension " + getId();
        result += WikittyUtil.tagValuesToString(tagValues);
        result += " {\n";
        for (String fieldName : fields.keySet()) {
            result += fields.get(fieldName).toDefinition(fieldName) + "\n";
        }
        result += "}";
        return result;
    }

}
