
package org.nuiton.web.tapestry5.services;

import java.lang.annotation.Annotation;
import org.apache.tapestry5.ioc.internal.util.TapestryException;
import org.apache.tapestry5.services.ApplicationStateManager;

/**
 * This class is an abstract implementation of {@link ServiceAuthentication}.
 * This service use {@link ApplicationStateManager} to manage connected user.
 * You can inherit this class to provide a momre specific authentication
 * managmentan depends on user type and annotation used to identify
 * pages that need authentication. This behavior is done in
 * {@link #isAllowed(Class)} that detect the annotation on the page (ex :
 * {@link RequiresAuthentication} annotation) and call the method
 * {@link #checkAuthorizations(Object, Annotation)} that is implemented by
 * default to do nothing (return true).
 * <br \>
 * <h2>AppModule contribution :</2>
 * Exemple with MyUser type for user and RequiresAnnotation for annotation :
 * <pre>
 *  public ServiceAuthentication<MyUser> buildServiceAuthentication(ApplicationStateManager stateManager) {
 *      return new ServiceAuthenticationImpl<MyUser, RequiresAuthentication>(stateManager);
 *  }
 * </pre>
 * No need to contribute to ApplicationStateManager to provide instantiation
 * of user. Even this service used the default constructor to instanciate a new
 * user.
 *
 * Created: 3 mai 2010
 *
 * @param <U> user type
 * @param <A> annotation type used to identify which page need authentication
 * @author fdesbois
 * $Id: ServiceAuthenticationImpl.java 4 2010-05-04 14:54:59Z fdesbois $
 */
public class ServiceAuthenticationImpl<U, A extends Annotation>
        implements ServiceAuthentication<U> {

    /**
     * Tapestry service used to save user in session and retrieve it when
     * needed.
     **/
    private final ApplicationStateManager stateManager;

    /** 
     * User type
     **/
    protected Class<U> userClass;

    /**
     * Annotation type to detect pages that need authentication (ex :
     * RequiresAuthentication).
     */
    protected Class<A> annotationClass;

    /**
     * Constructor to call in subclasses.
     *
     * @param stateManager from Tapestry
     * @param userClass to identify which type of user is managed
     * @param annotationClass to identify which type of annotation is managed
     * for authentication treatment.
     */
    public ServiceAuthenticationImpl(ApplicationStateManager stateManager,
            Class<U> userClass, Class<A> annotationClass) {
        this.stateManager = stateManager;
        this.userClass = userClass;
        this.annotationClass = annotationClass;
    }

    @Override
    public boolean isUserConnected() {
        return stateManager.exists(userClass);
    }

    @Override
    public U getUserConnected() {
        return stateManager.getIfExists(userClass);
    }

    @Override
    public void setUserConnected(U user) {
        stateManager.set(userClass, user);
    }

    /**
     * Detect the annotation in the given {@code page}. If annotation is
     * detected, the user authorizations need to be check using
     * {@link #checkAuthorizations(Object, Annotation)} method.
     *
     * @param page to check
     * @return true if connected user can display the page, false otherwise
     */
    @Override
    public boolean isAllowed(Class<?> page) {
        if (page.isAnnotationPresent(annotationClass) &&
                isUserConnected()) {

            A check = page.getAnnotation(annotationClass);

            U user = getUserConnected();

            return checkAuthorizations(user, check);
        }
        // No restriction if annotation is not present
        return true;
    }
        
    /**
     * Check the connected user authorizations. Called in
     * {@link #isAllowed(Class)} method after detect the annotation on the
     * current page. Override this method to provide more specific
     * authentication managment.
     *
     * @param user current user connected.
     * @param annotation from page to check authorizations.
     * @return true if the user is granted, false otherwise
     */
    protected boolean checkAuthorizations(U user, A annotation) {
        // The user is connected so, he can access to the page.
        return true;
    }

    /**
     * Default instanciation of the user. The user class need a default
     * constructor.
     *
     * @return a new user instance.
     */
    @Override
    public U getNewUserInstance() {
        try {
            return userClass.newInstance();
        } catch (Exception eee) {
            new TapestryException("Error on user instanciation " +
                    userClass.getName(), eee);
        }
        return null;
    }

}
